import os
import shutil
import time
import subprocess
import threading
import logging
from termcolor import colored
from seleniumwire import webdriver
from selenium.webdriver.chrome.service import Service
from selenium.webdriver.chrome.options import Options
from webdriver_manager.chrome import ChromeDriverManager
from selenium.webdriver.common.by import By
from selenium.common.exceptions import NoSuchElementException
from selenium.webdriver.common.keys import Keys
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
from datetime import datetime, timedelta

# Global variable to store recorded channels and the path recorded to
recorded_channels = {}


def handle_alerts(driver):
    try:
        alert = driver.switch_to.alert
        alert_text = alert.text
        alert.accept()  # or alert.dismiss() to close the alert
    except:
        pass

def read_channels_file(file_path):
    """Reads the channels configuration file and returns a list of URLs."""
    with open(file_path, 'r') as file:
        return [line.strip() for line in file if line.strip()]

def check_svg_in_page(driver, url, line_number):
    """Checks if the SVG file is present in the webpage and saves the HTML to debug.html."""
    try:
        driver.get(url)
        handle_alerts(driver)
        time.sleep(2)  # Wait for the page to load completely

        html_content = driver.page_source

        if 'src="/img/live.8cab136d.svg"' in html_content:
            print(colored(f"{url} is live!","light_green"))
            # Modify URL based on the condition
            if "creatorhome" in url:
                new_url = url.replace("creatorhome", "live")
            else:
                new_url = url + "/live"
            check_track_m3u8_in_network(driver, new_url, line_number)
        else:
            print(colored(f"{url} does not appear to be live.","light_yellow"))
            
    except Exception as e:
        print(f"An error occurred while accessing {url}: {e}")

def handle_popup(driver):
    """Handles the popup if it appears, e.g., clicks on 'I Confirm' button."""
    try:
        confirm_button = driver.find_element(By.XPATH, '//button[contains(text(), "I Confirm")]')
        confirm_button.click()
        print("Clicked 'I Confirm' button on the popup")
    except NoSuchElementException:
        print("No popup")
    except Exception as e:
        print(f"Error handling popup: {e}")

def check_track_m3u8_in_network(driver, url, line_number):
    """Intercepts network traffic to find 'track.m3u8' URL."""
    max_attempts = 3
    attempt = 1
    driver.get(url)
    handle_alerts(driver)
    
    while attempt <= max_attempts:
        try:
            time.sleep(5)  # Wait for the page to load completely

            # Check for popup again before proceeding
            handle_popup(driver)

            # Check for Twitch or Youtube stream
            iframes = driver.find_elements(By.TAG_NAME, "iframe")
            skip_channel = False
            for iframe in iframes:
                src = iframe.get_attribute("src")
                if "player.twitch.tv/?channel" in src or "youtu" in src:
                    print(colored(f"Skipping {url} as it appears to be a Twitch or YouTube stream","light_cyan"))
                    skip_channel = True
                    break
            if skip_channel:
                break
                
            found_m3u8 = False
            for request in driver.requests:
                if 'track.m3u8' in request.path:
                    print(f"Successfully found stream at {request.url}")
                    found_m3u8 = True
                    # Run ffmpeg command in a separate thread
                    threading.Thread(target=run_ffmpeg, args=(request.url, line_number)).start()
                    break
            
            if not found_m3u8:
                print(colored(f"{url} appears to be live, but m3u8 not found!!!","light_red"))

            break  # Exit the loop if successful
        except Exception as e:
            print(f"Attempt {attempt}: Error intercepting network requests: {e}")
        finally:
            attempt += 1
    if attempt > max_attempts:
        print(f"Reached maximum retry attempts ({max_attempts}). Unable to intercept network requests.")

def run_ffmpeg(m3u8_url, line_number):
    try:
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S_%f")
        output_file = f"{timestamp}.ts"

        # Construct ffmpeg command
        command = [
            "ffmpeg",
            "-referer", "https://rplay.live/",
            "-user_agent", user_agent,
            "-i", m3u8_url,
            "-c", "copy",
            output_file
        ]
        command_str = ' '.join(command)
        
        # Open a new console window for ffmpeg instance
        subprocess.Popen(["start", "cmd", "/c"] + command, shell=True)

        print(colored(f"Recording to {output_file}","light_green"))

        recorded_channels[line_number] = output_file  # Store filename channel is being recorded to

    except Exception as e:
        print(f"An error occurred while running ffmpeg: {e}")


def is_recently_recorded(line_number):
    """Checks if a channel has been recorded within the last minute."""
    if line_number in recorded_channels:
        ts_filename = recorded_channels[line_number]
        ts_filepath = os.path.join(os.getcwd(), ts_filename)  
        if os.path.exists(ts_filepath):
            last_modified_time = os.path.getmtime(ts_filepath)
            current_time = time.time()  # Current Unix timestamp in seconds
            
            try:
                delta = timedelta(seconds=current_time - last_modified_time)
                if delta < timedelta(minutes=1):
                    return True
                else:
                    # Remove the entry if the file hasn't been modified recently
                    del recorded_channels[line_number]
                    return False
            except ValueError as e:
                print(f"Error calculating delta: {e}")
                return False
        else:
            print(f"TS file '{ts_filename}' not found for line {line_number}")
            # Remove the entry if the file does not exist
            del recorded_channels[line_number]
            return False
    else:
        return False

def close_popup_if_present(driver):
    try:
        popup_close_button = WebDriverWait(driver, 5).until(
            EC.element_to_be_clickable((By.XPATH, "//button[@class='button-close-popup']"))
        )
        popup_close_button.click()
        print(colored("Closed initial popup","light_cyan"))
    except:
        print(colored("No initial popup","light_yellow"))


def login(driver, email, password, username):
    html_content = driver.page_source
    if not username in html_content:
        driver.get("https://rplay.live/")
        try:
            close_popup_if_present(driver)  # Close any initial popup if present

        # First login button to trigger the login popup
            first_login_button = WebDriverWait(driver, 10).until(
                EC.element_to_be_clickable((By.XPATH, "//div[contains(@class, 'bg-charge-btn-back') and contains(text(), 'Login')]"))
            )
            first_login_button.click()

            # Wait for the email field in the popup
            email_field = WebDriverWait(driver, 10).until(
                EC.element_to_be_clickable((By.CSS_SELECTOR, "input[type='email'][placeholder='Email']"))
            )
            email_field.send_keys(email)

        # Fill in the password field
            password_field = driver.find_element(By.CSS_SELECTOR, "input[type='password'][placeholder='Password']")
            password_field.send_keys(password)
        
        # Press Enter after entering the password
            password_field.send_keys(Keys.RETURN)

        # Wait for login to complete
            time.sleep(5)  # Adjust the delay as needed

        # Click on the NSFW button
            nsfw_button = WebDriverWait(driver, 10).until(
                   EC.element_to_be_clickable((By.XPATH, "//button[contains(@class, 'bg-plaxgray-90') and contains(., 'NSFW')]"))
            )
            nsfw_button.click()
        
            time.sleep(2)
        
            print(colored("Login successful!","light_yellow"))
        except Exception as e:
            print(colored(f"LOGIN FAILED!\n Error: {e}","light_red"))
    else:
        print(colored("Still logged in!","light_yellow"))
        
def main():
    channels_file = 'channels.cfg'
    login_file = 'login.cfg'
    login_details = read_channels_file(login_file)
    email = login_details[0]
    password = login_details[1]
    username = login_details[2]
    check_interval = 30   # breather time		

    options = Options()
    options.add_argument("--headless")  # Run in headless mode
    options.add_argument("--mute-audio")  # Mute audio output
    options.add_argument("--log-level=3")
    driver = webdriver.Chrome(service=Service(ChromeDriverManager().install()), options=options)
    
    global user_agent
    user_agent = "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/126.0.0.0 Safari/537.36"
    
    
    while True:
        login(driver, email, password, username)    
        urls = read_channels_file(channels_file)
        for idx, url in enumerate(urls, start=1):
            if not is_recently_recorded(idx):
                check_svg_in_page(driver, url, idx)
            else:
                print(colored(f"Skipping {url} as it is already being recorded.","light_cyan"))
            del driver.requests
        print(colored("Taking a breather...","light_magenta"))
        time.sleep(check_interval)

if __name__ == '__main__':
    main()
