//===================================================================
/**
 * @file    sparring_enemy_actor.cpp
 * @brief   Xp[OGl~[AN^[
 * @author  soga
 * @date    12.05.21
*/
//===================================================================

#include  "sound/sound.h"
#include  "sound/SoundMiddleID.h"
#include  "system/gfl_use.h"
#include  "sparring_enemy_actor.h"
#include  "../sparring/sparring_main.h"
#include  "sparring_act_def.h"

namespace actor {

ACT_MOVE_RESULT (SparringEnemyActor::*SparringEnemyActor::EnemyTypeFuncTbl[])( void* act_move_param ) = 
{ 
  &SparringEnemyActor::MoveFlyFunc,
  &SparringEnemyActor::MoveBulletFunc,
  &SparringEnemyActor::MoveMineFunc,
};

static  void  dir_calc( gfl::math::MTX34* mtx, const gfl::math::VEC3& dir );
static  bool check_bullet_collision( app::sparring::SPARRING_GAME_WORK* sgw,
                                     gfl::math::VEC3& now_pos,
                                     gfl::math::VEC3& old_pos );
static  bool check_touch_panel_hit( s32 scr_x, s32 scr_y, u16 tx, u16 ty );

static  void init_jump_param( f32& jump_speed, f32& jump_accele, f32 jump_height, f32 jump_distance );

//----------------------------------------------------------------------------
/**
 *	@brief    RXgN^
 */
//-----------------------------------------------------------------------------
SparringEnemyActor::SparringEnemyActor() :
  ActorBase(),
  mType( ENEMY_TYPE_NONE ),
  mSpeed( 0.0f, 0.0f, 0.0f ),
  mDamage( 0 )
{ 
}

//-----------------------------------------------------------------------------
/**
 * @brief ֐
 *
 * @retval ACT_MOVE_RESULT_CONTINUE p@ACT_MOVE_RESULT_FINISH I
 */
//-----------------------------------------------------------------------------
ACT_MOVE_RESULT SparringEnemyActor::Move( void* act_move_param )
{ 
  ACT_MOVE_RESULT result = ACT_MOVE_RESULT_CONTINUE;

  if( mType == ENEMY_TYPE_NONE ) return ACT_MOVE_RESULT_CONTINUE;

  result = ( this->*EnemyTypeFuncTbl[ mType ] )( act_move_param );

  mTranslate += mSpeed;

  return result;
}

//-----------------------------------------------------------------------------
/**
 * @brief Generate֐(FlyType)
 */
//-----------------------------------------------------------------------------
void  SparringEnemyActor::GenerateTypeFly( gfl::math::VEC3& center, f32 radius )
{ 
  mEnemyParam.fly_param.center_x = center.x;
  mEnemyParam.fly_param.center_y = center.y;
  mEnemyParam.fly_param.center_z = center.z;
  mEnemyParam.fly_param.radius   = radius;
  mTranslate = center;
  SetVisible( true );
  mType = ENEMY_TYPE_FLY;
}

//-----------------------------------------------------------------------------
/**
 * @brief Generate֐(BulletType)
 */
//-----------------------------------------------------------------------------
void  SparringEnemyActor::GenerateTypeBullet( gfl::math::VEC3& src, gfl::math::VEC3& dst, f32 speed )
{ 
  mTranslate = src;
  gfl::math::VEC3 distance = dst - src;
  gfl::math::VEC3Normalize( &distance, &distance );
  mSpeed = distance * speed;
  mSeqNo = 0;
  mType = ENEMY_TYPE_BULLET;
  //mEnemyParam.bullet_param.move_frame = gfl::math::VEC3Dist( &src, &dst ) / speed;
  mEnemyParam.bullet_param.move_frame = 300.0f / speed;
  gfl::math::MTX34 mtx;
  dir_calc( &mtx, distance );
  SetRotateMatrix( mtx );
  mEnemyParam.bullet_param.old_x = src.x;
  mEnemyParam.bullet_param.old_y = src.y;
  mEnemyParam.bullet_param.old_z = src.z;
}

//-----------------------------------------------------------------------------
/**
 * @brief Generate֐(MineType)
 */
//-----------------------------------------------------------------------------
void  SparringEnemyActor::GenerateTypeMine( gfl::math::VEC3& src, gfl::math::VEC3& dst, f32 height, f32 speed )
{ 
  GenerateTypeBullet( src, dst, speed );
  mOfs.Set( 0.0f, 0.0f, 0.0f );
  mType = ENEMY_TYPE_MINE;
  init_jump_param( mEnemyParam.mine_param.jump_speed, mEnemyParam.mine_param.jump_accele, height, gfl::math::VEC3Dist( &src, &dst ) / speed );
}

//-----------------------------------------------------------------------------
/**
 * @brief Move֐(FlyType)
 */
//-----------------------------------------------------------------------------
ACT_MOVE_RESULT SparringEnemyActor::MoveFlyFunc( void* act_move_param )
{ 
  app::sparring::SPARRING_GAME_WORK* sgw = ( app::sparring::SPARRING_GAME_WORK* )act_move_param;
  u32 value = xy_system::GflUse::GetPublicRand( 6 );
  f32 speed = ( value & 1 ) ? 1.0f : -1.0f;
  gfl::math::VEC3 center = gfl::math::VEC3( mEnemyParam.fly_param.center_x,
                                            mEnemyParam.fly_param.center_y,
                                            mEnemyParam.fly_param.center_z );
  mSpeed.Set( 0.0f, 0.0f, 0.0f );

  switch( value / 2 ){ 
  case 0:
    mSpeed.x = speed;
    break;
  case 1:
    mSpeed.y = speed;
    break;
  case 2:
    mSpeed.z = speed;
    break;
  }
  
  gfl::math::VEC3 move_pos = mTranslate + mSpeed;

  if( gfl::math::VEC3Dist( &move_pos, &center ) > mEnemyParam.fly_param.radius )
  { 
    mSpeed *= -1;
  }

  //̌vZ
  { 
    gfl::math::MTX34 mtx;
    gfl::math::VEC3 cam_pos;
    sgw->sub_wk->pGraWork->GetCameraPosition( cam_pos );
    gfl::math::VEC3 distance = cam_pos - mTranslate;
    gfl::math::VEC3Normalize( &distance, &distance );
    dir_calc( &mtx, distance );
    SetRotateMatrix( mtx );
  }

  if( ( mEnemyParam.fly_param.bullet_flag == false ) &&
      ( mDamage == 0 ) )
  { 
    if( xy_system::GflUse::GetPublicRand() % 50 == 0 )
    { 
      u32 sysID = sgw->actor_manager->AddSceneActor( ACT_SPARRING_SWP, 0 );
      actor::SparringEnemyActor* enemy_actor = ( actor::SparringEnemyActor* )sgw->actor_manager->GetActorBaseWithSysID( ACT_SPARRING_SWP, sysID );
      gfl::math::VEC3 dst;
      sgw->sub_wk->pGraWork->GetCameraPosition( dst );
      enemy_actor->GenerateTypeBullet( mTranslate, dst, 4.0f );
      enemy_actor->SetAnime( 0 );
      mEnemyParam.fly_param.bullet_flag = true;
      mEnemyParam.fly_param.bullet_id = sysID;
      xy_snd::PlaySE( SEQ_SE_BT_BOWA1 );
    }
  }
  else
  { 
    if( !sgw->actor_manager->IsEnableActorBaseWithSysID( ACT_SPARRING_SWP, mEnemyParam.fly_param.bullet_id ) )
    { 
      mEnemyParam.fly_param.bullet_flag = false;
    }
  }

  if( mDamage )
  { 
    --mDamage;
    bool visible;
    gfl::grp::ColorF32  col;
    if( mDamage == 0 )
    { 
      visible = true;
      col.r = 0.0f;
      col.g = 0.0f;
      col.b = 0.0f;
      col.a = 1.0f;
    }
    else
    { 
      col.r = 0.5f;
      visible = GetVisible();
      visible ^= true;
    }
    SetVisible( visible );
    SetConstantColor( col );
  }
  else
  { 
    if( sgw->shoot )
    { 
      s32 scr_x;
      s32 scr_y;
      sgw->sub_wk->pGraWork->ConvertPosWorldToScreen( mTranslate, &scr_x, &scr_y );
      if( check_touch_panel_hit( scr_x, scr_y, sgw->tx, sgw->ty ) )
      { 
        if( sgw->hit )
        { 
          gfl::math::VEC3 cam_pos;
          sgw->sub_wk->pGraWork->GetCameraPosition( cam_pos );
          if( gfl::math::VEC3Dist( &cam_pos, &mTranslate ) < gfl::math::VEC3Dist( &cam_pos, &sgw->hit_pos ) )
          { 
            sgw->hit_pos = mTranslate;
          }
        }
        else
        { 
          sgw->hit = true;
          sgw->hit_pos = mTranslate;
        }
      }
    }
    if( sgw->actor_collision->CheckCollision( this ) )
    { 
      xy_snd::PlaySE( SEQ_SE_BT_KOUKA_M );
      mDamage = mDamageFrame;
    }
  }


  return ACT_MOVE_RESULT_CONTINUE;
}

//-----------------------------------------------------------------------------
/**
 * @brief Move֐(BulletType)
 */
//-----------------------------------------------------------------------------
ACT_MOVE_RESULT SparringEnemyActor::MoveBulletFunc( void* act_move_param )
{ 
  app::sparring::SPARRING_GAME_WORK* sgw = ( app::sparring::SPARRING_GAME_WORK* )act_move_param;
  ACT_MOVE_RESULT result = ACT_MOVE_RESULT_CONTINUE;
  switch( mSeqNo ){ 
  case 0:
    SetVisible( true );
    mSeqNo++;
    /*fallthru*/
  case 1:
    { 
      gfl::math::VEC3 pos = gfl::math::VEC3( mEnemyParam.bullet_param.old_x,
                                             mEnemyParam.bullet_param.old_y,
                                             mEnemyParam.bullet_param.old_z ); 
      bool hit = check_bullet_collision( sgw, mTranslate, pos );

      mEnemyParam.bullet_param.move_frame -= 1.0f;
      if( ( mEnemyParam.bullet_param.move_frame < 0 ) || ( hit ) )
      { 
        mType = ENEMY_TYPE_NONE;
        result = ACT_MOVE_RESULT_FINISH;
      }
      if( hit )
      { 
        sgw->damage_req = true;
      }
    }
    break;
  }
  mEnemyParam.bullet_param.old_x = mTranslate.x;
  mEnemyParam.bullet_param.old_y = mTranslate.y;
  mEnemyParam.bullet_param.old_z = mTranslate.z;
  return result;
}

//-----------------------------------------------------------------------------
/**
 * @brief Move֐(MineType)
 */
//-----------------------------------------------------------------------------
ACT_MOVE_RESULT SparringEnemyActor::MoveMineFunc( void* act_move_param )
{ 
  app::sparring::SPARRING_GAME_WORK* sgw = ( app::sparring::SPARRING_GAME_WORK* )act_move_param;
  ACT_MOVE_RESULT result = ACT_MOVE_RESULT_CONTINUE;

  switch( mSeqNo ){ 
  case 0:
    SetVisible( true );
    mSeqNo++;
    /*fallthru*/
  case 1:
    mEnemyParam.mine_param.jump_speed -= mEnemyParam.mine_param.jump_accele;
    mOfs.y += mEnemyParam.mine_param.jump_speed;
    if( ( mTranslate.y + mOfs.y < 0.0f ) || ( mCollision ) )
    { 
      mCollision = false;
      sgw->actor_collision->RemoveCollisionTable( mCollisionIndex );
      result = ACT_MOVE_RESULT_FINISH;
    }
    break;
  }

  return result;
}

//-----------------------------------------------------------------------------
/**
 * @brief vZ
 *
 * @param[out]  mtx ]s̊i[
 * @param[in]   dir xNg
 */
//-----------------------------------------------------------------------------
static  void  dir_calc( gfl::math::MTX34* mtx, const gfl::math::VEC3& dir )
{ 
  gfl::math::VEC3 base = gfl::math::VEC3( 0.0f, 0.0f, 1.0f );
  f32 dot = gfl::math::VEC3Dot( &base, &dir );
  f32 angle = gfl::math::AcosRad( dot );
  gfl::math::VEC3 axis;
  gfl::math::VEC3Cross( &axis, &base, &dir );
  if( axis.Length() != 0.0f )
  { 
    gfl::math::MTX34RotAxisRad( mtx, &axis, angle );
  }
  else
  { 
    mtx->Identity();
  }
}

//-----------------------------------------------------------------------------
/**
 * @brief eۏՓ˔
 *
 * @param[in] sgw     SPARRING_GAME_WORK
 * @param[in] now_pos eۂ݈̌ʒu
 * @param[in] old_pos eۂ1t[Öʒu
 *
 * @retval  true:qbg false:m[qbg
 */
//-----------------------------------------------------------------------------
static  bool check_bullet_collision( app::sparring::SPARRING_GAME_WORK* sgw,
                                     gfl::math::VEC3& now_pos,
                                     gfl::math::VEC3& old_pos )
{ 
  //ʒuɕω΃m[qbg
  if( now_pos == old_pos ) return false;

  s32 scr_x;
  s32 scr_y;
  sgw->sub_wk->pGraWork->ConvertPosWorldToScreen( now_pos, &scr_x, &scr_y );
  //ʊOȂm[qbg
  if( ( scr_x < 0 ) || ( scr_x > 400 -1 ) || ( scr_y < 0 ) || ( scr_y > 240 -1 ) ) return false;

  gfl::math::VEC4 pos;
  gfl::math::VEC4 vec;
  gfl::math::VEC4 field;
  gfl::math::VEC3 cam_pos;
  gfl::math::VEC3 cam_tar;
  gfl::math::VEC3 norm;

  sgw->sub_wk->pGraWork->GetCameraAndTargetPosition( cam_pos, cam_tar );
  norm = cam_tar - cam_pos;
  gfl::math::VEC3Normalize( &norm, &norm );
  field.x = norm.x;
  field.y = norm.y;
  field.z = norm.z;
  field.w = gfl::math::VEC3Dist( &cam_tar, &cam_pos );

  pos.x = old_pos.x;
  pos.y = old_pos.y;
  pos.z = old_pos.z;
  pos.w = 1;

  vec.x = now_pos.x - old_pos.x;
  vec.y = now_pos.y - old_pos.y;
  vec.z = now_pos.z - old_pos.z;
  vec.w = 0;

  f32 a, b, t;

  a = pos.Dot( field );
  b = vec.Dot( field );

  //ʂƕsȂ̂ŁAm[qbg
  if( b == 0.0f ) return false;

  t = a / b;

  if( t < 0.0f || t > 1.0f ) return false;

  //ړt[ԂɏՓ˂
  return true;
}

//-----------------------------------------------------------------------------
/**
 * @brief ^b`plڐG
 *
 * @param[in] scr_x   fXW
 * @param[in] scr_y   fYW
 * @param[in] tx      ^b`XW
 * @param[in] ty      ^b`YW
 *
 * @retval  true:qbg false:m[qbg
 */
//-----------------------------------------------------------------------------
static  bool check_touch_panel_hit( s32 scr_x, s32 scr_y, u16 tx, u16 ty )
{ 
  bool  result = false;
  f32 value;

  value = ( scr_x - 200 ) * 0.8f;
  scr_x = value + 160.0f;

  if( ( gfl::math::Abs( scr_x - tx ) < 16 ) && ( gfl::math::Abs( scr_y - ty ) < 16 ) )
  { 
    result = true;
  }

  return result;
}

//-----------------------------------------------------------------------------
/**
 * @brief evZ
 *
 * @param[out]  jump_speed    x
 * @param[out]  jump_accele   x
 * @param[in]   jump_height   _
 * @param[in]   jump_distance 
 */
//-----------------------------------------------------------------------------
static  void init_jump_param( f32& jump_speed, f32& jump_accele, f32 jump_height, f32 jump_distance )
{ 
  f32 time1 = jump_distance / 2.0f;
  f32 time2 = time1 + 1.0f;
  f32 jump_time = time1 * time2 * 0.5f;

  jump_accele = jump_height / jump_time;
  jump_speed  = jump_accele * time1;
}

} // namespace actor
