import requests
import shutil
import subprocess
import re
import xml.etree.ElementTree as ET
from pathlib import Path
from concurrent.futures import ThreadPoolExecutor, wait, FIRST_COMPLETED
from requests.adapters import HTTPAdapter
from urllib3.util.retry import Retry
from tqdm import tqdm

# --- Configuration ---
IA_IDENTIFIER = "BBC_Essential_Mix_Collection"
BASE_URL = f"https://archive.org/download/{IA_IDENTIFIER}/"
METADATA_URL = f"https://archive.org/download/{IA_IDENTIFIER}/{IA_IDENTIFIER}_files.xml"
OUTPUT_DIR = Path("essential_mix_downloads")
THREADS = 4 
BITRATE = "96k"

def get_session():
    session = requests.Session()
    retries = Retry(total=3, backoff_factor=1, status_forcelist=[500, 502, 503, 504])
    session.mount("https://", HTTPAdapter(max_retries=retries))
    return session

http = get_session()
HAS_FFMPEG = shutil.which("ffmpeg") is not None

def fetch_archive_files():
    print("Fetching file list from Internet Archive...")
    try:
        r = http.get(METADATA_URL, timeout=15)
        r.raise_for_status()
        root = ET.fromstring(r.content)
        # Only grab MP3s
        files = [f.get('name') for f in root.findall('file') 
                 if f.get('name') and f.get('name').endswith('.mp3')]
        return sorted(files)
    except Exception as e:
        print(f"Error fetching metadata: {e}")
        return []

def parse_selection(selection_str, max_val):
    """Parses strings like '1, 3, 5-10' into a list of indices."""
    indices = set()
    parts = selection_str.replace(',', ' ').split()
    for part in parts:
        if '-' in part:
            try:
                start, end = map(int, part.split('-'))
                indices.update(range(start, end + 1))
            except ValueError: continue
        else:
            try:
                indices.add(int(part))
            except ValueError: continue
    return [i - 1 for i in indices if 1 <= i <= max_val]

def process_mix(filename, convert, slot_id):
    url = f"{BASE_URL}{filename}"
    save_path = OUTPUT_DIR / filename
    temp_path = OUTPUT_DIR / f"{filename}.part"
    opus_path = save_path.with_suffix(".opus")
    
    if opus_path.exists() or (not convert and save_path.exists()):
        return "exists", filename, slot_id

    try:
        with http.get(url, stream=True, timeout=20) as r:
            if r.status_code == 404: return "404", filename, slot_id
            r.raise_for_status()
            total_size = int(r.headers.get('content-length', 0))
            display_name = (filename[:25] + '..') if len(filename) > 27 else filename

            with tqdm(desc=display_name, total=total_size, unit='iB', unit_scale=True, 
                      leave=False, position=slot_id, mininterval=0.1) as bar:
                with open(temp_path, 'wb') as f:
                    for chunk in r.iter_content(chunk_size=16384):
                        f.write(chunk)
                        bar.update(len(chunk))
        
        temp_path.rename(save_path)
        if convert and HAS_FFMPEG:
            cmd = ["ffmpeg", "-y", "-i", str(save_path), "-c:a", "libopus", "-b:a", BITRATE,
                   "-vbr", "on", "-loglevel", "error", str(opus_path)]
            subprocess.run(cmd, check=True, capture_output=True)
            save_path.unlink()
        return "success", filename, slot_id
    except Exception as e:
        if temp_path.exists(): temp_path.unlink()
        return f"error: {e}", filename, slot_id

def main():
    print("=== BBC Essential Mix Downloader (EMdl) ===")
    OUTPUT_DIR.mkdir(exist_ok=True, parents=True)
    
    all_files = fetch_archive_files()
    if not all_files: return

    while True: # Main Search Loop
        search_term = input("\nSearch for artist/year (or 'q' to quit): ").strip().lower()
        
        if search_term == 'q':
            print("Goodbye!")
            break
            
        matches = [f for f in all_files if search_term in f.lower()]
        
        if not matches:
            print("No matches found. Try a different search.")
            continue

        print(f"\nFound {len(matches)} mixes:")
        for i, filename in enumerate(matches, 1):
            print(f"{i:3}. {filename}")

        # Selection Loop
        while True:
            selection_input = input("\nEnter numbers (e.g. 1, 4, 8-12), 'all', or 'b' for back: ").strip().lower()
            
            if selection_input in ['b', 'back']:
                break # Breaks out of selection to return to search prompt
            
            if selection_input == 'all':
                queue = matches
            else:
                selected_indices = parse_selection(selection_input, len(matches))
                queue = [matches[i] for i in selected_indices]

            if not queue:
                print("No valid selection made. Try again or press 'b' to go back.")
                continue

            msg = "Convert to Opus? This saves about 50% in file size, while perserving the sound quality. If file size isn't an issue, press N. [y/n]: "
            do_conv = input(msg).lower().startswith('y')

            print(f"\nStarting {len(queue)} downloads...")
            active_futures = set()
            available_slots = list(range(THREADS))
            idx = 0
            
            with ThreadPoolExecutor(max_workers=THREADS) as executor:
                while idx < len(queue) or active_futures:
                    while len(active_futures) < THREADS and idx < len(queue):
                        slot = available_slots.pop(0)
                        f = executor.submit(process_mix, queue[idx], do_conv, slot)
                        active_futures.add(f)
                        idx += 1

                    if not active_futures: break
                    done, active_futures = wait(active_futures, return_when=FIRST_COMPLETED)
                    for f in done:
                        _, _, slot = f.result()
                        available_slots.append(slot)
                        available_slots.sort()

            print("\n" * THREADS + "Download batch complete.")
            break # After downloading, return to the main search loop

if __name__ == "__main__":
    main()