/*:
 * @author Casper Gaming
 * @url https://www.caspergaming.com/plugins/cgmz/mailbox/
 * @target MZ
 * @base CGMZ_Core
 * @orderAfter CGMZ_Core
 * @plugindesc Add a mailbox and mail system to your game
 * @help
 * ============================================================================
 * For terms and conditions using this plugin in your game please visit:
 * https://www.caspergaming.com/terms-of-use/
 * ============================================================================
 * Become a Patron to get access to beta/alpha plugins plus other goodies!
 * https://www.patreon.com/CasperGamingRPGM
 * ============================================================================
 * Version: Alpha R2
 * ----------------------------------------------------------------------------
 * Compatibility: Only tested with my CGMZ plugins.
 * Made for RPG Maker MZ 1.7.0
 * ----------------------------------------------------------------------------
 * Description: This plugin adds a mail system to your game, where you can
 * send the player mail via plugin commands and they can then open and read
 * that mail via a mailbox scene.
 * ----------------------------------------------------------------------------
 * Documentation:
 * ----------------------------Alpha Notes-------------------------------------
 * Planned features to be added:
 * 1) Option to display that the attachments were collected already
 * 2) Option to hide letter information before opened
 * 3) Option to allow player to delete letters
 *
 * Want additional features not already present/listed above? Make suggestions
 * on the Patreon Post or in my discord under the #suggestions channel!
 * https://discord.gg/Gbx7JXP
 * ----------------------------Text Codes--------------------------------------
 * In the Body of the letter, you can use special text codes to display images
 * Use the text code:
 * %img[x]
 * where x is the corresponding body image you want to display.
 *
 * For example, %img[1] will display the first body image. If using this text
 * code, please make sure the body image actually exists, otherwise you will
 * run into a crash. For example, if you use %img[5] but you only have 1 body
 * image set up in the plugin, this will crash.
 * -------------------------Plugin Commands------------------------------------
 * • Call Scene
 * This calls the mailbox scene.
 *
 * • Send Mail
 * This sends a piece of mail to the player via letter id
 * ---------------------------Script Calls-------------------------------------
 * To call the mailbox scene via JS, use:
 * SceneManager.push(CGMZ_Scene_Mailbox);
 * ---------------------------Saved Games--------------------------------------
 * This plugin fully supports saved games.
 * ✓ You should be able to add this  plugin to a saved game and add new mail
 * ✓ You can modify mail data and it will reflect accurately in game
 * ✓ You can remove this plugin with no issue to save data
 * -----------------------------Filename---------------------------------------
 * The filename of this plugin's JavaScript file MUST be CGMZ_Mailbox.js
 * This is what it comes as when downloaded. The filename is used to load
 * parameters and execute plugin commands. If you change it, things will begin
 * behaving incorrectly and your game will probably crash. Please do not
 * rename the js file.
 * -------------------------Version History------------------------------------
 * Version Alpha R2
 * - Added sound effects for opening letters
 * - Added Toast Manager integration for receive letter and collect attachment
 * 
 * @command Call Scene
 * @desc Calls the Mailbox scene
 * 
 * @command Send Mail
 * @desc Marks a letter as sent to the player
 * 
 * @arg id
 * @desc The id of the letter to send
 *
 * @param Letters
 * @type struct<Mail>[]
 * @desc Set up letters for the mailbox here
 * @default []
 *
 * @param Fallback Settings
 *
 * @param Fallback Unread Image
 * @parent Fallback Settings
 * @type file
 * @dir img/
 * @desc The default image used when the mail is unread, if no letter-specific image provided
 *
 * @param Fallback Read Image
 * @parent Fallback Settings
 * @type file
 * @dir img/
 * @desc The default image used when the mail is read, if no letter-specific image provided
 *
 * @param Fallback Open SE
 * @parent Fallback Settings
 * @type struct<SoundEffect>
 * @default {"Name":"","Volume":"90","Pitch":"100","Pan":"0"}
 * @desc The default sound effect to play when opening the letter
 *
 * @param Fallback Attachment Toast
 * @parent Fallback Settings
 * @type struct<ToastParam>
 * @default {"Display":"true","Text Line 1":"Attachment Collected:","Text Line 2":"%atchamountx %atchicon%atchname","Sound Effect":"{\"Name\":\"\",\"Volume\":\"90\",\"Pitch\":\"100\",\"Pan\":\"0\"}","Tone":"","Background Style":"Window","Windowskin":"","Width":"0","Height":"0"}
 * @desc The default attachment collected toast
 *
 * @param Fallback Received Toast
 * @parent Fallback Settings
 * @type struct<ToastParam>
 * @default {"Display":"true","Text Line 1":"Letter Received:","Text Line 2":"%subject","Sound Effect":"{\"Name\":\"\",\"Volume\":\"90\",\"Pitch\":\"100\",\"Pan\":\"0\"}","Tone":"{\"Red\":\"-256\",\"Green\":\"0\",\"Blue\":\"0\"}","Background Style":"Window","Windowskin":"","Width":"0","Height":"0"}
 * @desc The default letter received toast
 *
 * @param Scene Options
 *
 * @param Letter Display Info
 * @type select[]
 * @option Subject
 * @option From
 * @option Date
 * @option Body
 * @option Attachment Header
 * @option Attachments
 * @option Blank Line
 * @desc The information and order to display when drawing letter info
 * @default ["Subject","From","Date","Body","Attachment Header","Attachments"]
 * @parent Scene Options
 *
 * @param Transparent Windows
 * @parent Scene Options
 * @type boolean
 * @desc Whether the mailbox windows are transparent or not
 * @default false
 *
 * @param Background Image
 * @parent Scene Options
 * @type file
 * @dir img/pictures
 * @desc Image to show in the background of the scene. Default blurry map used if none provided.
 *
 * @param List Window Width
 * @parent Scene Options
 * @type number
 * @min 0
 * @max 100
 * @desc The width as a percentage of the List Window
 * @default 40
 *
 * @param List Window Right
 * @parent Scene Options
 * @type boolean
 * @desc Determine if the list window displays on the right side of the screen
 * @default false
 *
 * @param Disable Touch UI Space
 * @parent Scene Options
 * @type boolean
 * @desc If true, will not leave space for Touch UI buttons if Touch UI is disabled
 * @default false
 *
 * @param Scroll Speed
 * @parent Scene Options
 * @type number
 * @min 0
 * @desc speed at which the mailbox display window scrolls (if needed)
 * @default 1
 *
 * @param Scroll Wait
 * @parent Scene Options
 * @type number
 * @min 0
 * @desc amount of time (in frames) to wait before beginning to scroll
 * @default 300
 *
 * @param Scroll Deceleration
 * @parent Scene Options
 * @type number
 * @min 0.01
 * @max 0.99
 * @decimals 2
 * @desc Rate of deceleration after letting go of touch
 * @default 0.92
 *
 * @param Auto Scroll
 * @parent Scene Options
 * @type boolean
 * @desc Determine if the display window should automatically scroll after so long of no user input
 * @default true
 *
 * @param Play OK Sound
 * @parent Scene Options
 * @type boolean
 * @desc Determine if the list window plays an ok sound when ok is pressed
 * @default false
 *
 * @param Text Options
 *
 * @param Label Text Color
 * @parent Text Options
 * @type color
 * @default 1
 * @desc The color of the label text
 *
 * @param No Mail Text
 * @parent Text Options
 * @desc Text to describe when there is no mail
 * @default No Mail
 *
 * @param Subject Label Text
 * @parent Text Options
 * @desc Text to describe the Subject label
 * @default Subject:  
 *
 * @param From Label Text
 * @parent Text Options
 * @desc Text to describe the From label
 * @default From:  
 *
 * @param Date Label Text
 * @parent Text Options
 * @desc Text to describe the Date label
 * @default Date:  
 *
 * @param Attachment Header Text
 * @parent Text Options
 * @desc Text to describe the Attachments in the Header
 * @default Attached
*/
/*~struct~Mail:
 * @param id
 * @desc The id used to refer to the letter, should be unique and not blank
 *
 * @param From Text
 * @desc The text to display in the "from" line
 *
 * @param Subject Text
 * @desc Text displayed in the "subject" line
 *
 * @param Date Text
 * @desc Text to display in the "date" line
 *
 * @param Body Text
 * @type note
 * @default ""
 * @desc Text to display in the "body" of the letter
 *
 * @param Body Images
 * @type file[]
 * @dir img/
 * @default []
 * @desc Images you can mix into the body of the letter. See documentation.
 *
 * @param Unread Image
 * @type file
 * @dir img/
 * @desc The image used when the mail is unread. Leave blank to use fallback image.
 *
 * @param Read Image
 * @type file
 * @dir img/
 * @desc The image used when the mail is read. Leave blank to use fallback image.
 *
 * @param Attachment Currency
 * @type struct<Currency>[]
 * @default []
 * @desc Currency to attach to the letter
 *
 * @param Attachment Items
 * @type struct<Item>[]
 * @default []
 * @desc Items to attach to the letter
 *
 * @param Attachment Weapons
 * @type struct<Weapon>[]
 * @default []
 * @desc Weapons to attach to the letter
 *
 * @param Attachment Armors
 * @type struct<Armor>[]
 * @default []
 * @desc Armors to attach to the letter
 *
 * @param Open SE
 * @type struct<SoundEffect>
 * @default {"Name":"","Volume":"90","Pitch":"100","Pan":"0"}
 * @desc The sound effect to play when opening the letter
*/
/*~struct~Currency:
 * @param id
 * @desc The id of the currency (if using CGMZ Currency System) to attach
 *
 * @param Amount
 * @type number
 * @default 0
 * @desc Amount of the currency to attach
*/
/*~struct~Item:
 * @param Item
 * @type item
 * @desc The item to attach
 *
 * @param Amount
 * @type number
 * @default 0
 * @desc Amount of the item to attach
*/
/*~struct~Weapon:
 * @param Weapon
 * @type weapon
 * @desc The weapon to attach
 *
 * @param Amount
 * @type number
 * @default 0
 * @desc Amount of the weapon to attach
*/
/*~struct~Armor:
 * @param Armor
 * @type armor
 * @desc The armor to attach
 *
 * @param Amount
 * @type number
 * @default 0
 * @desc Amount of the armor to attach
*/
/*~struct~SoundEffect:
 * @param Name
 * @type file
 * @dir audio/se
 * @desc Sound Effect file to play. Leave blank to not use.
 *
 * @param Volume
 * @type number
 * @default 90
 * @min 0
 * @max 100
 * @desc Volume of the sound effect
 *
 * @param Pitch
 * @type number
 * @default 100
 * @min 50
 * @max 150
 * @desc Pitch of the sound effect
 *
 * @param Pan
 * @type number
 * @default 0
 * @min -100
 * @max 100
 * @desc Pan of the sound effect
*/
/*~struct~Tone:
 * @param Red
 * @type number
 * @min -256
 * @max 255
 * @desc The red in the tone. Set to -256 to not use a custom tone
 *
 * @param Green
 * @type number
 * @min -255
 * @max 255
 * @desc The green in the tone.
 *
 * @param Blue
 * @type number
 * @min -255
 * @max 255
 * @desc The blue in the tone.
*/
/*~struct~ToastParam:
 * @param Display
 * @type boolean
 * @default true
 * @desc If the toast displays or not
 *
 * @param Text Line 1
 * @default Attachment Collected:
 * @desc The text to display on the first line of the toast
 *
 * @param Text Line 2
 * @default %atchamountx %atchicon %atchname
 * @desc The text to display on the second line of the toast
 *
 * @param Sound Effect
 * @type struct<SoundEffect>
 * @default {"Name":"","Volume":"90","Pitch":"100","Pan":"0"}
 * @desc The sound effect to play when the toast displays
 *
 * @param Tone
 * @type struct<Tone>
 * @desc The tone of the toast window
 *
 * @param Background Style
 * @type select
 * @option Transparent
 * @option Dim
 * @option Window
 * @default Window
 * @desc The background style of the toast
 *
 * @param Windowskin
 * @type file
 * @dir img/
 * @desc The windowskin to use for the toast
 *
 * @param Width
 * @type number
 * @min 0
 * @default 0
 * @desc The width (in pixels) of the toast. Leave 0 to use default
 *
 * @param Height
 * @type number
 * @min 0
 * @default 0
 * @desc The height (in text lines) of the toast. Leave 0 to use default
*/
var Imported = Imported || {};
Imported.CGMZ_Mailbox = true;
var CGMZ = CGMZ || {};
CGMZ.Versions = CGMZ.Versions || {};
CGMZ.Versions["Mailbox"] = "Alpha R2";
CGMZ.Mailbox = {};
CGMZ.Mailbox.parameters = PluginManager.parameters('CGMZ_Mailbox');
CGMZ.Mailbox.SceneBackgroundImage = CGMZ.Mailbox.parameters["Background Image"];
CGMZ.Mailbox.FallbackReadImage = CGMZ.Mailbox.parameters["Fallback Read Image"];
CGMZ.Mailbox.FallbackUnreadImage = CGMZ.Mailbox.parameters["Fallback Unread Image"];
CGMZ.Mailbox.NoMailText = CGMZ.Mailbox.parameters["No Mail Text"];
CGMZ.Mailbox.SubjectLabelText = CGMZ.Mailbox.parameters["Subject Label Text"];
CGMZ.Mailbox.FromLabelText = CGMZ.Mailbox.parameters["From Label Text"];
CGMZ.Mailbox.DateLabelText = CGMZ.Mailbox.parameters["Date Label Text"];
CGMZ.Mailbox.AttachmentHeaderText = CGMZ.Mailbox.parameters["Attachment Header Text"];
CGMZ.Mailbox.ListWindowWidth = Number(CGMZ.Mailbox.parameters["List Window Width"]);
CGMZ.Mailbox.LabelTextColor = Number(CGMZ.Mailbox.parameters["Label Text Color"]);
CGMZ.Mailbox.ScrollSpeed = Number(CGMZ.Mailbox.parameters["Scroll Speed"]);
CGMZ.Mailbox.ScrollWait = Number(CGMZ.Mailbox.parameters["Scroll Wait"]);
CGMZ.Mailbox.ScrollDeceleration = parseFloat(CGMZ.Mailbox.parameters["Scroll Deceleration"]);
CGMZ.Mailbox.AutoScroll = (CGMZ.Mailbox.parameters["Auto Scroll"] === "true");
CGMZ.Mailbox.TransparentWindows = (CGMZ.Mailbox.parameters["Transparent Windows"] === "true");
CGMZ.Mailbox.ListWindowRight = (CGMZ.Mailbox.parameters["List Window Right"] === "true");
CGMZ.Mailbox.DisableTouchUISpace = (CGMZ.Mailbox.parameters["Disable Touch UI Space"] === "true");
CGMZ.Mailbox.PlayOkSound = (CGMZ.Mailbox.parameters["Play OK Sound"] === "true");
CGMZ.Mailbox.FallbackOpenSE = CGMZ_Utils.parseSoundEffectJSON(CGMZ.Mailbox.parameters["Fallback Open SE"], "CGMZ Mailbox");
CGMZ.Mailbox.Letters = CGMZ_Utils.parseJSON(CGMZ.Mailbox.parameters["Letters"], [], "CGMZ Mailbox", "Letters parameter was not valid JSON.");
CGMZ.Mailbox.LetterDisplayInfo = CGMZ_Utils.parseJSON(CGMZ.Mailbox.parameters["Letter Display Info"], [], "CGMZ Mailbox", "Letters Display Info parameter was not valid JSON.");
CGMZ.Mailbox.FallbackAttachmentToastParsed = CGMZ_Utils.parseJSON(CGMZ.Mailbox.parameters["Fallback Attachment Toast"], null, "CGMZ Mailbox", "Fallback Attachment Toast parameter was not valid JSON.");
CGMZ.Mailbox.FallbackReceivedToastParsed = CGMZ_Utils.parseJSON(CGMZ.Mailbox.parameters["Fallback Received Toast"], null, "CGMZ Mailbox", "Fallback Received Toast parameter was not valid JSON.");
//=============================================================================
// CGMZ_Mailbox_SaveData
//-----------------------------------------------------------------------------
// Data class used to store mailbox data that is included in save files
//=============================================================================
function CGMZ_Mailbox_SaveData() {
    this.initialize(...arguments);
}
//-----------------------------------------------------------------------------
// Initialize
//-----------------------------------------------------------------------------
CGMZ_Mailbox_SaveData.prototype.initialize = function(letter) {
	this._id = letter.id;
	this._sortOrder = 0;
	this._isRead = false;
	this._isReceived = false;
	this._isItemsCollected = false;
};
//-----------------------------------------------------------------------------
// Receive the letter
//-----------------------------------------------------------------------------
CGMZ_Mailbox_SaveData.prototype.receiveLetter = function() {
	this._isReceived = true;
	this._sortOrder = $cgmz.getMailboxLetterReceivedOrder();
};
//-----------------------------------------------------------------------------
// Read the letter
//-----------------------------------------------------------------------------
CGMZ_Mailbox_SaveData.prototype.readLetter = function() {
	this._isRead = true;
};
//-----------------------------------------------------------------------------
// When letter items are collected
//-----------------------------------------------------------------------------
CGMZ_Mailbox_SaveData.prototype.onCollectItems = function() {
	this._isItemsCollected = true;
};
//=============================================================================
// CGMZ_Mailbox_TempData
//-----------------------------------------------------------------------------
// Data class used to store temporary mailbox data (not saved)
//=============================================================================
function CGMZ_Mailbox_TempData() {
    this.initialize(...arguments);
}
//-----------------------------------------------------------------------------
// Initialize
//-----------------------------------------------------------------------------
CGMZ_Mailbox_TempData.prototype.initialize = function(letter) {
	this._from = letter["From Text"];
	this._subject = letter["Subject Text"];
	this._date = letter["Date Text"];
	this._body = CGMZ_Utils.parseJSON(letter["Body Text"], "", "CGMZ Mailbox", "Could not parse letter body JSON. Check your letter parameters for letter id: " + letter.id);
	this._bodyImages = CGMZ_Utils.parseJSON(letter["Body Images"], [], "CGMZ Mailbox", "Could not parse letter body image JSON. Check your letter parameters for letter id: " + letter.id);
	this._readImage = letter["Read Image"];
	this._unreadImage = letter["Unread Image"];
	this._openSe = CGMZ_Utils.parseSoundEffectJSON(letter["Open SE"], "CGMZ Mailbox");
	this._attachments = [];
	const attachedCurrencies = CGMZ_Utils.parseJSON(letter["Attachment Currency"], [], "CGMZ Mailbox", "Could not parse letter currency attachment JSON. Check your letter parameters for letter id: " + letter.id);
	const attachedItems = CGMZ_Utils.parseJSON(letter["Attachment Items"], [], "CGMZ Mailbox", "Could not parse letter item attachment JSON. Check your letter parameters for letter id: " + letter.id);
	const attachedWeapons = CGMZ_Utils.parseJSON(letter["Attachment Weapons"], [], "CGMZ Mailbox", "Could not parse letter weapon attachment JSON. Check your letter parameters for letter id: " + letter.id);
	const attachedArmors = CGMZ_Utils.parseJSON(letter["Attachment Armors"], [], "CGMZ Mailbox", "Could not parse letter armor attachment JSON. Check your letter parameters for letter id: " + letter.id);
	for(const currency of attachedCurrencies) {
		const data = CGMZ_Utils.parseJSON(currency, null, "CGMZ Mailbox", "Could not parse letter currency attachment JSON. Check your letter parameters for letter id: " + letter.id);
		if(data) {
			const obj = {type: "currency", id: data.id, amount: Number(data.Amount)}
			this._attachments.push(obj);
		}
	}
	for(const item of attachedItems) {
		const data = CGMZ_Utils.parseJSON(item, null, "CGMZ Mailbox", "Could not parse letter item attachment JSON. Check your letter parameters for letter id: " + letter.id);
		if(data) {
			const obj = {type: "item", id: Number(data.Item), amount: Number(data.Amount)}
			this._attachments.push(obj);
		}
	}
	for(const weapon of attachedWeapons) {
		const data = CGMZ_Utils.parseJSON(weapon, null, "CGMZ Mailbox", "Could not parse letter weapon attachment JSON. Check your letter parameters for letter id: " + letter.id);
		if(data) {
			const obj = {type: "weapon", id: Number(data.Weapon), amount: Number(data.Amount)}
			this._attachments.push(obj);
		}
	}
	for(const armor of attachedArmors) {
		const data = CGMZ_Utils.parseJSON(armor, null, "CGMZ Mailbox", "Could not parse letter armor attachment JSON. Check your letter parameters for letter id: " + letter.id);
		if(data) {
			const obj = {type: "armor", id: Number(data.Armor), amount: Number(data.Amount)}
			this._attachments.push(obj);
		}
	}
};
//-----------------------------------------------------------------------------
// Check if this letter has attachments
//-----------------------------------------------------------------------------
CGMZ_Mailbox_TempData.prototype.hasAttachments = function() {
	return this._attachments.length > 0;
};
//-----------------------------------------------------------------------------
// Check if this letter has a custom open se
//-----------------------------------------------------------------------------
CGMZ_Mailbox_TempData.prototype.hasCustomOpenSe = function() {
	return !!this._openSe.name;
};
//-----------------------------------------------------------------------------
// Collect attachments
//-----------------------------------------------------------------------------
CGMZ_Mailbox_TempData.prototype.onCollectItems = function() {
	for(const attachment of this._attachments) {
		const options = {};
		options.atchamount = attachment.amount;
		switch(attachment.type) {
			case "item":
			case "weapon":
			case "armor":
				const dataItem = CGMZ_Utils.lookupItem(attachment.type, attachment.id);
				$gameParty.gainItem(dataItem, attachment.amount, false);
				options.atchicon = dataItem.iconIndex;
				options.atchname = dataItem.name;
				$cgmzTemp.showMailboxToast("attachment", options);
				break;
			case "currency":
				if(!Imported.CGMZ_CurrencySystem || !attachment.id  || attachment.id === "default") {
					$gameParty.gainGold(attachment.amount);
					options.atchname = TextManager.currencyUnit;
					$cgmzTemp.showMailboxToast("attachment", options);
				} else {
					const currency = $cgmz.getCurrency(attachment.id);
					const currencyTemp = $cgmzTemp.getCurrency(attachment.id);
					if(currency && currencyTemp) {
						options.atchicon = currencyTemp._iconIndex;
						options.atchname = currencyTemp._name;
						$cgmzTemp.showMailboxToast("attachment", options);
						currency.gainCurrency(attachment.amount);
					}
				}
		}
	}
};
//=============================================================================
// CGMZ_Core
//-----------------------------------------------------------------------------
// Handle saved Mailbox data
//=============================================================================
//-----------------------------------------------------------------------------
// Initialize mailbox data
//-----------------------------------------------------------------------------
const alias_CGMZ_Mailbox_CGMZ_Core_createPluginData = CGMZ_Core.prototype.createPluginData;
CGMZ_Core.prototype.createPluginData = function() {
	alias_CGMZ_Mailbox_CGMZ_Core_createPluginData.call(this);
	this.initializeMailboxData(false);
};
//-----------------------------------------------------------------------------
// Initialize Mailbox Data
//-----------------------------------------------------------------------------
CGMZ_Core.prototype.initializeMailboxData = function(reinitialize) {
	if(!this._mailboxData || reinitialize) {
		this._mailboxData = {};
		this._mailboxSortOrder = 1;
	}
	for(const letterJSON of CGMZ.Mailbox.Letters) {
		const letter = CGMZ_Utils.parseJSON(letterJSON, null, "CGMZ Mailbox", "Could not parse letter JSON. Check your letter parameters.");
		if(letter && !this._mailboxData[letter.id]) {
			this._mailboxData[letter.id] = new CGMZ_Mailbox_SaveData(letter);
		}
	}
};
//-----------------------------------------------------------------------------
// Check if new letters have been added after load
//-----------------------------------------------------------------------------
const alias_CGMZ_Mailbox_CGMZ_Core_onAfterLoad = CGMZ_Core.prototype.onAfterLoad;
CGMZ_Core.prototype.onAfterLoad = function() {
	alias_CGMZ_Mailbox_CGMZ_Core_onAfterLoad.call(this);
	this.initializeMailboxData(false);
};
//-----------------------------------------------------------------------------
// Get the sort order when a new letter is received
//-----------------------------------------------------------------------------
CGMZ_Core.prototype.getMailboxLetterReceivedOrder = function() {
	return this._mailboxSortOrder++;
};
//-----------------------------------------------------------------------------
// Get Specific Letter
//-----------------------------------------------------------------------------
CGMZ_Core.prototype.getMailboxLetter = function(id) {
	return this._mailboxData[id];
};
//-----------------------------------------------------------------------------
// Get Received Letters, returns array of letter IDs that are received
//-----------------------------------------------------------------------------
CGMZ_Core.prototype.getMailboxReceivedLetters = function() {
	return Object.keys(this._mailboxData).filter(key => this._mailboxData[key]._isReceived);
};
//-----------------------------------------------------------------------------
// Get Read Letters, returns array of letter IDs that are read
//-----------------------------------------------------------------------------
CGMZ_Core.prototype.getMailboxReadLetters = function() {
	return Object.keys(this._mailboxData).filter(key => this._mailboxData[key]._isRead);
};
//-----------------------------------------------------------------------------
// Get Unified Letter Object (save data + temp data)
//-----------------------------------------------------------------------------
CGMZ_Core.prototype.getMailboxUnifiedLetter = function(id) {
	const letterSave = this.getMailboxLetter(id);
	const letterTemp = $cgmzTemp.getMailboxLetter(id);
	return {...letterSave, ...letterTemp};
};
//=============================================================================
// CGMZ_Temp
//-----------------------------------------------------------------------------
// Add temp mailbox data
//=============================================================================
//-----------------------------------------------------------------------------
// Also initialize mailbox data
//-----------------------------------------------------------------------------
const alias_CGMZ_Mailbox_CGMZ_Temp_createPluginData = CGMZ_Temp.prototype.createPluginData;
CGMZ_Temp.prototype.createPluginData = function() {
	alias_CGMZ_Mailbox_CGMZ_Temp_createPluginData.call(this);
	this.initializeMailboxData();
};
//-----------------------------------------------------------------------------
// Initialize Mailbox Temp Data
//-----------------------------------------------------------------------------
CGMZ_Temp.prototype.initializeMailboxData = function() {
	this._mailboxData = {};
	this.initializeMailboxToasts();
	for(const letterJSON of CGMZ.Mailbox.Letters) {
		const letter = CGMZ_Utils.parseJSON(letterJSON, null, "CGMZ Mailbox", "Could not parse letter JSON. Check your letter parameters.");
		if(letter) {
			this._mailboxData[letter.id] = new CGMZ_Mailbox_TempData(letter);
		}
	}
};
//-----------------------------------------------------------------------------
// Initialize Mailbox Toasts
//-----------------------------------------------------------------------------
CGMZ_Temp.prototype.initializeMailboxToasts = function() {
	this._mailboxToasts = {attachment: null, receive: null};
	this._mailboxToasts.attachment = this.setupMailboxToast(CGMZ.Mailbox.FallbackAttachmentToastParsed);
	this._mailboxToasts.receive = this.setupMailboxToast(CGMZ.Mailbox.FallbackReceivedToastParsed);
};
//-----------------------------------------------------------------------------
// Set up a parsed mailbox toast
//-----------------------------------------------------------------------------
CGMZ_Temp.prototype.setupMailboxToast = function(parsedToast) {
	const toast = CGMZ_Utils.setupToast(parsedToast, "CGMZ Mailbox");
	if(toast) {
		toast.isMailboxToast = true;
		toast.line1 = parsedToast["Text Line 1"];
		toast.line2 = parsedToast["Text Line 2"];
	}
	return toast;
};
//-----------------------------------------------------------------------------
// Show a mailbox toast
//-----------------------------------------------------------------------------
CGMZ_Temp.prototype.showMailboxToast = function(type, options) {
	if(!Imported.CGMZ_ToastManager) return;
	const toast = this._mailboxToasts[type];
	if(!toast) return;
	const toastObject = JSON.parse(JSON.stringify(toast)); // make deep copy of toast object so it can be manipulated
	toastObject.mailboxOptions = options;
	this.createNewToast(toastObject);
};
//-----------------------------------------------------------------------------
// Register Mailbox Plugin Commands
//-----------------------------------------------------------------------------
const alias_CGMZ_Mailbox_CGMZ_Temp_registerPluginCommands = CGMZ_Temp.prototype.registerPluginCommands;
CGMZ_Temp.prototype.registerPluginCommands = function() {
	alias_CGMZ_Mailbox_CGMZ_Temp_registerPluginCommands.call(this);
	PluginManager.registerCommand("CGMZ_Mailbox", "Call Scene", this.pluginCommandMailboxCallScene);
	PluginManager.registerCommand("CGMZ_Mailbox", "Send Mail", this.pluginCommandMailboxSendMail);
};
//-----------------------------------------------------------------------------
// Plugin Command - Call Scene
//-----------------------------------------------------------------------------
CGMZ_Temp.prototype.pluginCommandMailboxCallScene = function() {
	SceneManager.push(CGMZ_Scene_Mailbox);
};
//-----------------------------------------------------------------------------
// Plugin Command - Send Mail
//-----------------------------------------------------------------------------
CGMZ_Temp.prototype.pluginCommandMailboxSendMail = function(args) {
	$cgmzTemp.receiveMailboxLetter(args.id);
};
//-----------------------------------------------------------------------------
// Handling when a mailbox letter is received
//-----------------------------------------------------------------------------
CGMZ_Temp.prototype.receiveMailboxLetter = function(id) {
	const letter = $cgmz.getMailboxLetter(id);
	const letterTemp = this.getMailboxLetter(id);
	if(letter && letterTemp) {
		letter.receiveLetter();
		const options = {};
		options.subject = letterTemp._subject;
		this.showMailboxToast("receive", options);
	}
};
//-----------------------------------------------------------------------------
// Get Specific Letter
//-----------------------------------------------------------------------------
CGMZ_Temp.prototype.getMailboxLetter = function(id) {
	return this._mailboxData[id];
};
//=============================================================================
// CGMZ_Scene_Mailbox
//-----------------------------------------------------------------------------
// Handle the mailbox scene
//=============================================================================
function CGMZ_Scene_Mailbox(types) {
    this.initialize.apply(this, arguments);
}
CGMZ_Scene_Mailbox.prototype = Object.create(Scene_MenuBase.prototype);
CGMZ_Scene_Mailbox.prototype.constructor = CGMZ_Scene_Mailbox;
//-----------------------------------------------------------------------------
// Create mailbox windows
//-----------------------------------------------------------------------------
CGMZ_Scene_Mailbox.prototype.create = function() {
    Scene_MenuBase.prototype.create.call(this);
	this.createListWindow();
	this.createDisplayWindow();
};
//-----------------------------------------------------------------------------
// Create list window
//-----------------------------------------------------------------------------
CGMZ_Scene_Mailbox.prototype.createListWindow = function() {
	const rect = this.listWindowRect();
    this._listWindow = new CGMZ_Window_MailboxList(rect);
	this._listWindow.setHandler('cancel', this.popScene.bind(this));
	this._listWindow.setHandler('ok', this.onListOk.bind(this));
	if(this._listWindow._data.length > 0) {
		this._listWindow.select(0);
	}
    this.addWindow(this._listWindow);
};
//-----------------------------------------------------------------------------
// Get the list window rect
//-----------------------------------------------------------------------------
CGMZ_Scene_Mailbox.prototype.listWindowRect = function() {
	const width = Graphics.boxWidth * (CGMZ.Mailbox.ListWindowWidth / 100.0);
	const x = CGMZ.Mailbox.ListWindowRight ? Graphics.boxWidth - width : 0;
	const y = this.hasTouchUI() ? this.buttonAreaHeight() : 0;
	const height = Graphics.boxHeight - y;
    return new Rectangle(x, y, width, height);
};
//-----------------------------------------------------------------------------
// Create display window
//-----------------------------------------------------------------------------
CGMZ_Scene_Mailbox.prototype.createDisplayWindow = function() {
	const rect = this.displayWindowRect();
    this._displayWindow = new CGMZ_Window_MailboxDisplay(rect);
	this._listWindow.setDisplayWindow(this._displayWindow);
    this.addWindow(this._displayWindow);
};
//-----------------------------------------------------------------------------
// Get the display window rect
//-----------------------------------------------------------------------------
CGMZ_Scene_Mailbox.prototype.displayWindowRect = function() {
	const x = CGMZ.Mailbox.ListWindowRight ? 0 : this._listWindow.width;
	const y = this._listWindow.y;
	const height = this._listWindow.height;
	const width = Graphics.boxWidth - this._listWindow.width;
    return new Rectangle(x, y, width, height);
};
//-----------------------------------------------------------------------------
// Check if should make room for Touch UI
//-----------------------------------------------------------------------------
CGMZ_Scene_Mailbox.prototype.hasTouchUI = function() {
	return !CGMZ.Mailbox.DisableTouchUISpace || ConfigManager.touchUI;
};
//-----------------------------------------------------------------------------
// Add custom background image
//-----------------------------------------------------------------------------
CGMZ_Scene_Mailbox.prototype.createBackground = function() {
	if(CGMZ.Mailbox.SceneBackgroundImage) {
		this._backgroundCustomSprite = new Sprite();
		this._backgroundCustomSprite.bitmap = ImageManager.loadPicture(CGMZ.Mailbox.SceneBackgroundImage);
		this.addChild(this._backgroundCustomSprite);
	} else {
		Scene_MenuBase.prototype.createBackground.call(this);
	}
};
//-----------------------------------------------------------------------------
// On List OK
//-----------------------------------------------------------------------------
CGMZ_Scene_Mailbox.prototype.onListOk = function() {
	const letter = this._listWindow.item();
	const saveData = $cgmz.getMailboxLetter(letter._id);
	const tempData = $cgmzTemp.getMailboxLetter(letter._id);
	if(!saveData._isRead) {
		this.playLetterOpenSe(tempData);
		saveData.readLetter();
		tempData.onCollectItems();
		saveData.onCollectItems();
		this._displayWindow.setItem($cgmz.getMailboxUnifiedLetter(letter._id), true);
	}
	this._listWindow.refresh();
	this._listWindow.activate();
};
//-----------------------------------------------------------------------------
// Play open se
//-----------------------------------------------------------------------------
CGMZ_Scene_Mailbox.prototype.playLetterOpenSe = function(tempLetterObj) {
	if(tempLetterObj.hasCustomOpenSe()) {
		AudioManager.playSe(tempLetterObj._openSe);
	} else {
		if(CGMZ.Mailbox.FallbackOpenSE) AudioManager.playSe(CGMZ.Mailbox.FallbackOpenSE);
	}
};
//=============================================================================
// CGMZ_Window_MailboxList
//-----------------------------------------------------------------------------
// Selectable window for choosing a letter in a list
//=============================================================================
function CGMZ_Window_MailboxList(rect, types) {
    this.initialize.apply(this, arguments);
}
CGMZ_Window_MailboxList.prototype = Object.create(Window_Selectable.prototype);
CGMZ_Window_MailboxList.prototype.constructor = CGMZ_Window_MailboxList;
//-----------------------------------------------------------------------------
// Initialize
//-----------------------------------------------------------------------------
CGMZ_Window_MailboxList.prototype.initialize = function(rect) {
    Window_Selectable.prototype.initialize.call(this, rect);
	this.refresh();
	this.activate();
	this.setBackgroundType(2 * (CGMZ.Mailbox.TransparentWindows));
};
//-----------------------------------------------------------------------------
// Max items
//-----------------------------------------------------------------------------
CGMZ_Window_MailboxList.prototype.maxItems = function() {
    return this._data ? this._data.length : 1;
};
//-----------------------------------------------------------------------------
// Current item
//-----------------------------------------------------------------------------
CGMZ_Window_MailboxList.prototype.item = function() {
    return this._data[this.index()];
};
//-----------------------------------------------------------------------------
// Item Height
//-----------------------------------------------------------------------------
CGMZ_Window_MailboxList.prototype.itemHeight = function() {
    return this.lineHeight() * 2 + 8;
};
//-----------------------------------------------------------------------------
// Play the ok sound
//-----------------------------------------------------------------------------
CGMZ_Window_MailboxList.prototype.playOkSound = function() {
	if(CGMZ.Mailbox.PlayOkSound) Window_Selectable.prototype.playOkSound.call(this);
};
//-----------------------------------------------------------------------------
// Refresh
//-----------------------------------------------------------------------------
CGMZ_Window_MailboxList.prototype.refresh = function() {
    this.makeItemList();
    Window_Selectable.prototype.refresh.call(this);
	this.drawNoMail();
};
//-----------------------------------------------------------------------------
// Make item list
//-----------------------------------------------------------------------------
CGMZ_Window_MailboxList.prototype.makeItemList = function() {
	this._data = [];
	const receivedLetterIDs = $cgmz.getMailboxReceivedLetters();
	for(const letter of receivedLetterIDs) {
		this._data.push($cgmz.getMailboxUnifiedLetter(letter));
	}
	this._data.sort((a, b) => (a._sortOrder > b._sortOrder) ? -1 : 1);
};
//-----------------------------------------------------------------------------
// Draw text when no mail
//-----------------------------------------------------------------------------
CGMZ_Window_MailboxList.prototype.drawNoMail = function() {
	if(this._data.length < 1) {
		this.resetFontSettings();
		this.drawText(CGMZ.Mailbox.NoMailText, 0, 0, this.contents.width, 'center');
	}
};
//-----------------------------------------------------------------------------
// Draw item in list
//-----------------------------------------------------------------------------
CGMZ_Window_MailboxList.prototype.drawItem = function(index) {
	this.resetFontSettings();
	const letter = this._data[index];
    const rect = this.itemRectWithPadding(index);
	this.loadBitmap(letter, rect);
	const x = rect.x + 80;
    this.CGMZ_drawTextLine(letter._subject, x, rect.y, rect.width - x, 'left');
	this.CGMZ_drawTextLine(letter._from, x, rect.y + this.lineHeight(), rect.width - x, 'left');
};
//-----------------------------------------------------------------------------
// Load the letter bitmap
//-----------------------------------------------------------------------------
CGMZ_Window_MailboxList.prototype.loadBitmap = function(letter, rect) {
	const path = letter._isRead ? letter._readImage || CGMZ.Mailbox.FallbackReadImage : letter._unreadImage || CGMZ.Mailbox.FallbackUnreadImage;
	const imageData = CGMZ_Utils.getImageData(path, "img");
    const bitmap = ImageManager.loadBitmap(imageData.folder, imageData.filename);
	bitmap.addLoadListener(this.onLetterImageLoad.bind(this, bitmap, rect));
};
//-----------------------------------------------------------------------------
// Draw the letter bitmap after load
//-----------------------------------------------------------------------------
CGMZ_Window_MailboxList.prototype.onLetterImageLoad = function(bitmap, rect) {
	const sw = bitmap.width;
    const sh = bitmap.height;
    const sx = sy = 0;
	const dw = dh = 72;
	const dx = rect.x;
	const dy = rect.y + 4;
    this.contents.blt(bitmap, sx, sy, sw, sh, dx, dy, dw, dh);
};
//-----------------------------------------------------------------------------
// Set display window
//-----------------------------------------------------------------------------
CGMZ_Window_MailboxList.prototype.setDisplayWindow = function(displayWindow) {
    this._displayWindow = displayWindow;
    this.callUpdateHelp();
};
//-----------------------------------------------------------------------------
// See if can update display window
//-----------------------------------------------------------------------------
CGMZ_Window_MailboxList.prototype.callUpdateHelp = function() {
    if(this.active && this._displayWindow) {
		this._displayWindow.setItem(this.item());
	}
};
//=============================================================================
// CGMZ_Window_MailboxDisplay
//-----------------------------------------------------------------------------
// Window displaying mailbox letter information
//=============================================================================
function CGMZ_Window_MailboxDisplay() {
    this.initialize.apply(this, arguments);
}
CGMZ_Window_MailboxDisplay.prototype = Object.create(CGMZ_Window_Scrollable.prototype);
CGMZ_Window_MailboxDisplay.prototype.constructor = CGMZ_Window_MailboxDisplay;
//-----------------------------------------------------------------------------
// Initialize
//-----------------------------------------------------------------------------
CGMZ_Window_MailboxDisplay.prototype.initialize = function(rect) {
	const heightMultiplier = 5; // maximum of 5 windows tall of data to scroll
    CGMZ_Window_Scrollable.prototype.initialize.call(this, rect, heightMultiplier, CGMZ.Mailbox.ScrollWait, CGMZ.Mailbox.ScrollSpeed, CGMZ.Mailbox.AutoScroll, CGMZ.Mailbox.ScrollDeceleration);
	this.setBackgroundType(2 * (CGMZ.Mailbox.TransparentWindows));
	this._bitmaps = [];
	this._totalSpritesToLoad = 0;
	this._bitmapsLoaded = 0;
	this._letter = null;
};
//-----------------------------------------------------------------------------
// Set the letter to be displayed (do nothing if already being displayed)
//-----------------------------------------------------------------------------
CGMZ_Window_MailboxDisplay.prototype.setItem = function(letter, forceRefresh = false) {
	if(!letter) return;
	if(!forceRefresh && this._letter !== null && this._letter._id === letter._id) return;
	this._letter = letter;
	this.setupWindowForNewEntry();
	if(this._letter._bodyImages.length > 0) {
		this.loadAllBodyImages();
	} else {
		this.requestRefresh();
	}
};
//-----------------------------------------------------------------------------
// Load all of the body images to display
//-----------------------------------------------------------------------------
CGMZ_Window_MailboxDisplay.prototype.loadAllBodyImages = function() {
	this._bitmaps = new Array(this._letter._bodyImages.length);
	this._totalSpritesToLoad = this._letter._bodyImages.length;
	this._bitmapsLoaded = 0;
	for(let i = 0; i < this._letter._bodyImages.length; i++) {
		const imageData = CGMZ_Utils.getImageData(this._letter._bodyImages[i], "img");
		const bitmap = ImageManager.loadBitmap(imageData.folder, imageData.filename);
		bitmap.addLoadListener(this.onBitmapLoad.bind(this, bitmap, i));
	}
};
//-----------------------------------------------------------------------------
// Load all of the body images to display
//-----------------------------------------------------------------------------
CGMZ_Window_MailboxDisplay.prototype.onBitmapLoad = function(bitmap, i) {
	this._bitmaps[i] = bitmap;
	this._bitmapsLoaded++;
	if(this._bitmapsLoaded >= this._totalSpritesToLoad) {
		this.requestRefresh();
	}
};
//-----------------------------------------------------------------------------
// Refresh
//-----------------------------------------------------------------------------
CGMZ_Window_MailboxDisplay.prototype.refresh = function() {
	if(!this._letter) return;
	const displayAttachments = (this._letter._attachments.length > 0 && !this._letter._isItemsCollected);
	this.contents.clear();
	this.resetFontSettings();
	for(const infoType of CGMZ.Mailbox.LetterDisplayInfo) {
		switch(infoType) {
			case "Subject":
				this.drawStandardLine(CGMZ.Mailbox.SubjectLabelText, 0, this._neededHeight, this._letter._subject);
				this._neededHeight += this.lineHeight();
				break;
			case "From":
				this.drawStandardLine(CGMZ.Mailbox.FromLabelText, 0, this._neededHeight, this._letter._from);
				this._neededHeight += this.lineHeight();
				break;
			case "Date":
				this.drawStandardLine(CGMZ.Mailbox.DateLabelText, 0, this._neededHeight, this._letter._date);
				this._neededHeight += this.lineHeight();
				break;
			case "Body":
				this.drawMailBody(); // this function handles needed height itself
				break;
			case "Attachment Header":
				if(displayAttachments) {
					this.CGMZ_drawHeader(CGMZ.Mailbox.AttachmentHeaderText, this._neededHeight);
					this._neededHeight += this.lineHeight();
				}
				break;
			case "Attachments":
				if(displayAttachments) {
					this.drawMailAttachments(); // this function handles needed height itself
				}
				break;
			case "Blank Line": this._neededHeight += this.lineHeight();
		}
	}
};
//-----------------------------------------------------------------------------
// Draw standard line
//-----------------------------------------------------------------------------
CGMZ_Window_MailboxDisplay.prototype.drawStandardLine = function(label, x, y, text) {
	const string = "\\c[" + CGMZ.Mailbox.LabelTextColor + "]" + label + "\\c[0]" + text;
	this.CGMZ_drawTextLine(string, x, y, this.contents.width);
};
//-----------------------------------------------------------------------------
// Draw attachments
//-----------------------------------------------------------------------------
CGMZ_Window_MailboxDisplay.prototype.drawMailAttachments = function() {
	for(const attachment of this._letter._attachments) {
		switch(attachment.type) {
			case "currency":
				if(Imported.CGMZ_CurrencySystem && attachment.id) {
					const currency = $cgmzTemp.getCurrency(attachment.id);
					if(currency) {
						const text = attachment.amount + " \\c[" + currency._color + "]\\i[" + currency._iconIndex + "]" + currency._name + "\\c[0]";
						this.CGMZ_drawTextLine(text, 0, this._neededHeight, this.contents.width);
					} else {
						const text = attachment.amount + " \\c[" + CGMZ.CurrencySystem.DefaultColor + "]\\i[" + CGMZ.CurrencySystem.DefaultIconIndex + "]" + CGMZ.CurrencySystem.DefaultName + "\\c[0]";
						this.CGMZ_drawTextLine(text, 0, this._neededHeight, this.contents.width);
					}
				} else {
					const text = attachment.amount + " " + TextManager.currencyUnit;
					this.CGMZ_drawTextLine(text, 0, this._neededHeight, this.contents.width);
				}
				break;
			case "item":
			case "weapon":
			case "armor":
				const item = CGMZ_Utils.lookupItem(attachment.type, attachment.id);
				const text = attachment.amount + " \\i[" + item.iconIndex + "]" + item.name;
				this.CGMZ_drawTextLine(text, 0, this._neededHeight, this.contents.width);
				break;
		}
		this._neededHeight += this.lineHeight();
	}
};
//-----------------------------------------------------------------------------
// Draw body
//-----------------------------------------------------------------------------
CGMZ_Window_MailboxDisplay.prototype.drawMailBody = function() {
	const imgRegex = /%img\[(\d+)\]/gi;
	const imgIds = Array.from(this._letter._body.matchAll(imgRegex), m => Number(m[1]));
	const textRegex = /%img\[\d+\]/;
	const bodyStrings = this._letter._body.split(textRegex);
	for(let i = 0; i < bodyStrings.length; i++) {
		this._neededHeight += this.CGMZ_drawText(bodyStrings[i], 0, 0, this._neededHeight, this.contents.width);
		if(imgIds[i]) {
			const bitmap = this._bitmaps[imgIds[i] - 1];
			const sx = sy = dx = 0;
			const sw = bitmap.width;
			const sh = bitmap.height;
			let scale = 1;
			if(sw > this.contents.width) scale = this.contents.width / sw;
			const dw = sw * scale;
			const dh = sh * scale;
			const dy = this._neededHeight;
			this.contents.blt(bitmap, sx, sy, sw, sh, dx, dy, dw, dh);
			this._neededHeight += dh;
		}
	}
};
if(Imported.CGMZ_ToastManager) {
//=============================================================================
// CGMZ_Window_Toast
//-----------------------------------------------------------------------------
// Handling for Mailbox Toasts
//=============================================================================
//-----------------------------------------------------------------------------
// Handle mailbox toasts
//-----------------------------------------------------------------------------
const alias_CGMZ_Mailbox_CGMZ_Window_Toast_processCustomToast = CGMZ_Window_Toast.prototype.processCustomToast;
CGMZ_Window_Toast.prototype.processCustomToast = function(toastObject) {
	alias_CGMZ_Mailbox_CGMZ_Window_Toast_processCustomToast.call(this, toastObject);
	if(toastObject.hasOwnProperty('isMailboxToast')) {
		const atchamount = toastObject.mailboxOptions.atchamount;
		const atchicon = toastObject.mailboxOptions.atchicon;
		const atchname = toastObject.mailboxOptions.atchname;
		const subject = toastObject.mailboxOptions.subject;
		const line1 = toastObject.line1.replace("%atchamount", atchamount).replace("%atchicon", (atchicon) ? '\\i[' + atchicon + ']' : "").replace("%atchname", atchname).replace("%subject", subject);
		const line2 = toastObject.line2.replace("%atchamount", atchamount).replace("%atchicon", (atchicon) ? '\\i[' + atchicon + ']' : "").replace("%atchname", atchname).replace("%subject", subject);
		this.CGMZ_drawTextLine(line1, 0, 0, this.contents.width, 'center');
		this.CGMZ_drawTextLine(line2, 0, this.lineHeight(), this.contents.width, 'center');
	}
};
}