package com.mcd.restaurant.deployment;

import io.restassured.RestAssured;
import io.restassured.builder.RequestSpecBuilder;
import io.restassured.specification.RequestSpecification;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.web.server.LocalServerPort;
import org.springframework.restdocs.RestDocumentationContextProvider;
import org.springframework.restdocs.RestDocumentationExtension;
import org.springframework.restdocs.operation.preprocess.Preprocessors;
import org.springframework.restdocs.restassured3.RestAssuredRestDocumentation;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit.jupiter.SpringExtension;

import java.util.stream.Collectors;
import java.util.stream.Stream;

import static org.hamcrest.CoreMatchers.*;

@ExtendWith({SpringExtension.class, RestDocumentationExtension.class})
@SpringBootTest(webEnvironment = SpringBootTest.WebEnvironment.RANDOM_PORT)
@ActiveProfiles("test")
class DeploymentControllerTest {

    private RequestSpecification spec;

    @LocalServerPort
    int port;


    @BeforeEach
    public void setUp(RestDocumentationContextProvider restDocumentation) {
        RestAssured.port = port;

        this.spec = new RequestSpecBuilder()
                .addFilter(RestAssuredRestDocumentation.documentationConfiguration(restDocumentation)
                        .operationPreprocessors().withResponseDefaults(Preprocessors.prettyPrint()))
                .build();
    }

    @Test
    void postDeploymentRecordSuccess() {
        String jsonBody = "{\n" +
                "            \"deploymentType\": \"docker\",\n" +
                "    \"deployComponents\": [\n" +
                "        {\n" +
                "            \"componentName\": \"TestComponent\",\n" +
                "            \"updateFromVersion\": \"1.0.3\",\n" +
                "            \"updateToVersion\": \"2\"\n" +
                "        }\n" +
                "    ],\n" +
                "            \"status\": \"InProgress\",\n" +
                "            \"deployedBy\": \"David\",\n" +
                "            \"restaurantIds\": [\n" +
                "                1,2\n" +
                "            ]\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .post("/deployment/records")
                .then().assertThat().statusCode(is(201))
                .body("data.tagId", notNullValue())
                .body("data.deploymentIds", notNullValue())
                .body("data.deploymentIds.size()", equalTo(1));
    }

    @Test
    void postBundleDeploymentRecordSuccess() {
        String jsonBody = "{\n" +
                "            \"deploymentType\": \"crd\",\n" +
                "            \"isBundleDeployment\": \"true\",\n" +
                "            \"bundleId\": \"2\",\n" +
                "    \"deployComponents\": [\n" +
                "        {\n" +
                "            \"componentName\": \"image-cache\"\n" +
                "        }\n" +
                "    ],\n" +
                "            \"status\": \"InProgress\",\n" +
                "            \"deployedBy\": \"David\",\n" +
                "            \"eventBridgeTime\": \"3\",\n" +
                "            \"restaurantIds\": [\n" +
                "                1,2\n" +
                "            ]\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .post("/deployment/records")
                .then().assertThat().statusCode(is(201))
                .body("data.tagId", notNullValue())
                .body("data.deploymentIds", notNullValue())
                .body("data.deploymentIds.size()", equalTo(1));
    }

    @Test
    void postBundleDeploymentRecordInvalidBundle() {
        String jsonBody = "{\n" +
                "            \"deploymentType\": \"crd\",\n" +
                "            \"isBundleDeployment\": \"true\",\n" +
                "            \"bundleId\": \"6\",\n" +
                "    \"deployComponents\": [\n" +
                "        {\n" +
                "            \"componentName\": \"image-cache\"\n" +
                "        }\n" +
                "    ],\n" +
                "            \"status\": \"InProgress\",\n" +
                "            \"deployedBy\": \"David\",\n" +
                "            \"restaurantIds\": [\n" +
                "                1,2\n" +
                "            ]\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .post("/deployment/records")
                .then().assertThat().statusCode(is(500));
    }

    @Test
    void postBundleDeploymentRecordMissingBundleId() {
        String jsonBody = "{\n" +
                "            \"deploymentType\": \"docker\",\n" +
                "            \"isBundleDeployment\": \"true\",\n" +
                "    \"deployComponents\": [\n" +
                "        {\n" +
                "            \"componentName\": \"image-cache\"\n" +
                "        }\n" +
                "    ],\n" +
                "            \"status\": \"InProgress\",\n" +
                "            \"deployedBy\": \"David\",\n" +
                "            \"restaurantIds\": [\n" +
                "                1,2\n" +
                "            ]\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .post("/deployment/records")
                .then().assertThat().statusCode(is(400));
    }

    @Test
    void postDeploymentRecordNoComponentRequest() {
        String jsonBody = "{\n" +
                "            \"status\": \"InProgress\",\n" +
                "            \"restaurantIds\": [\n" +
                "                1,2\n" +
                "            ]\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .post("/deployment/records")
                .then().assertThat().statusCode(is(400));

    }

    @Test
    void postProductDeploymentRecordSuccess() {
        String jsonBody = "{\n" +
                "            \"deploymentType\": \"docker\",\n" +
                "    \"deployProducts\": [\n" + "2\n" + "],\n" +
                "            \"status\": \"InProgress\",\n" +
                "            \"isProductDeployment\": \"true\",\n" +
                "            \"restaurantIds\": [\n" +
                "                1,2\n" +
                "            ]\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .post("/deployment/records")
                .then().assertThat().statusCode(is(201))
                .body("data.tagId", notNullValue())
                .body("data.deploymentIds", notNullValue())
                .body("data.deploymentIds.size()", equalTo(1));
    }

    @Test
    void postProductDeploymentRecordBadType() {
        String jsonBody = "{\n" +
                "            \"deploymentType\": \"firmware\",\n" +
                "    \"deployProducts\": [\n" + "2\n" + "],\n" +
                "            \"status\": \"InProgress\",\n" +
                "            \"isProductDeployment\": \"true\",\n" +
                "            \"restaurantIds\": [\n" +
                "                1,2\n" +
                "            ]\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .post("/deployment/records")
                .then().assertThat().statusCode(is(400));
    }

    @Test
    void postProductDeploymentRecordBadRequest() {
        String jsonBody = "{\n" +
                "    \"deployComponents\": [\n" +
                "        {\n" +
                "            \"productVersionId\": \"2\",\n" +
                "            \"deploymentType\": \"firmware\"\n" +
                "        }\n" +
                "    ],\n" +
                "            \"status\": \"InProgress\",\n" +
                "            \"isProductDeployment\": \"true\",\n" +
                "            \"restaurantIds\": [\n" +
                "                1,2\n" +
                "            ]\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .post("/deployment/records")
                .then().assertThat().statusCode(is(400));
    }

    @Test
    void postProductDeploymentRecordFailureWhenTypeNotDocker() {
        String jsonBody = "{\n" +
                "    \"deployComponents\": [\n" +
                "        {\n" +
                "            \"deploymentType\": \"docker\"\n" +
                "        }\n" +
                "    ],\n" +
                "            \"status\": \"InProgress\",\n" +
                "            \"isProductDeployment\": \"true\",\n" +
                "            \"restaurantIds\": [\n" +
                "                1,2\n" +
                "            ]\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .post("/deployment/records")
                .then().assertThat().statusCode(is(400));
    }


    @Test
    void postFirmwareDeploymentRecordFailure() {
        String jsonBody = "{\n" +
                "            \"deploymentType\": \"firmware\",\n" +
                "    \"deployComponents\": [\n" +
                "        {\n" +
                "            \"vendor\": \"Axis\",\n" +
                "            \"modelNumber\": \"41\",\n" +
                "            \"updateFromVersion\": \"1.0.3\",\n" +
                "            \"updateToVersion\": \"2\"\n" +
                "        }\n" +
                "    ],\n" +
                "            \"restaurantIds\": [\n" +
                "                2\n" +
                "            ]\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .post("/deployment/records")
                .then().assertThat().statusCode(is(400))
                .body("data.tagId", nullValue())
                .body("data.deploymentIds", nullValue())
                .body("data.deploymentIds.size()", equalTo(null));
    }

    @Test
    void verifyDeploymentHistory() {
        String jsonBody = "{\n" +
                "            \"deploymentType\": \"docker\",\n" +
                "    \"deployComponents\": [\n" +
                "        {\n" +
                "            \"componentName\": \"TestDeploymentHistory\",\n" +
                "            \"updateFromVersion\": \"1.0.3\",\n" +
                "            \"updateToVersion\": \"2\"\n" +
                "        }\n" +
                "    ],\n" +
                "            \"status\": \"Scheduled\",\n" +
                "            \"restaurantIds\": [\n" +
                "                1\n" +
                "            ]\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .post("/deployment/records")
                .then().assertThat().statusCode(is(201))
                .body("data.tagId", notNullValue())
                .body("data.deploymentIds", notNullValue())
                .body("data.deploymentIds.size()", equalTo(1));

        // Verify deployment history
        RestAssured.given(this.spec).accept("application/json")
                .when()
                .get("/restaurant_assets/deployment_history?filter[status]=Scheduled")
                .then().statusCode(is(200))
                .body("data.size", equalTo(1));
    }

    @Test
    void postForComponentNotPresent() {
        String jsonBody = "{\n" +
                "            \"deploymentType\": \"docker\",\n" +
                "    \"deployComponents\": [\n" +
                "        {\n" +
                "            \"componentName\": \"C1\",\n" +
                "            \"updateFromVersion\": \"1.0.3\",\n" +
                "            \"updateToVersion\": \"2\"\n" +

                "        }\n" +
                "    ],\n" +
                "            \"restaurantIds\": [\n" +
                "                2\n" +
                "            ]\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .post("/deployment/records")
                .then().assertThat().statusCode(is(201));
    }

    @Test
    void postForInvalidRestaurant() {
        String jsonBody = "{\n" +
                "            \"deploymentType\": \"docker\",\n" +
                "    \"deployComponents\": [\n" +
                "        {\n" +
                "            \"componentName\": \"TestComponent\",\n" +
                "            \"updateFromVersion\": \"1.0.3\",\n" +
                "            \"updateToVersion\": \"2\"\n" +
                "        }\n" +
                "    ],\n" +
                "            \"restaurantIds\": [\n" +
                "                9\n" +
                "            ]\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .post("/deployment/records")
                .then().assertThat().statusCode(is(400));
    }

    @Test
    void postForInvalidRestaurant_mixed() {
        String jsonBody = "{\n" +
                "            \"deploymentType\": \"docker\",\n" +
                "    \"deployComponents\": [\n" +
                "        {\n" +
                "            \"componentName\": \"TestComponent\",\n" +
                "            \"updateFromVersion\": \"1.0.3\",\n" +
                "            \"updateToVersion\": \"2\"\n" +
                "        }\n" +
                "    ],\n" +
                "            \"status\": \"Scheduled\",\n" +
                "            \"restaurantIds\": [\n" +
                "                1,2,3\n" +
                "            ]\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .post("/deployment/records")
                .then().assertThat().statusCode(is(201));
    }

    @Test
    void patchDeploymentStatus() {
        // Verify Status is different from "In-Progress"
        RestAssured.given(this.spec).accept("application/json")
                .when()
                .get("/restaurant_assets/deployment_history?filter[deployment_group.id]=1")
                .then().statusCode(is(200))
                .body("data.get(0).attributes.status", equalTo("Completed"));

        // Patch Status to In-Progress for the given deployment Id.
        // Enum Value: InProgress
        String jsonBody = "{\n" +
                "    \"status\": \"InProgress\"" +
                "}";
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .patch("/deployment/group/1")
                .then().assertThat().statusCode(is(200));

        // Verify Status is equal to "In-Progress"
        RestAssured.given(this.spec).accept("application/json")
                .when()
                .get("/restaurant_assets/deployment_history?filter[deployment_group.id]=1")
                .then().statusCode(is(200))
                .body("data.get(0).attributes.status", equalTo("In-Progress"));
    }

    @Test
    void patchDeploymentStatus_invalidId() {
        String jsonBody = "{\n" +
                "    \"status\": \"InProgress\"" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .patch("/deployment/group/123")
                .then().assertThat().statusCode(is(500));
    }

    @Test
    void updateDeploymentHistoryStatusSuccess() {
        String jsonBody = "{\n" +
                "    \"applicationName\":\"abc\",\n" +
                "    \"applicationVersion\":\"1.0\",\n" +
                "    \"restaurantNo\":\"101\",\n" +
                "    \"marketName\":\"US\",\n" +
                "    \"deploymentGroupId\":\"4\",\n" +
                "    \"status\":\"Completed\"\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .patch("/deployment/status")
                .then().assertThat().statusCode(is(200));
    }

    @Test
    void updateComponentForStoreSuccess() {
        String jsonBody = "{\n" +
                "    \"applicationName\":\"TestDeploymentHistory\",\n" +
                "    \"applicationVersion\":\"1.0.3\",\n" +
                "    \"restaurantNo\":\"101\",\n" +
                "    \"nameSpace\":\"Test\",\n" +
                "    \"marketName\":\"US\"\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .patch("/deployment/component")
                .then().assertThat().statusCode(is(200));
    }

    @Test
    void updateComponentForInvalidMarket() {
        String jsonBody = "{\n" +
                "    \"applicationName\":\"TestDeploymentHistory\",\n" +
                "    \"applicationVersion\":\"1.0.3\",\n" +
                "    \"restaurantNo\":\"101\",\n" +
                "    \"nameSpace\":\"Test\",\n" +
                "    \"marketName\":\"invalid\"\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .patch("/deployment/component")
                .then().assertThat().statusCode(is(400));
    }

    @Test
    void updateComponentForInvalidStoreNo() {
        String jsonBody = "{\n" +
                "    \"applicationName\":\"TestDeploymentHistory\",\n" +
                "    \"applicationVersion\":\"1.0.3\",\n" +
                "    \"restaurantNo\":\"262456\",\n" +
                "    \"nameSpace\":\"Test\",\n" +
                "    \"marketName\":\"US\"\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .patch("/deployment/component")
                .then().assertThat().statusCode(is(400));
    }

    @Test
    void updateComponentForStoreSuccessMissingComponent() {
        String jsonBody = "{\n" +
                "    \"applicationName\":\"TestDeploymentHistoryNew\",\n" +
                "    \"applicationVersion\":\"262\",\n" +
                "    \"restaurantNo\":\"101\",\n" +
                "    \"nameSpace\":\"Test\",\n" +
                "    \"marketName\":\"US\"\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .patch("/deployment/component")
                .then().assertThat().statusCode(is(200));
    }


    @Test
    void getSchedulerTimeSuccess() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .get("/deployment/scheduler/time")
                .then().assertThat().statusCode(is(200))
                .body("data.failAfter", notNullValue())
                .body("data.intialDelay", notNullValue())
                .body("data.fixedDelay", notNullValue());
    }

    @Test
    void fetchDeploymentDetailsSuccess() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .get("/deployment/details/6")
                .then().assertThat().statusCode(is(200))
                .body("data.storeList", notNullValue())
                .body("data.isProductDeployment", notNullValue())
                .body("data.marketName", notNullValue());
    }

    @Test
    void fetchDeploymentDetailsPreCheckFlowSuccess() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .get("/deployment/details/5")
                .then().assertThat().statusCode(is(200))
                .body("data.storeList", notNullValue())
                .body("data.isProductDeployment", notNullValue())
                .body("data.marketName", notNullValue());
    }

    @Test
    void fetchDeploymentDetailsProductSuccess() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .get("/deployment/details/5")
                .then().assertThat().statusCode(is(200))
                .body("data.storeList", notNullValue())
                .body("data.isProductDeployment", notNullValue())
                .body("data.marketName", notNullValue());
    }

    @Test
    void fetchDeploymentDetailsNoRecordPresent() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .get("/deployment/details/556")
                .then().assertThat().statusCode(is(400));
    }

    @Test
    void getArchivalDataSuccess() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .get("/deployment/archival/group?archiveBefore=1605771000000&limit=100&offset=0")
                .then().assertThat().statusCode(is(200))
                .body("data.deploymentGroupList", notNullValue());
    }

    @Test
    void deletePostArchivalSuccess() {
        String jsonBody = "{\n" +
                "    \"deploymentGroupId\":[3],\n" +
                "    \"deploymentTagId\":[1]\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .delete("/deployment/archival/group")
                .then().assertThat().statusCode(is(200));
    }

    @Test
    void deletePostArchivalException() {
        String jsonBody = "{\n" +
                "   \n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .delete("/deployment/archival/group")
                .then().assertThat().statusCode(is(500));
    }

    @Test
    void postSuggestionListSuccess() {
        String jsonBody = "{\n" +
                "    \"suggestionType\":\"DEPLOYMENT_NAME\",\n" +
                "    \"searchParam\":\"Dem\"\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .post("/deployment/suggestion")
                .then().assertThat().statusCode(is(200));
    }

    @Test
    void postSuggestionListDeployedBYSuccess() {
        String jsonBody = "{\n" +
                "    \"suggestionType\":\"DEPLOYED_BY\",\n" +
                "    \"searchParam\":\"em\"\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .post("/deployment/suggestion")
                .then().assertThat().statusCode(is(200));
    }

    @Test
    void postSuggestionListException() {
        String jsonBody = "{\n" +
                "   \n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .post("/deployment/suggestion")
                .then().assertThat().statusCode(is(500));
    }

    @Test
    void getGraphStatisticsSuccess() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .get("/deployment/graph/details?startDate=1658016000000&endDate=1658153006000&segment=WEEK")
                .then().assertThat().statusCode(is(200));
    }

    @Test
    void postDeploymentRecordForBundleApplicationSuccess() {
        String jsonBody = "{\n" +
                "            \"deploymentGroupId\": \"7\"\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .post("/deployment/bundle")
                .then().assertThat().statusCode(is(201))
                .body("data.deploymentIds", notNullValue())
                .body("data.deploymentIds.size()", equalTo(1));
    }

    @Test
    void postDeploymentRecordForBundleProductSuccess() {
        String jsonBody = "{\n" +
                "            \"deploymentGroupId\": \"8\"\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .post("/deployment/bundle")
                .then().assertThat().statusCode(is(201))
                .body("data.deploymentIds", notNullValue())
                .body("data.deploymentIds.size()", equalTo(1));
    }

    @Test
    void postDeploymentRecordForBundleProductBadRequest() {
        String jsonBody = "{\n" +
                "            \"deploymentGroupId\": \"\"\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .post("/deployment/bundle")
                .then().assertThat().statusCode(is(400));
    }

    @Test
    void postDeploymentRecordForBundleProductGenericException() {
        String jsonBody = "{\n" +
                "            \"deploymentGroupId\": \"500\"\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .post("/deployment/bundle")
                .then().assertThat().statusCode(is(500));
    }

    @Test
    void fetchApplicationDetailApplicationListSuccess() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .get("/deployment/application?suggestionType=APPLICATION_NAME")
                .then().assertThat().statusCode(is(200));
    }

    @Test
    void fetchApplicationDetailApplicationVersionListSuccess() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .get("/deployment/application?suggestionType=APPLICATION_VERSION&applicationName=svcs-restaurant-assets")
                .then().assertThat().statusCode(is(200));
    }

    @Test
    void fetchApplicationDetailBadRequest() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .get("/deployment/application?suggestionType=DEPLOYMENT_NAME")
                .then().assertThat().statusCode(is(400));
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .get("/deployment/application?suggestionType=APPLICATION_VERSION")
                .then().assertThat().statusCode(is(400));
    }

    @Test
    void insertComponentDataSuccess() {
        String jsonBody = "{\n" +
                "    \"storeName\":\"101\",\n" +
                "    \"marketName\":\"US\",\n" +
                "    \"components\":[{\n" +
                "        \"applicationName\":\"AIDT-test\",\n" +
                "        \"applicationVersion\":\"456edr\"\n" +
                "    },\n" +
                "    {\n" +
                "        \"applicationName\":\"test_component_01\",\n" +
                "        \"applicationVersion\":\"1\"\n" +
                "    }]\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .post("/deployment/component/data")
                .then().assertThat().statusCode(is(201));
    }

    @Test
    void postDeploymentCheckRecordBadRequest() {
        String jsonDeploymentIDMissing = "{\n" +
                "    \"status\":\"InProgress\",\n" +
                "    \"type\":\"PRE_DEPLOYMENT_CHECK\",\n" +
                "    \"initiatedBy\":\"Siddhant Kapoor\"\n" +
                "}";
        String jsonTypeMissing = "{\n" +
                "    \"deploymentGroupId\":9,\n" +
                "    \"status\":\"InProgress\",\n" +
                "    \"initiatedBy\":\"Siddhant Kapoor\"\n" +
                "}";
        String jsonInitiatedByMissing = "{\n" +
                "    \"deploymentGroupId\":9,\n" +
                "    \"status\":\"InProgress\",\n" +
                "    \"type\":\"PRE_DEPLOYMENT_CHECK\"\n" +
                "}";

        for (String json : Stream.of(jsonDeploymentIDMissing, jsonTypeMissing, jsonInitiatedByMissing).collect(Collectors.toList())) {
            RestAssured.given(this.spec)
                    .accept("application/json")
                    .contentType("application/json")
                    .body(json)
                    .when()
                    .post("/deployment/check")
                    .then().assertThat().statusCode(is(400));
        }
        ;
    }

    @Test
    void postDeploymentCheckRecordInvalidID() {
        String jsonInvalidId = "{\n" +
                "    \"deploymentGroupId\":200,\n" +
                "    \"status\":\"InProgress\",\n" +
                "    \"type\":\"PRE_DEPLOYMENT_CHECK\",\n" +
                "    \"initiatedBy\":\"Siddhant Kapoor\"\n" +
                "}";
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonInvalidId)
                .when()
                .post("/deployment/check")
                .then().assertThat().statusCode(is(500));

    }

    @Test
    void postDeploymentCheckRecordBadTypeAndStatusRequest() {
        String jsonDeploymentApplication = "{\n" +
                "    \"deploymentGroupId\":11,\n" +
                "    \"status\":\"InProgress\",\n" +
                "    \"type\":\"PRE_DEPLOYMENT_CHECK\",\n" +
                "    \"initiatedBy\":\"Siddhant Kapoor\"\n" +
                "}";
        String jsonCheckInprogress = "{\n" +
                "    \"deploymentGroupId\":9,\n" +
                "    \"status\":\"InProgress\",\n" +
                "    \"type\":\"PRE_DEPLOYMENT_CHECK\",\n" +
                "    \"initiatedBy\":\"Siddhant Kapoor\"\n" +
                "}";

        for (String json : Stream.of(jsonDeploymentApplication, jsonCheckInprogress).collect(Collectors.toList())) {
            RestAssured.given(this.spec)
                    .accept("application/json")
                    .contentType("application/json")
                    .body(json)
                    .when()
                    .post("/deployment/check")
                    .then().assertThat().statusCode(is(400));
        }
        ;
    }



    @Test
    void postDeploymentCheckRecordSuccess() {
        String jsonDeploymentApplication = "{\n" +
                "    \"deploymentGroupId\":13,\n" +
                "    \"status\":\"InProgress\",\n" +
                "    \"type\":\"PRE_DEPLOYMENT_CHECK\",\n" +
                "    \"initiatedBy\":\"Siddhant Kapoor\"\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonDeploymentApplication)
                .when()
                .post("/deployment/check")
                .then().assertThat().statusCode(is(201));
    }

    @Test
    void downloadFileBadRequest() {

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .get("/deployment/download/1")
                .then().assertThat().statusCode(is(400));
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .get("/deployment/download/2")
                .then().assertThat().statusCode(is(500));
    }
    @Test
    void getAllDeploymentQueueDataSuccess() {
        RestAssured.given(this.spec)
                .accept("*/*")
                .contentType("application/vnd.api+json")
                .when()
                .get("/restaurant_assets/deployment_queue?sort=-id,id&page[offset]=0&page[limit]=25&filter[type]=Application,Product,Bundle&filter[status]=Completed,Failed,In-Progress")
                .then().assertThat().statusCode(is(200));
    }

    @Test
    void getAllDeploymentQueueDataException() {
        RestAssured.given(this.spec)
                .accept("*/*")
                .contentType("application/vnd.api+json")
                .when()
                .get("/restaurant_assets/deployment_queue?sort=-id,id&page[offset]=0&page[limit]=25&filter[type]=Application,Product,Bundle&filter[invalidKey]=Completed,Failed,In-Progress")
                .then().assertThat().statusCode(is(500));
    }

    @Test
    void getAllDeploymentQueueDataById() {
        RestAssured.given(this.spec)
                .accept("*/*")
                .contentType("application/vnd.api+json")
                .when()
                .get("/restaurant_assets/deployment_queue/1")
                .then().assertThat().statusCode(is(405));
    }


    @Test
    void patchUpdateDeploymentHistoryStatus() {
        String jsonDeploymentApplication = "{\n" +
                "    \"deploymentGroupId\":\"14\",\n" +
                "    \"unhealthyRestaurants\":[\"101\"]\n" +
                "}";

            RestAssured.given(this.spec)
                    .accept("application/json")
                    .contentType("application/json")
                    .body(jsonDeploymentApplication)
                    .when()
                    .patch("/deployment/precheck/status")
                    .then().assertThat().statusCode(is(200));
    }

    @Test
    void patchUpdateDeploymentHistoryStatusBadRequest() {
        String jsonDeploymentApplication = "{\n" +
                "    \"unhealthyRestaurants\":[\"101\"]\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonDeploymentApplication)
                .when()
                .patch("/deployment/precheck/status")
                .then().assertThat().statusCode(is(400));
    }


    @Test
    void patchUpdateDeploymentStore() {
        String jsonDeploymentApplication = "{\n" +
                "    \"deploymentGroupId\":\"14\",\n" +
                "    \"unSelectedRestaurants\":[\"1\"],\n" +
                "    \"modifiedBy\":\"Test\",\n" +
                "    \"isStatusUpdate\":\"true\"\n" +

                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonDeploymentApplication)
                .when()
                .patch("/deployment/records")
                .then().assertThat().statusCode(is(200));
    }

}
