package com.mcd.restaurant.bundleRelease;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.mcd.restaurant.bundleRelease.testUtils.BundleReleaseTestUtility;
import com.mcd.restaurant.bundledrelease.controller.BundledReleaseController;
import com.mcd.restaurant.bundledrelease.controller.view.request.BundledReleaseRequest;
import com.mcd.restaurant.bundledrelease.controller.view.request.UpdateBundleDeploymentDetailsRequest;
import com.mcd.restaurant.bundledrelease.controller.view.request.UpdateBundleReleaseRequest;
import com.mcd.restaurant.bundledrelease.error.BundleReleaseInvalidRequestException;
import com.mcd.restaurant.bundledrelease.service.BundledReleaseService;
import com.mcd.restaurant.bundledrelease.validator.BundleReleaseValidator;
import com.mcd.restaurant.repository.BundleReleaseAppVersionRepository;
import com.mcd.restaurant.repository.BundleReleaseRepository;
import io.restassured.RestAssured;
import io.restassured.builder.RequestSpecBuilder;
import io.restassured.specification.RequestSpecification;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.web.server.LocalServerPort;
import org.springframework.http.HttpStatus;
import org.springframework.restdocs.RestDocumentationContextProvider;
import org.springframework.restdocs.RestDocumentationExtension;
import org.springframework.restdocs.operation.preprocess.Preprocessors;
import org.springframework.restdocs.restassured3.RestAssuredRestDocumentation;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit.jupiter.SpringExtension;
import org.springframework.test.util.ReflectionTestUtils;

import java.util.ArrayList;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static org.hamcrest.CoreMatchers.*;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertThrows;

@ExtendWith({SpringExtension.class, RestDocumentationExtension.class})
@SpringBootTest(webEnvironment = SpringBootTest.WebEnvironment.RANDOM_PORT)
@ActiveProfiles("test")
class BundleReleaseControllerTest {

    private RequestSpecification spec;

    @LocalServerPort
    int port;

    @InjectMocks
    private BundledReleaseController bundledReleaseController;
    @InjectMocks
    private BundleReleaseTestUtility bundleReleaseTestUtility;
    @Mock
    private BundledReleaseService bundledReleaseService;
    @Mock
    private BundleReleaseValidator bundleReleaseValidator;
    @Mock
    private BundleReleaseAppVersionRepository bundleReleaseAppVersionRepository;
    @Mock
    private BundleReleaseRepository bundleReleaseRepositoryMock;


    @BeforeEach
    public void setUp(RestDocumentationContextProvider restDocumentation) {
        RestAssured.port = port;

        this.spec = new RequestSpecBuilder()
                .addFilter(RestAssuredRestDocumentation.documentationConfiguration(restDocumentation)
                        .operationPreprocessors().withResponseDefaults(Preprocessors.prettyPrint()))
                .build();
    }

    @BeforeEach
    void setUp() {
        MockitoAnnotations.initMocks(this);
    }


    @Test
    void testpostBundleReleaseReleaseNameMissing() {

        BundledReleaseRequest bundledReleaseRequest = bundleReleaseTestUtility.prepareBundleReleaseRequestWithReleaseNameMissing();
        Mockito.doCallRealMethod().when(bundleReleaseValidator).validateBundleReleaseRequest(bundledReleaseRequest);
        assertThrows(BundleReleaseInvalidRequestException.class,
                () -> bundledReleaseController.postBundleRelease(bundledReleaseRequest));
    }


    @Test
    void testpostBundleReleaseCreatedByMissing() {
        BundledReleaseRequest bundledReleaseRequest = bundleReleaseTestUtility.prepareBundleReleaseRequestWithCreatedByMissing();
        Mockito.doCallRealMethod().when(bundleReleaseValidator).validateBundleReleaseRequest(bundledReleaseRequest);
        assertThrows(BundleReleaseInvalidRequestException.class,
                () -> bundledReleaseController.postBundleRelease(bundledReleaseRequest));
    }

    @Test
    void testpostBundleReleaseAppVersionListMissing() {
        BundledReleaseRequest bundledReleaseRequest = bundleReleaseTestUtility.prepareBundleReleaseRequestWithAppVersionListMissing();
        Mockito.doCallRealMethod().when(bundleReleaseValidator).validateBundleReleaseRequest(bundledReleaseRequest);
        assertThrows(BundleReleaseInvalidRequestException.class,
                () -> bundledReleaseController.postBundleRelease(bundledReleaseRequest));
    }

    @Test
    void testpostBundleReleaseAppSuccess() {
        ReflectionTestUtils.setField(bundleReleaseValidator,"bundleReleaseRepository",bundleReleaseRepositoryMock);
        BundledReleaseRequest bundledReleaseRequest = bundleReleaseTestUtility.prepareBundleReleaseRequestSuccess();
        Mockito.doCallRealMethod().when(bundleReleaseValidator).validateBundleReleaseRequest(bundledReleaseRequest);
        Mockito.when(bundleReleaseRepositoryMock.findByReleaseName(Mockito.anyString())).thenReturn(new ArrayList<>());
        Mockito.when(bundledReleaseService.saveBundleRelease(bundledReleaseRequest)).thenReturn(bundleReleaseTestUtility.prepareSuccessResponse());
        assertEquals(HttpStatus.CREATED, bundledReleaseController.postBundleRelease(bundledReleaseRequest).getStatusCode());
    }


    @Test
    void testEditBundleReleasereleaseIdMissing() {
        UpdateBundleReleaseRequest bundledReleaseRequest = bundleReleaseTestUtility.prepareUpdateBundleReleaseRequestWithReleaseIdMissing();
        Mockito.doCallRealMethod().when(bundleReleaseValidator).validateUpdateBundleReleaseRequest(bundledReleaseRequest);
        assertThrows(BundleReleaseInvalidRequestException.class,
                () -> bundledReleaseController.editBundleRelease(bundledReleaseRequest));
    }

    @Test
    void testEditBundleReleaseAppVersionListMissing() {
        UpdateBundleReleaseRequest bundledReleaseRequest = bundleReleaseTestUtility.prepareUpdateBundleReleaseRequestWithAppsMissing();
        Mockito.doCallRealMethod().when(bundleReleaseValidator).validateUpdateBundleReleaseRequest(bundledReleaseRequest);
        assertThrows(BundleReleaseInvalidRequestException.class,
                () -> bundledReleaseController.editBundleRelease(bundledReleaseRequest));
    }

    @Test
    void testEditBundleReleaseAppSuccess() {
        UpdateBundleReleaseRequest bundledReleaseRequest = bundleReleaseTestUtility.prepareUpdateBundleReleaseRequestSuccess();
        Mockito.doCallRealMethod().when(bundleReleaseValidator).validateUpdateBundleReleaseRequest(bundledReleaseRequest);
        Mockito.when(bundledReleaseService.editBundleRelease(bundledReleaseRequest)).thenReturn(bundleReleaseTestUtility.prepareSuccessResponseForEdit());
        assertEquals(HttpStatus.OK, bundledReleaseController.editBundleRelease(bundledReleaseRequest).getStatusCode());
    }


    @Test
    void postBundleReleaseNameExistException() {
        String jsonBody = "{  \n" +
                "  \"releaseName\": \"R-1\",\n" +
                "  \"templateId\": 1,\n" +
                "  \"createdBy\": \"Jon\",\n" +
                "  \"apps\": [\n" +
                "    {\n" +
                "      \"appName\": \"1\",\n" +
                "      \"version\": \"rel-asrservice-2021-09-25.rc1\",\n" +
                "      \"path\": \"artifactory.bre.mcd.com/docker/aidt-asrservic/rel-asrservice-2021-09-20.rc1\"\n" +
                "    }\n" +
                "  ]\n" +
                "}";
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .post("/bundledRelease/versions")
                .then().assertThat().statusCode(is(400))
                .body("errors[0].detail", equalTo("ReleaseName exist: R-1"));
    }

    @Test
    void postProductBundleReleaseIntegrationSuccess() {
        String jsonBody = "{  \n" +
                "  \"releaseName\": \"R-788888888\",\n" +
                "  \"createdBy\": \"Jon\",\n" +
                "  \"products\":[2]\n" +
                "}";
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .post("/bundledRelease/versions")
                .then().assertThat().statusCode(is(201))
                .body("data.isSuccess", equalTo(true))
                .body("data.totalApps", equalTo(1));
    }

    @Test
    void postProductBundleReleaseIntegrationDuplicateApp() {
        String jsonBody = "{  \n" +
                "  \"releaseName\": \"R-788888888\",\n" +
                "  \"createdBy\": \"Jon\",\n" +
                "  \"products\":[2,3]\n" +
                "}";
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .post("/bundledRelease/versions")
                .then().assertThat().statusCode(is(400));

    }


    @Test
    void postProductBundleReleaseIntegrationBadRequest() {
        String jsonBody = "{  \n" +
                "  \"releaseName\": \"R-788888888\",\n" +
                "  \"createdBy\": \"Jon\"\n" +
                "}";
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .post("/bundledRelease/versions")
                .then().assertThat().statusCode(is(400));
    }

    @Test
    void deleteBundleNotFoundException() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .delete("/bundledRelease/version/400")
                .then().assertThat().statusCode(is(400))
                .body("errors[0].code", equalTo("BUNDLE_NOT_FOUND"));

    }

    @Test
    void deleteBundleInitiatedException() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .delete("/bundledRelease/version/3")
                .then().assertThat().statusCode(is(500))
                .body("errors[0].code", equalTo("BUNDLE_RELEASE_EXCEPTION"))
                .body("errors[0].detail", equalTo("Bundle is 'Initiated', hence cannot be edited"));

    }


    @Test
    void patchUpdateDeploymentDetailsSuccess() {
        String jsonBody = "{\n" +
                " \"releaseId\":\"2\",\n" +
                " \"commitId\":\"CommitID\",\n" +
                " \"gitTag\":\"GITTAG\",\n" +
                " \"status\":\"CREATED\",\n" +
                " \"remarks\":\"ABC\"\n" +
                "}";
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .patch("/bundledRelease/deployment/details")
                .then().assertThat().statusCode(is(200))
                .body("data.isSuccess", equalTo(true))
                .body("data.totalApps", equalTo(1));
    }

    @Test
    void patchUpdateDeploymentDetailsWithOutRemarkSuccess() {
        String jsonBody = "{\n" +
                " \"releaseId\":\"1\",\n" +
                " \"commitId\":\"CommitID\",\n" +
                " \"status\":\"STAGED\",\n" +
                " \"gitTag\":\"GITTAG\"\n" +
                "}";
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .patch("/bundledRelease/deployment/details")
                .then().assertThat().statusCode(is(200))
                .body("data.isSuccess", equalTo(true))
                .body("data.totalApps", equalTo(1));
    }

    @Test
    void patchUpdateDeploymentDetailsBadRequest() throws JsonProcessingException {
        UpdateBundleDeploymentDetailsRequest request1 = new UpdateBundleDeploymentDetailsRequest();
        UpdateBundleDeploymentDetailsRequest request2 = new UpdateBundleDeploymentDetailsRequest();
        request2.setReleaseId(1);
        request2.setCommitId("123");
        UpdateBundleDeploymentDetailsRequest request3 = new UpdateBundleDeploymentDetailsRequest();
        request3.setReleaseId(1);
        request3.setGitTag("tag-12");
        UpdateBundleDeploymentDetailsRequest request4 = new UpdateBundleDeploymentDetailsRequest();
        request4.setReleaseId(1);
        for (UpdateBundleDeploymentDetailsRequest request : Stream.of(request1, request2, request3, request4).collect(Collectors.toList())) {
            String json = new ObjectMapper().writeValueAsString(request);
            RestAssured.given(this.spec)
                    .accept("application/json")
                    .contentType("application/json")
                    .body(json)
                    .when()
                    .patch("/bundledRelease/deployment/details")
                    .then().assertThat().statusCode(is(400));
        }
    }


}
