package com.mcd.restaurant;

import io.restassured.RestAssured;
import io.restassured.builder.RequestSpecBuilder;
import io.restassured.specification.RequestSpecification;
import net.minidev.json.JSONObject;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.CsvSource;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.web.server.LocalServerPort;
import org.springframework.restdocs.RestDocumentationContextProvider;
import org.springframework.restdocs.RestDocumentationExtension;
import org.springframework.restdocs.operation.preprocess.Preprocessors;
import org.springframework.restdocs.restassured3.RestAssuredRestDocumentation;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit.jupiter.SpringExtension;

import static org.hamcrest.CoreMatchers.is;

@ExtendWith({ SpringExtension.class, RestDocumentationExtension.class })
@SpringBootTest(webEnvironment = SpringBootTest.WebEnvironment.RANDOM_PORT)
@ActiveProfiles("test")
 class VendorTests {

	private RequestSpecification spec;

	@LocalServerPort
	int port;

	@BeforeEach
	public void setUp(RestDocumentationContextProvider restDocumentation) {
		RestAssured.port = port;

		this.spec = new RequestSpecBuilder()
				.addFilter(RestAssuredRestDocumentation.documentationConfiguration(restDocumentation)
						.operationPreprocessors().withResponseDefaults(Preprocessors.prettyPrint()))
				.build();
	}

	@Test
	 void getVendorWhenIDIsInvalid() {
		RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/vendors/12").then()
				.assertThat().statusCode(is(404));
	}

	@Test
	 void getVendorWhenIDIsNull() {
		RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/vendors/null").then()
				.assertThat().statusCode(is(500));
	}

	@Test
	 void getVendorWhenIDIsMissing() {
		RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/vendors/").then()
				.statusCode(is(200));
	}

	@Test
	 void postVendorWhenInputJSONIsEmpty() {
		String jsonBody = "";

		RestAssured.given(this.spec).contentType("application/vnd.api+json").body(jsonBody).when()
				.post("/restaurant_assets/vendors").then().assertThat().statusCode(is(400));
	}

	@Test
	 void postVendorWhenInputJSONIsInvalid() {
		String jsonBody = "{}";

		RestAssured.given(this.spec).contentType("application/vnd.api+json").body(jsonBody).when()
				.post("/restaurant_assets/vendors").then().assertThat().statusCode(is(400));
	}

	@Test
	 void postVendorWhenInputJSONIsInvalidDataKeyMissing() {
		JSONObject body = new JSONObject();
		JSONObject attributeJSON = new JSONObject();
		attributeJSON.put("name", "Vendor_Name_1");
		attributeJSON.put("notes", "Vendor_Contact_Info_Notes");
		// fill data JSON
		body.put("type", "vendors");
		body.put("attributes", attributeJSON);

		RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
				.post("/restaurant_assets/vendors").then().assertThat().statusCode(is(400));
	}

	@Test
	 void postVendorTypeAndNameMissing() {
		JSONObject body = new JSONObject();
		JSONObject dataJSON = new JSONObject();
		body.put("data", dataJSON);

		RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
				.post("/restaurant_assets/vendors").then().assertThat().statusCode(is(422));
	}

	@Test
	 void postVendorWhenTypeNameBothMissing() {
		JSONObject body = new JSONObject();
		JSONObject dataJSON = new JSONObject();
		JSONObject attributeJSON = new JSONObject();
		attributeJSON.put("notes", "Vendor_Contact_Info_Notes");
		// fill data JSON
		dataJSON.put("attributes", attributeJSON);
		body.put("data", dataJSON);

		RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
				.post("/restaurant_assets/vendors").then().assertThat().statusCode(is(422));
	}

	@Test
	 void postVendorWhenNotesIsMissing() {
		JSONObject body = new JSONObject();
		JSONObject dataJSON = new JSONObject();
		JSONObject attributeJSON = new JSONObject();
		attributeJSON.put("name", "Vendor_Name_1");
		// fill data JSON
		dataJSON.put("type", "vendors");
		dataJSON.put("attributes", attributeJSON);
		body.put("data", dataJSON);

		RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
				.post("/restaurant_assets/vendors").then().assertThat().statusCode(is(201));
	}

	@Test
	 void postVendorWhenNotesIsEmpty() {
		JSONObject body = new JSONObject();
		JSONObject dataJSON = new JSONObject();
		JSONObject attributeJSON = new JSONObject();
		attributeJSON.put("name", "Vendor_Name_1");
		attributeJSON.put("notes", "");
		// fill data JSON
		dataJSON.put("type", "vendors");
		dataJSON.put("attributes", attributeJSON);
		body.put("data", dataJSON);

		RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
				.post("/restaurant_assets/vendors").then().assertThat().statusCode(is(201));
	}

	@ParameterizedTest
	@CsvSource({"12,404", "abc,500", "null,500", ",500"})
	 void patchVendorWhenVendorIdIsInvalid(String id,int code) {
		JSONObject body = new JSONObject();
		JSONObject dataJSON = new JSONObject();
		JSONObject attributeJSON = new JSONObject();
		attributeJSON.put("name", "Vendor_Name_1");
		attributeJSON.put("notes", "Vendor_Contact_Info_Notes");
		// fill data JSON
		dataJSON.put("type", "vendors");
		dataJSON.put("attributes", attributeJSON);
		body.put("data", dataJSON);

		RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
				.patch("/restaurant_assets/vendors/"+id).then().assertThat().statusCode(is(code));
	}

	@Test
	 void patchVendorWhenVendorIdIsMising() {
		JSONObject body = new JSONObject();
		JSONObject dataJSON = new JSONObject();
		JSONObject attributeJSON = new JSONObject();
		attributeJSON.put("name", "Vendor_Name_1");
		attributeJSON.put("notes", "Vendor_Contact_Info_Notes");
		// fill data JSON
		dataJSON.put("type", "vendors");
		dataJSON.put("attributes", attributeJSON);
		body.put("data", dataJSON);

		RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
				.patch("/restaurant_assets/vendors/").then().assertThat().statusCode(is(400));
	}


	@Test
	 void patchVendorWhenInputJSONIsEmptyString() {
		String body = "";

		RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
				.patch("/restaurant_assets/vendors/1").then().assertThat().statusCode(is(400));
	}

	@Test
	 void patchVendorWhenInputJSONIsEmptyAndVendorIdIsInvalid() {
		String body = "";

		RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
				.patch("/restaurant_assets/vendors/abc").then().assertThat().statusCode(is(500));
	}
}
