package com.mcd.restaurant.service;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.util.ContentCachingRequestWrapper;
import org.springframework.web.util.ContentCachingResponseWrapper;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

@Slf4j
@Service
public class ApiLoggingService {

    @Autowired
    private ObjectMapper objectMapper;

    /**
     * Logging for every HTTP Request.
     *
     * @param requestWrapper  request details
     * @param responseWrapper response details
     */
    public void logHttp(ContentCachingRequestWrapper requestWrapper, ContentCachingResponseWrapper responseWrapper) {
        String requestId = UUID.randomUUID().toString();
        MDC.put("request_id", requestId);
        MDC.put("ip", requestWrapper.getRemoteAddr());
        MDC.put("http", getHttpDetails(requestWrapper, responseWrapper));
        log.info("HTTP REQUEST & RESPONSE details logged");
        MDC.clear();
    }

    /**
     * Logging Error
     *
     * @param ex exception
     * @return error id
     */
    public String logError(Throwable ex) {
        String errorId = UUID.randomUUID().toString();
        MDC.put("errorId", errorId);
        log.error("failed to process request", ex);
        return errorId;
    }

    /**
     * Logging Warning
     *
     * @param ex exception
     * @return error id
     */
    public String logWarn(Throwable ex) {
        String errorId = UUID.randomUUID().toString();
        MDC.put("errorId", errorId);
        log.warn("failed to process request", ex);
        return errorId;
    }

    private String getHttpDetails(ContentCachingRequestWrapper requestWrapper,
                                  ContentCachingResponseWrapper responseWrapper) {
        Map<String, Object> nestedMdcHttpMap = new HashMap<>();
        nestedMdcHttpMap.put("url", requestWrapper.getRequestURL().toString());
        nestedMdcHttpMap.put("method", requestWrapper.getMethod());

        Map<String, Object> nestedMdcHttpRequestMap = getHttpRequestMap(requestWrapper);
        nestedMdcHttpMap.put("request", nestedMdcHttpRequestMap);

        Map<String, Object> nestedMdcHttpResponseMap = getHttpResponseMap(responseWrapper);
        nestedMdcHttpMap.put("response", nestedMdcHttpResponseMap);
        try {
            return objectMapper.writeValueAsString(nestedMdcHttpMap);
        } catch (JsonProcessingException e) {
            log.error("failed to process log", e);
        }
        return null;
    }

    private Map<String, Object> getHttpRequestMap(ContentCachingRequestWrapper requestWrapper) {
        Map<String, Object> nestedMdcHttpRequestMap = new HashMap<>();
        nestedMdcHttpRequestMap.put("headers", Collections.list(requestWrapper.getHeaderNames()).stream()
                .map(name -> (name + "=" + requestWrapper.getHeader(name)))
                .collect(Collectors.joining(", ", "{", "}")));
        Map<String, String[]> parameterMap = requestWrapper.getParameterMap();
        nestedMdcHttpRequestMap.put("params", parameterMap.keySet().stream()
                .map(key -> (key + "=" + Arrays.asList(parameterMap.get(key))
                        .toString().substring(1, Arrays.asList(parameterMap.get(key)).toString().length() - 1)))
                .collect(Collectors.joining(", ", "{", "}")));
        nestedMdcHttpRequestMap.put("body", new String(requestWrapper.getContentAsByteArray()));
        return nestedMdcHttpRequestMap;
    }

    private Map<String, Object> getHttpResponseMap(ContentCachingResponseWrapper responseWrapper) {
        Map<String, Object> nestedMdcHttpResponseMap = new HashMap<>();
        nestedMdcHttpResponseMap.put("headers", responseWrapper.getHeaderNames().stream()
                .map(name -> (name + "=" + responseWrapper.getHeader(name)))
                .collect(Collectors.joining(", ", "{", "}")));
        nestedMdcHttpResponseMap.put("statusCode", String.valueOf(responseWrapper.getStatus()));
        return nestedMdcHttpResponseMap;
    }
}
