package com.mcd.restaurant.dashboard.service;


import com.mcd.restaurant.common.MapperUtils;
import com.mcd.restaurant.dashboard.controller.view.request.*;
import com.mcd.restaurant.dashboard.error.StoreDashboardBadRequestException;
import com.mcd.restaurant.dashboard.controller.view.response.*;
import com.mcd.restaurant.dashboard.enums.HierarchyLevel;
import com.mcd.restaurant.dashboard.enums.StoreJobType;
import com.mcd.restaurant.dashboard.error.StoreDashboardException;
import com.mcd.restaurant.dashboard.utils.StoreDashboardConstants;
import com.mcd.restaurant.dashboard.utils.StoreDashboardUtility;
import com.mcd.restaurant.dashboard.utils.csvhandler.CSVHierarchyHandler;
import com.mcd.restaurant.model.*;
import com.mcd.restaurant.repository.*;
import com.mcd.restaurant.repository.bulk.RestaurantBulkRepository;
import io.crnk.core.exception.CrnkException;
import io.crnk.core.queryspec.QuerySpec;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.web.multipart.MultipartFile;

import javax.transaction.Transactional;
import java.io.IOException;
import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;


@Slf4j
@Service("storeDashboardService")
public class StoreDashboardService {


    @Autowired
    private HardwareDetailRepository hardwareDetailRepository;
    @Autowired
    private StoreDashboardUtility storeDashboardUtility;
    @Autowired
    private RestaurantViewRepository restaurantViewRepository;
    @Autowired
    private ComponentRepository componentRepository;
    @Autowired
    private BreHierarchyNodeRepository breHierarchyNodeRepository;
    @Autowired
    private CSVHierarchyHandler csvHierarchyHandler;
    @Autowired
    private StoreDemographicRepository storeDemographicRepository;
    @Autowired
    private MapperUtils mapperUtils;
    @Autowired
    private RestaurantRepository restaurantRepository;
    @Autowired
    private MarketRepository marketRepository;
    @Autowired
    private RestaurantBulkRepository restaurantBulkRepository;

    /**
     * This method consolidate the various store detail information and sends as response
     *
     * @param storeId is the primary key of the restaurant
     * @return dashboardDetailResponseDTO contains the consolidated restaurant details
     */
    @Transactional
    public DashboardDetailResponseDTO getStoreDetails(Integer storeId) {
        try {
            RestaurantView restaurantView = restaurantViewRepository.findOne(storeId, new QuerySpec(RestaurantView.class));
            Integer serverCount = hardwareDetailRepository.findAll(hardwareDetailRepository.prepareQuerySpecFilterForHardwareDetail(storeId)).size();
            Integer platformCount = componentRepository.findAll(componentRepository.prepareQuerySpecFilterForPlatformCount(storeId)).size();
            Integer hardwareCount = componentRepository.findAll(componentRepository.prepareQuerySpecFilterForHardware(storeId)).size();
            return storeDashboardUtility.buildDashboardDetailResponseDTO(restaurantView, serverCount, platformCount, hardwareCount);
        } catch (Exception e) {
            throw new StoreDashboardException("Unable to fetch the details");
        }

    }

    /**
     * This method upload/update the data in bre_hierarchy_node table
     *
     * @param hierarchyFile is the RFM sheet which contains market,region,coop and store data
     * @return BREHierarchyNodeDataUploadResponseDTO containing the total number of data inserted/updated
     */
    @Transactional
    public BREHierarchyNodeDataUploadResponseDTO uploadDataToDataBase(MultipartFile hierarchyFile) {
        try {
            storeDashboardUtility.validateOnGoingStoreJobs();
            storeDashboardUtility.validateFile(hierarchyFile);
            //DS initialization
            BREHierarchyNodeDataUploadResponseDTO responseDTO = new BREHierarchyNodeDataUploadResponseDTO();
            List<BreHierarchyNode> storeBreHierarchyNodes = new ArrayList<>();
            List<BreHierarchyNode> updatedStoreBreHierarchyNodes = new ArrayList<>();

            List<BreHierarchyNode> existingMarket = breHierarchyNodeRepository.findAll(breHierarchyNodeRepository.prepareQuerySpecFilterForBreHierarchyNode(HierarchyLevel.MARKET));
            Map<String, Integer> existingMarketMap = existingMarket.stream().collect(Collectors.toMap(BreHierarchyNode::getNodeName, BreHierarchyNode::getId));
            existingMarket.clear();

            List<BreHierarchyNode> existingRegion = breHierarchyNodeRepository.findAll(breHierarchyNodeRepository.prepareQuerySpecFilterForBreHierarchyNode(HierarchyLevel.REGION));
            Map<String, Integer> existingRegionMap = existingRegion.stream().collect(Collectors.toMap(BreHierarchyNode::getNodeName, BreHierarchyNode::getId));
            Map<Integer, Integer> existingRegionMarketMap = existingRegion.stream().collect(Collectors.toMap(BreHierarchyNode::getId, BreHierarchyNode::getMarketNodeId));
            existingRegion.clear();

            List<BreHierarchyNode> existingCoop = breHierarchyNodeRepository.findAll(breHierarchyNodeRepository.prepareQuerySpecFilterForBreHierarchyNode(HierarchyLevel.COOP));
            Map<String, Integer> existingCoopMap = existingCoop.stream().collect(Collectors.toMap(BreHierarchyNode::getNodeName, BreHierarchyNode::getId));
            Map<Integer, Integer> existingCoopMarketMap = existingCoop.stream().collect(Collectors.toMap(BreHierarchyNode::getId, BreHierarchyNode::getMarketNodeId));
            Map<Integer, Integer> existingCoopRegionMap = existingCoop.stream().collect(Collectors.toMap(BreHierarchyNode::getId, BreHierarchyNode::getRegionNodeId));
            existingCoop.clear();

            List<BreHierarchyNode> existingStore = breHierarchyNodeRepository.findAll(breHierarchyNodeRepository.prepareQuerySpecFilterForBreHierarchyNode(HierarchyLevel.RESTAURANT));
            Map<String, String> existingStoreIpAddress = new HashMap<>();
            existingStore.forEach(store ->
                    existingStoreIpAddress.put(store.getRestaurantNo(), store.getIpAddress())
            );
            Map<String, Integer> existingStoreMap = existingStore.stream().collect(Collectors.toMap(BreHierarchyNode::getRestaurantNo, BreHierarchyNode::getId));
            Map<String, Timestamp> existingStoreCreatedMap = existingStore.stream().collect(Collectors.toMap(BreHierarchyNode::getRestaurantNo, BreHierarchyNode::getCreated));
            Map<String, Integer> existingStoreMarketMap = existingStore.stream().collect(Collectors.toMap(BreHierarchyNode::getRestaurantNo, BreHierarchyNode::getMarketNodeId));
            existingStore.clear();

            //market entry handler
            storeDashboardUtility.marketHierarchyNodeHandler(hierarchyFile, responseDTO, existingMarketMap);
            //region entry handler
            storeDashboardUtility.regionHierarchyNodeHandler(hierarchyFile, responseDTO, existingRegionMap, existingMarketMap, existingRegionMarketMap);
            //coop entry handler
            storeDashboardUtility.coopHierarchyNodeHandler(hierarchyFile, responseDTO, existingRegionMap, existingRegionMarketMap, existingCoopMap, existingCoopMarketMap, existingCoopRegionMap);
            //store entry handler
            storeDashboardUtility.storeHierarchyNodeHandler(hierarchyFile, HierarchyStructureCreationBean.builder().existingCoopMap(existingCoopMap).existingCoopMarketMap(existingCoopMarketMap).existingCoopRegionMap(existingCoopRegionMap).existingStoreMap(existingStoreMap).existingStoreCreatedMap(existingStoreCreatedMap).existingStoreMarketMap(existingStoreMarketMap).existingStoreIpAddress(existingStoreIpAddress).build(), storeBreHierarchyNodes, updatedStoreBreHierarchyNodes);
            StoreDataJob storeDataJob = storeDashboardUtility.saveStoreDataOperation(StoreJobType.HIERARCHY_NODE_INSERTION);
            responseDTO.setStoreInserted(storeBreHierarchyNodes.size());
            responseDTO.setStoreUpdated(updatedStoreBreHierarchyNodes.size());
            storeDashboardUtility.saveStoreHierarchyNode(storeBreHierarchyNodes, updatedStoreBreHierarchyNodes, storeDataJob);
            return responseDTO;
        } catch (Exception e) {
            throw new StoreDashboardException(e.getMessage());
        }
    }

    /**
     * This method upload/update the data in store_demographic table
     *
     * @param nodeFile contains the demographic and other details of the store like contact-number, address etc.
     * @return BREStoreDemographicUploadResponseDTO containing the total number of data inserted/updated
     */
    @Transactional
    public BREStoreDemographicUploadResponseDTO uploadDemographicDataToDataBase(MultipartFile nodeFile) {
        try {
            storeDashboardUtility.validateOnGoingStoreJobs();
            storeDashboardUtility.validateFile(nodeFile);
            BREStoreDemographicUploadResponseDTO storeDemographicUploadResponseDTO = new BREStoreDemographicUploadResponseDTO();
            List<DemographicStructureBean> demographicStructureBeans = csvHierarchyHandler.convertDemographicCSVToBean(nodeFile, "Market");
            if (!demographicStructureBeans.isEmpty() && demographicStructureBeans.size() == 1) {
                handleDemographicDataInsertionForAMarket(nodeFile, storeDemographicUploadResponseDTO, demographicStructureBeans);
            } else {
                throw new StoreDashboardException("Single market information is needed for demographic data uploda into database");
            }
            return storeDemographicUploadResponseDTO;
        } catch (Exception e) {
            throw new StoreDashboardException(e.getMessage());
        }
    }

    /**
     * @param nodeFile                          contains the demographic and other details of the store like contact-number, address etc.
     * @param storeDemographicUploadResponseDTO to update the total number of data inserted/updated
     * @param demographicStructureBeans         contains the converted market data present in RFM sheet
     * @throws IOException
     */
    @Transactional
    public void handleDemographicDataInsertionForAMarket(MultipartFile nodeFile, BREStoreDemographicUploadResponseDTO storeDemographicUploadResponseDTO, List<DemographicStructureBean> demographicStructureBeans) throws IOException {
        List<BreHierarchyNode> marketIdList = breHierarchyNodeRepository.findAll(breHierarchyNodeRepository.prepareQuerySpecFilterForBreHierarchyNodeMarketName(HierarchyLevel.MARKET, demographicStructureBeans.get(0).getNodeName()));
        if (!marketIdList.isEmpty() && marketIdList.size() == 1) {
            Integer marketId = marketIdList.get(0).getId();

            List<BreHierarchyNode> existingStore = breHierarchyNodeRepository.findAll(breHierarchyNodeRepository.prepareQuerySpecFilterForBreHierarchyNodeMarketId(HierarchyLevel.RESTAURANT, marketId));
            Map<String, BreHierarchyNode> existingStoreMap = existingStore.stream().collect(Collectors.toMap(BreHierarchyNode::getRestaurantNo, i -> i));


            List<StoreDemographic> storeDemographics = storeDemographicRepository.findAll(storeDemographicRepository.prepareQuerySpecFilterForStoreDemographicMarketId(HierarchyLevel.RESTAURANT, marketId));
            Map<String, Integer> existingRestaurantNumberSdIdMap = storeDemographics.stream().collect(Collectors.toMap(i -> i.getHierarchyNode().getRestaurantNo(), StoreDemographic::getId));
            Map<String, Timestamp> existingRestaurantNumberCreatedMap = storeDemographics.stream().collect(Collectors.toMap(i -> i.getHierarchyNode().getRestaurantNo(), StoreDemographic::getCreated));
            List<String> existingRestaurantNumber = new ArrayList<>(existingRestaurantNumberSdIdMap.keySet());


            List<DemographicStructureBean> demographicStoreStructureBeans = csvHierarchyHandler.convertDemographicCSVToBean(nodeFile, "Store");
            List<DemographicStructureBean> demographicStructureBeansWithNumber = demographicStoreStructureBeans.stream().filter(i -> !StringUtils.isBlank(i.getRestaurantNumber())).collect(Collectors.toList());
            demographicStructureBeans.clear();
            List<DemographicStructureBean> newDemographicInformation = demographicStructureBeansWithNumber.stream().filter(i -> !existingRestaurantNumber.contains(i.getRestaurantNumber())).collect(Collectors.toList());
            demographicStructureBeansWithNumber.removeAll(newDemographicInformation);

            List<StoreDemographic> newStoreDemographics = new ArrayList<>();

            manageNewDemographicEntries(existingStoreMap, newDemographicInformation, newStoreDemographics);
            storeDemographicUploadResponseDTO.setNewStoreDemograhicsAdded(newStoreDemographics.size());

            List<StoreDemographic> existingStoreDemographics = new ArrayList<>();
            demographicStructureBeansWithNumber.forEach(i -> {
                StoreDemographic storeDemographic = StoreDemographic.builder()
                        .zipCode(i.getZipCode())
                        .latitude(StringUtils.isBlank(i.getLatitude()) ? null : Double.parseDouble(i.getLatitude()))
                        .longitude(StringUtils.isBlank(i.getLongitude()) ? null : Double.parseDouble(i.getLongitude()))
                        .country(i.getCountry())
                        .storeAddress(i.getStoreAddress())
                        .city(i.getCity())
                        .hierarchyNode(existingStoreMap.get(i.getRestaurantNumber()))
                        .contactNumber(i.getContactNumber())
                        .state(i.getState())
                        .ownerName(i.getOwnerName())
                        .ownerType(i.getOwnerType())
                        .build();
                storeDemographic.setId(existingRestaurantNumberSdIdMap.get(i.getRestaurantNumber()));
                storeDemographic.setCreated(existingRestaurantNumberCreatedMap.get(i.getRestaurantNumber()));
                existingStoreDemographics.add(storeDemographic);
            });
            storeDemographicUploadResponseDTO.setUpdatedStoreDemographics(existingStoreDemographics.size());
            StoreDataJob storeDataJob = storeDashboardUtility.saveStoreDataOperation(StoreJobType.STORE_DEMOGRAPHIC_INSERTION);
            storeDashboardUtility.saveChangesForStoreDemographic(newStoreDemographics, existingStoreDemographics, storeDataJob);
        }
    }

    /**
     * This method prepares the new demographic data to be inserted in store_demographic table
     *
     * @param existingStoreMap          contains map of store number and its corresponding BreHierarchyNode object
     * @param newDemographicInformation contains new store data objects from RFM sheet
     * @param newStoreDemographics      contains new entries to be inserted in database
     */
    public void manageNewDemographicEntries(Map<String, BreHierarchyNode> existingStoreMap, List<DemographicStructureBean> newDemographicInformation, List<StoreDemographic> newStoreDemographics) {
        newDemographicInformation.forEach(i -> {
            if (existingStoreMap.get(i.getRestaurantNumber()) != null) {
                StoreDemographic storeDemographic = StoreDemographic.builder()
                        .zipCode(i.getZipCode())
                        .latitude(StringUtils.isBlank(i.getLatitude()) ? null : Double.parseDouble(i.getLatitude()))
                        .longitude(StringUtils.isBlank(i.getLongitude()) ? null : Double.parseDouble(i.getLongitude()))
                        .country(i.getCountry()).storeAddress(i.getStoreAddress())
                        .city(i.getCity())
                        .hierarchyNode(existingStoreMap.get(i.getRestaurantNumber()))
                        .contactNumber(i.getContactNumber())
                        .state(i.getState())
                        .ownerName(i.getOwnerName())
                        .ownerType(i.getOwnerType())
                        .build();
                newStoreDemographics.add(storeDemographic);
            }
        });
    }

    /**
     * This method will map restaurants with its corresponding bre_hierarchy_node data
     *
     * @param marketName present in bre_hierarchy_node table for which restaurants will be mapped to its bre_hierarchy_node data
     * @param dbMarketId present in Market table for which restaurants will be mapped
     * @return total number of restaurants that will be mapped with BreHierarchyNode table
     */
    @Transactional
    public BREStoreToNodeMappingResponeDTO updateRestaurantWithHierarchyNodeId(String marketName, Integer dbMarketId) {
        try {
            storeDashboardUtility.validateOnGoingStoreJobs();
            Map<String, Integer> marketStoreMap = new HashMap<>();
            List<BreHierarchyNode> marketNodes = breHierarchyNodeRepository.findAll(breHierarchyNodeRepository.prepareQuerySpecFilterForBreHierarchyNodeMarketName(HierarchyLevel.MARKET, marketName));
            if (marketNodes.isEmpty()) {
                throw new StoreDashboardException(StoreDashboardConstants.NO_NODE_EXIST);
            }
            if (marketNodes.size() > 1) {
                throw new StoreDashboardException(StoreDashboardConstants.DUPLICATE_NODE_EXIST + marketName);
            }
            BreHierarchyNode individualMarket = marketNodes.get(0);
            List<BreHierarchyNode> existingStore = breHierarchyNodeRepository.findAll(breHierarchyNodeRepository.prepareQuerySpecFilterForBreHierarchyNodeMarketId(HierarchyLevel.RESTAURANT, individualMarket.getId()));
            Map<String, BreHierarchyNode> existingStoreMap = existingStore.stream().collect(Collectors.toMap(BreHierarchyNode::getRestaurantNo, i -> i));
            List<Restaurant> restaurantList = restaurantRepository.findAll(restaurantRepository.prepareQuerySpecFilterForRestaurantByMarketName(dbMarketId));
            List<Restaurant> restaurantUpdatedList = new ArrayList<>();
            for (Restaurant i : restaurantList) {
                if (existingStoreMap.keySet().contains(i.getName())) {
                    i.setBreHierarchyNode(existingStoreMap.get(i.getName()));
                    restaurantUpdatedList.add(i);
                }
            }
            marketStoreMap.put(individualMarket.getNodeName(), restaurantUpdatedList.size());
            StoreDataJob storeDataJob = storeDashboardUtility.saveStoreDataOperation(StoreJobType.HIERARCHY_TO_RESTAURANT_MAPPING);
            storeDashboardUtility.saveRestaurantInBulk(restaurantUpdatedList, storeDataJob);
            return BREStoreToNodeMappingResponeDTO.builder().totalRestaurantMapped(marketStoreMap).build();
        } catch (Exception ex) {
            throw new StoreDashboardException(ex.getMessage());
        }
    }

    /**
     * This methods fetches the list of Market, Region, Coop and store list for hierarchy filtering dropdown feature
     *
     * @return HierarchySearchResponseDTO containing list of Market name, Region name, Coop name and store numbers
     */
    @Transactional
    public HierarchySearchResponseDTO fetchHierarchyFilterData() {
        try {
            List<BreHierarchyNode> marketNodes = breHierarchyNodeRepository.findAll(breHierarchyNodeRepository.prepareQuerySpecFilterForBreHierarchyNode(HierarchyLevel.MARKET));
            List<BreHierarchyNode> regionNodes = breHierarchyNodeRepository.findAll(breHierarchyNodeRepository.prepareQuerySpecFilterForBreHierarchyNode(HierarchyLevel.REGION));
            List<BreHierarchyNode> coopNodes = breHierarchyNodeRepository.findAll(breHierarchyNodeRepository.prepareQuerySpecFilterForBreHierarchyNode(HierarchyLevel.COOP));
            List<BreHierarchyNode> storeNodes = breHierarchyNodeRepository.findAll(breHierarchyNodeRepository.prepareQuerySpecFilterForBreHierarchyNode(HierarchyLevel.RESTAURANT));
            List<Restaurant> mappedRestaurants = restaurantRepository.findAll(restaurantRepository.prepareQuerySpecFilterForRestaurantWithBreHierarchyNodePresent());
            List<String> storeNumber = mappedRestaurants.stream().map(Restaurant::getName).collect(Collectors.toList());
            List<BreHierarchyNode> mappedStore = storeNodes.stream().filter(node -> storeNumber.contains(node.getRestaurantNo())).collect(Collectors.toList());
            return HierarchySearchResponseDTO.builder().markets(marketNodes).regions(regionNodes).coops(coopNodes).stores(mappedStore).build();
        } catch (Exception e) {
            throw new StoreDashboardException(e.getMessage());
        }
    }

    /**
     * This method insert/update the ipaddress for stores present in BreHierarchyNode table
     *
     * @param marketName  present in bre_hierarchy_node table for which restuarants will be updated with its IP Address
     * @param networkFile containing IP-Address for restaurants
     * @return BREStoreIPAddressResponseDTO containing the total number of data inserted/updated
     */
    @Transactional
    public BREStoreIPAddressResponseDTO updateStoreHierarchyNodeWithIpAddress(String marketName, MultipartFile networkFile) {
        Map<String, Integer> totalRestaurantMapped = new HashMap<>();
        try {
            storeDashboardUtility.validateOnGoingStoreJobs();
            storeDashboardUtility.validateFile(networkFile);
            List<BreHierarchyNode> marketNodes = breHierarchyNodeRepository.findAll(breHierarchyNodeRepository.prepareQuerySpecFilterForBreHierarchyNodeMarketName(HierarchyLevel.MARKET, marketName));
            if (marketNodes.isEmpty()) {
                throw new StoreDashboardException(StoreDashboardConstants.NO_NODE_EXIST);
            }
            if (marketNodes.size() > 1) {
                throw new StoreDashboardException(StoreDashboardConstants.DUPLICATE_NODE_EXIST + marketName);
            }
            BreHierarchyNode individualMarket = marketNodes.get(0);
            List<BreHierarchyNode> existingStore = breHierarchyNodeRepository.findAll(breHierarchyNodeRepository.prepareQuerySpecFilterForBreHierarchyNodeMarketId(HierarchyLevel.RESTAURANT, individualMarket.getId()));
            List<NetworkStructureBean> demographicStoreStructureBeans = csvHierarchyHandler.convertNetworkInfoCSVToBean(networkFile);
            Map<String, String> storeNumberToIpaddressMap = demographicStoreStructureBeans.stream().collect(Collectors.toMap(NetworkStructureBean::getRestaurant, NetworkStructureBean::getIpAddress));
            List<BreHierarchyNode> updtaedBreHierarchyNodes = new ArrayList<>();
            existingStore.stream().forEach(store -> {
                if (storeNumberToIpaddressMap.containsKey(store.getRestaurantNo())) {
                    BreHierarchyNode breHierarchyNode = BreHierarchyNode.builder().nodeName(store.getNodeName()).hierarchyLevel(store.getHierarchyLevel()).regionNodeId(store.getRegionNodeId()).coopNodeId(store.getCoopNodeId()).marketNodeId(store.getMarketNodeId()).restaurantNo(store.getRestaurantNo()).ipAddress(storeNumberToIpaddressMap.get(store.getRestaurantNo())).id(store.getId()).created(store.getCreated()).build();
                    updtaedBreHierarchyNodes.add(breHierarchyNode);
                }
            });
            totalRestaurantMapped.put(marketName, updtaedBreHierarchyNodes.size());
            StoreDataJob storeDataJob = storeDashboardUtility.saveStoreDataOperation(StoreJobType.IP_ADDRESS_INSERTION);
            storeDashboardUtility.saveStoreIPAddressInHierarchyNode(updtaedBreHierarchyNodes, storeDataJob);

        } catch (Exception ex) {
            throw new StoreDashboardException(ex.getMessage());
        }
        return BREStoreIPAddressResponseDTO.builder().totalIPAddressMapped(totalRestaurantMapped).build();
    }

    /**
     * This method updates the AOT Status for the given restaurant number
     *
     * @param requestDTO containing the AOT Status and restaurant number for which the AOT status needs to be updated
     * @return StoreStatusChangeResponseDTO containing the response for AOT status update
     */
    @Transactional
    public StoreStatusChangeResponseDTO updateAOTStatusForStore(StoreStatusChangeRequestDTO requestDTO) {
        try {
            Map<Integer, String> storeStatusMap = storeDashboardUtility.getValueNameMapForStoreStatus();

            if (StringUtils.isBlank(requestDTO.getRestaurantNo()) || StringUtils.isBlank(requestDTO.getMarketName()) || requestDTO.getStoreStatus() == null) {
                throw new StoreDashboardBadRequestException(StoreDashboardConstants.MARKET_STORE_NUMBER_ERROR);
            }
            if (storeStatusMap.get(requestDTO.getStoreStatus()) == null) {
                throw new StoreDashboardBadRequestException(StoreDashboardConstants.INVALID_STORE_STATUS_UPDATE);
            }
            List<Market> marketList = marketRepository.findAll(marketRepository.prepareQuerySpecFilterForFilteringMarketByName(requestDTO.getMarketName()));
            if (marketList.isEmpty()) {
                throw new StoreDashboardBadRequestException(StoreDashboardConstants.MARKET_NOT_FOUND);
            }

            List<Restaurant> restaurants = restaurantRepository.findAll(restaurantRepository.prepareQuerySpecFilterForRestaurantByRestaurantNumberAndMarketId(requestDTO.getRestaurantNo(), marketList.get(0).getId()));
            if (restaurants.isEmpty()) {
                throw new StoreDashboardBadRequestException(StoreDashboardConstants.STORE_NOT_FOUND);
            }

            Restaurant restaurant = restaurants.get(0);
            restaurant.setStoreStatus(storeStatusMap.get(requestDTO.getStoreStatus()));
            restaurantRepository.save(restaurant);
            return StoreStatusChangeResponseDTO.builder().isStatusChangeSuccess(true).build();
        } catch (CrnkException ex) {
            throw new StoreDashboardBadRequestException(ex.getMessage());
        } catch (Exception e) {
            throw new StoreDashboardException(e.getMessage());
        }

    }

    @Transactional
    public StoreRegionResponseDTO getDemograhicDetails(HierarchyLevel hierarchyLevel, String marketName, String regionName) {
        try {
            StoreRegionResponseDTO responseDTO = new StoreRegionResponseDTO();
            if (hierarchyLevel.equals(HierarchyLevel.RESTAURANT) && StringUtils.isBlank(regionName)) {
                throw new StoreDashboardBadRequestException(StoreDashboardConstants.REGION_REQUIRED);
            }
            List<BreHierarchyNode> marketNodes = breHierarchyNodeRepository.findAll(breHierarchyNodeRepository.prepareQuerySpecFilterForBreHierarchyNodeMarketName(HierarchyLevel.MARKET, marketName));
            if (marketNodes.isEmpty()) {
                throw new StoreDashboardBadRequestException(StoreDashboardConstants.NO_NODE_EXIST);
            }
            if (marketNodes.size() > 1) {
                throw new StoreDashboardBadRequestException(StoreDashboardConstants.DUPLICATE_NODE_EXIST + marketName);
            }
            BreHierarchyNode individualMarket = marketNodes.get(0);
            switch (hierarchyLevel) {
                case REGION:
                    List<BreHierarchyNode> regionNodes = breHierarchyNodeRepository.findAll(breHierarchyNodeRepository.prepareQuerySpecFilterForBreHierarchyNodeMarketId(HierarchyLevel.REGION, individualMarket.getId()));
                    responseDTO.setNodeList(regionNodes.stream().map(BreHierarchyNode::getNodeName).collect(Collectors.toList()));
                    break;
                case RESTAURANT:
                    List<BreHierarchyNode> regionList = breHierarchyNodeRepository.findAll(breHierarchyNodeRepository.prepareQuerySpecFilterForRegionBreHierarchyNodeMarketId(HierarchyLevel.REGION, individualMarket.getId(), regionName));
                    if (regionList.isEmpty()) {
                        throw new StoreDashboardException("No node exist for the given region name");
                    }
                    BreHierarchyNode individualRegion = regionList.get(0);
                    List<Restaurant> restaurantList = restaurantRepository.findAll(restaurantRepository.prepareQuerySpecFilterForRestaurantByRegionAndMarket(individualRegion.getId(), individualMarket.getId()));
                    responseDTO.setNodeList(restaurantList.stream().map(Restaurant::getName).collect(Collectors.toList()));
                    break;
                default:
                    throw new StoreDashboardBadRequestException(StoreDashboardConstants.UNSUPPORTED_LEVEL);
            }
            return responseDTO;
        } catch (StoreDashboardBadRequestException e) {
            throw e;
        } catch (Exception ex) {
            throw new StoreDashboardException(StringUtils.isBlank(ex.getMessage()) ? StoreDashboardConstants.SOME_EXCEPTION_OCCURRED : ex.getMessage());
        }
    }

    @Transactional
    public StoreDetailResponseDTO insertHardwareDetails(StoreDetailRequestDTO requestDTO) {
        try {
            storeDashboardUtility.validateStoreDetailRequestDTO(requestDTO);
            List<Restaurant> restaurants = storeDashboardUtility.getRestaurantByMarketNameAndRestaurantNumber(requestDTO.getMarketName(), requestDTO.getRestaurantNo());
            Restaurant restaurant = restaurants.get(0);
            List<HardwareDetail> hardwareDetails = storeDashboardUtility.getRestaurantIdAndNodeId(restaurant.getId(), requestDTO.getHardwareDetails());
            List<HardwareDetail> details = mapperUtils.map(requestDTO.getHardwareDetails(), HardwareDetail.class);
            return storeDashboardUtility.insertHardwareDetails(details, restaurant, hardwareDetails);

        } catch (StoreDashboardBadRequestException e) {
            throw e;
        } catch (Exception ex) {
            throw new StoreDashboardException(StringUtils.isBlank(ex.getMessage()) ? StoreDashboardConstants.SOME_EXCEPTION_OCCURRED : ex.getMessage());
        }
    }

    /*This method fetches the hardware details for the restaurants.
     * If restaurantNo is passed, then it fetches only for that restaurant else for all.
     */
    @Transactional
    public StoreHardwareDetailResponseDTO getHardwareDetails(String restaurantNo) {
        try {
            List<HardwareDetail> hardwareDetails;
            if (StringUtils.isBlank(restaurantNo))
                hardwareDetails = hardwareDetailRepository.findAll(new QuerySpec(HardwareDetail.class));
            else
                hardwareDetails = hardwareDetailRepository.findByRestaurantNumber(restaurantNo);
            List<IndividualHardwareResponseDTO> storeHardwareDetails = hardwareDetails
                    .stream()
                    .map(i -> {
                        IndividualHardwareResponseDTO responseDTO = new IndividualHardwareResponseDTO();
                        responseDTO.setRestaurantNumber(i.getRestaurant().getName());
                        responseDTO.setMacAddress(i.getMacAddress());
                        responseDTO.setModelNumber(i.getModelNumber());
                        responseDTO.setSerialNumber(i.getSerialNumber());
                        responseDTO.setManufacturer(i.getManufacturer());
                        responseDTO.setModelName(i.getModelName());
                        responseDTO.setVersion(i.getVersion());
                        responseDTO.setFamily(i.getFamily());
                        responseDTO.setOsType(i.getOsType());
                        responseDTO.setOsVersion(i.getOsVersion());
                        responseDTO.setId(i.getId());
                        return responseDTO;
                    }).collect(Collectors.toList());
            return StoreHardwareDetailResponseDTO
                    .builder()
                    .hardwareList(storeHardwareDetails)
                    .build();
        } catch (Exception ex) {
            throw new StoreDashboardException(StringUtils.isBlank(ex.getMessage()) ? StoreDashboardConstants.SOME_EXCEPTION_OCCURRED : ex.getMessage());
        }
    }

    @Transactional
    public DeviceDetailResponseDTO getInstalledHardwareDetails(String dataType, Long offset, Long limit) {
        try {
            List<HardwareDeviceDetailResponseDTO> deviceDetailResponseDTOS = new ArrayList<>();
            List<Component> components = componentRepository.getAllComponentInstalledHardwareDetails(dataType, offset, limit);
            components.forEach(component -> {
                HardwareDeviceDetailResponseDTO deviceDetails = mapperUtils.map(component, HardwareDeviceDetailResponseDTO.class);
                deviceDetails.setStoreNumber(component.getRestaurant().getName());
                deviceDetailResponseDTOS.add(deviceDetails);
            });
            return DeviceDetailResponseDTO.builder().responseDTOS(deviceDetailResponseDTOS).build();
        } catch (Exception ex) {
            throw new StoreDashboardException(StringUtils.isBlank(ex.getMessage()) ? StoreDashboardConstants.SOME_EXCEPTION_OCCURRED : ex.getMessage());
        }
    }

    /*This function adds the incoming data for restaurants from USIT into StoreDemographic
     *and BreHierarchyNode table
     */
    @Transactional
    public USITPatchResponseDTO insertUSITData(USITPatchRequestDTO requestDTO) {
        USITPatchResponseDTO responseDTO = USITPatchResponseDTO.builder().isPatchSuccessful(Boolean.TRUE).build();
        try {
            List<BreHierarchyNode> marketNodes = breHierarchyNodeRepository.findAll(breHierarchyNodeRepository.prepareQuerySpecFilterForBreHierarchyNodeMarketName(HierarchyLevel.MARKET, requestDTO.getMarketName()));
            if (marketNodes.isEmpty()) {
                throw new StoreDashboardBadRequestException(StoreDashboardConstants.NO_NODE_EXIST);
            }
            if (marketNodes.size() > 1) {
                throw new StoreDashboardBadRequestException(StoreDashboardConstants.DUPLICATE_NODE_EXIST + requestDTO.getMarketName());
            }

            List<StoreDemographic> storeDemographicList = storeDemographicRepository.findAll(storeDemographicRepository.prepareQuerySpecFilterForStoreDemographicHierarchyNodeIdAndRestaurantNumber(requestDTO.getStoreTrackerRequestDTOList().keySet(), marketNodes.get(0).getId()));
            List<String> restaurantNoPresent = storeDemographicList.stream().map(i -> i.getHierarchyNode().getRestaurantNo()).collect(Collectors.toList());
            List<String> restaurantMissing = requestDTO.getStoreTrackerRequestDTOList().keySet().stream().filter(i -> !restaurantNoPresent.contains(i)).collect(Collectors.toList());
            responseDTO.setMissingRestaurantNumber(restaurantMissing);
            SimpleDateFormat formatter = new SimpleDateFormat("MM/dd/yyyy");
            formatter.setTimeZone(TimeZone.getTimeZone("GMT"));
            List<StoreDemographic> updateRecords = storeDemographicList.stream().map(i -> {
                StoreTrackerRequestDTO dto = requestDTO.getStoreTrackerRequestDTOList().get(i.getHierarchyNode().getRestaurantNo());
                try {
                    i.setPrepVisitDate(!StringUtils.isBlank(dto.getPrepVisitDate()) ? new Timestamp(formatter.parse(dto.getPrepVisitDate()).getTime()) : null);
                } catch (ParseException e) {
                    log.error(String.format("************Unable to parse PrepVistDate for restaurant %s************", i.getHierarchyNode().getRestaurantNo()));
                    i.setPrepVisitDate(null);
                }
                i.setOtpName(dto.getOtpName());
                i.setOtpEmail(dto.getOtpEmail());
                i.setOtpPhone(dto.getOtpPhone());
                i.setHardwareInstallationDate(dto.getHardwareInstallationDate());
                i.setSmokeTestDate(dto.getSmokeTestDate());
                i.setSsxVersion(dto.getSsxVersion());
                i.setDriveThruType(dto.getDriveThruType());
                i.getHierarchyNode().setIpAddress(dto.getIpAddress());
                return i;
            }).collect(Collectors.toList());
            responseDTO.setStoreDemographicList(updateRecords);
            return responseDTO;
        } catch (StoreDashboardBadRequestException e) {
            throw e;
        } catch (Exception ex) {
            responseDTO.setIsPatchSuccessful(Boolean.FALSE);
            return responseDTO;
        }

    }

    @Transactional
    public USITRestaurantMetadataResponseDTO insertRestaurantMetadata(USITRestaurantMetadataRequestDTO requestDTO) {
        try {
            USITRestaurantMetadataResponseDTO responseDTO = USITRestaurantMetadataResponseDTO.builder().isDataUpdated(Boolean.TRUE).build();
            if (CollectionUtils.isEmpty(requestDTO.getStoreInfo()) || StringUtils.isBlank(requestDTO.getMarketName()) || !requestDTO.getStoreInfo().stream().filter(i -> StringUtils.isBlank(i.getRestaurantNumber())).collect(Collectors.toList()).isEmpty()) {
                throw new StoreDashboardBadRequestException(StoreDashboardConstants.INVALID_REQUEST);
            }
            List<Market> marketList = marketRepository.findAll(marketRepository.prepareQuerySpecFilterForFilteringMarketByName(requestDTO.getMarketName()));
            if (marketList.isEmpty()) {
                throw new StoreDashboardBadRequestException(StoreDashboardConstants.MARKET_NOT_FOUND);
            }
            Map<String, RestaurantMetadataRequestDTO> restaurantNumberMap = requestDTO.getStoreInfo().stream().collect(Collectors.toMap(RestaurantMetadataRequestDTO::getRestaurantNumber, i -> i));
            List<Restaurant> restaurants = restaurantRepository.findAll(restaurantRepository.prepareQuerySpecFilterForRestaurantByRestaurantNumbersAndMarketId(new ArrayList<>(restaurantNumberMap.keySet()), marketList.get(0).getId()));
            List<String> storePresent = new ArrayList<>();
            restaurants.forEach(restaurant -> {
                RestaurantMetadataRequestDTO metadata = restaurantNumberMap.get(restaurant.getName());
                restaurant.setStoreStatus(metadata.getStoreStatus());
                restaurant.setGoLiveDate(metadata.getGoLiveDate());
                restaurant.setUpdated(Timestamp.valueOf(LocalDateTime.now()));
                storePresent.add(restaurant.getName());
            });
            if (storePresent.size() != restaurantNumberMap.keySet().size()) {
                responseDTO.setIsDataUpdated(Boolean.FALSE);

                responseDTO.setStoreNotFound(restaurantNumberMap.keySet().stream().filter(i -> !storePresent.contains(i)).collect(Collectors.toList()));
            }
            storeDashboardUtility.saveBulkRestaurants(restaurants);
            return responseDTO;
        } catch (StoreDashboardBadRequestException e) {
            throw e;
        } catch (Exception ex) {
            throw ex;
        }
    }

    @Transactional
    public StoreOnboardResponseDTO getRestaurantsReadyForOnboarding(Long fromDate, Long toDate) {
        try {
            List<StoreDemographic> restaurants = storeDemographicRepository.getRestaurantsForOnboarding(fromDate, toDate);
            List<String> restarantList = restaurants.parallelStream().map(i -> i.getHierarchyNode().getRestaurantNo()).collect(Collectors.toList());
            return StoreOnboardResponseDTO.builder().restaurantNumbers(restarantList).build();
        } catch (Exception ex) {
            throw new StoreDashboardException(StringUtils.isBlank(ex.getMessage()) ? StoreDashboardConstants.SOME_EXCEPTION_OCCURRED : ex.getMessage());
        }
    }
}