package com.mcd.restaurant.dashboard.controller;

import com.mcd.restaurant.common.ResponseView;
import com.mcd.restaurant.dashboard.controller.view.request.StoreDetailRequestDTO;
import com.mcd.restaurant.dashboard.controller.view.request.StoreStatusChangeRequestDTO;
import com.mcd.restaurant.dashboard.controller.view.request.USITPatchRequestDTO;
import com.mcd.restaurant.dashboard.controller.view.request.USITRestaurantMetadataRequestDTO;
import com.mcd.restaurant.dashboard.controller.view.response.*;
import com.mcd.restaurant.dashboard.enums.HierarchyLevel;
import com.mcd.restaurant.dashboard.service.StoreDashboardService;
import com.mcd.restaurant.dashboard.utils.StoreDashboardUtility;
import com.mcd.restaurant.dashboard.utils.csvhandler.CSVHierarchyHandler;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;


@RestController
@RequestMapping(path = "/store-dashboard")
public class StoreDashboardController {

    @Autowired
    private StoreDashboardService storeDashboardService;
    @Autowired
    private CSVHierarchyHandler csvHierarchyHandler;
    @Autowired
    private StoreDashboardUtility storeDashboardUtility;

    /**
     * This API will consolidate the various store detail information and sends as response
     */
    @GetMapping(path = "/details/{id}")
    public ResponseEntity<ResponseView<DashboardDetailResponseDTO>> getStoreDetails(@PathVariable(name = "id") Integer storeId) {
        ResponseView<DashboardDetailResponseDTO> responseView = new ResponseView<>();
        responseView.setData(storeDashboardService.getStoreDetails(storeId));
        return ResponseEntity.status(HttpStatus.OK).body(responseView);
    }

    /**
     * This API will upload/update the data in bre_hierarchy_node table
     */
    @PostMapping(path = "/hierarchydata/on-board")
    public ResponseEntity<ResponseView<BREHierarchyNodeDataUploadResponseDTO>> onBoardDataToBREHierarchyNode(@RequestParam(name = "hierarchyFile") MultipartFile hierarchyFile) {
        ResponseView<BREHierarchyNodeDataUploadResponseDTO> responseView = new ResponseView<>();
        responseView.setData(storeDashboardService.uploadDataToDataBase(hierarchyFile));
        return ResponseEntity.status(HttpStatus.CREATED).body(responseView);
    }

    /**
     * This API will upload/update the data in store_demographic table
     */
    @PostMapping(path = "/demographic-data/on-board")
    public ResponseEntity<ResponseView<BREStoreDemographicUploadResponseDTO>> onBoardDemographicDataToBREHierarchyNode(@RequestParam(name = "nodeFile") MultipartFile hierarchyFile) {
        ResponseView<BREStoreDemographicUploadResponseDTO> responseView = new ResponseView<>();
        responseView.setData(storeDashboardService.uploadDemographicDataToDataBase(hierarchyFile));
        return ResponseEntity.status(HttpStatus.CREATED).body(responseView);
    }

    /**
     * This API will map restaurants with its corresponding bre_hierarchy_node data
     */
    @PatchMapping(path = "/restaurant-data/mapping")
    public ResponseEntity<ResponseView<BREStoreToNodeMappingResponeDTO>> mapRestaurantwithHierarchyNode(@RequestParam(value = "actualMarketName") String marketName, @RequestParam(value = "dbMarketId") Integer dbMarketId) {
        ResponseView<BREStoreToNodeMappingResponeDTO> responseView = new ResponseView<>();
        responseView.setData(storeDashboardService.updateRestaurantWithHierarchyNodeId(marketName, dbMarketId));
        return ResponseEntity.status(HttpStatus.OK).body(responseView);
    }

    /**
     * This API will insert/update the ipaddress for stores present in BreHierarchyNode table
     */
    @PatchMapping(path = "/hierarchydata/ipaddress")
    public ResponseEntity<ResponseView<BREStoreIPAddressResponseDTO>> addIpaddressToStore(@RequestParam(value = "actualMarketName") String marketName, @RequestParam(name = "networkFile") MultipartFile networkFile) {
        ResponseView<BREStoreIPAddressResponseDTO> responseView = new ResponseView<>();
        responseView.setData(storeDashboardService.updateStoreHierarchyNodeWithIpAddress(marketName, networkFile));
        return ResponseEntity.status(HttpStatus.OK).body(responseView);
    }

    /**
     * This API will fetch the list of Market, Region, Coop and store list for hierarchy filtering dropdown feature
     */
    @GetMapping(path = "/hierarchy-filter")
    public ResponseEntity<ResponseView<HierarchySearchResponseDTO>> getHierarchyFilterData() {
        ResponseView<HierarchySearchResponseDTO> responseView = new ResponseView<>();
        responseView.setData(storeDashboardService.fetchHierarchyFilterData());
        return ResponseEntity.status(HttpStatus.OK).body(responseView);
    }

    /**
     * This API will updates the AOT Status for the given restaurant number
     */
    @PatchMapping(path = "/status/update")
    public ResponseEntity<ResponseView<StoreStatusChangeResponseDTO>> updateAOTStatus(@RequestBody StoreStatusChangeRequestDTO requestDTO) {
        ResponseView<StoreStatusChangeResponseDTO> responseView = new ResponseView<>();
        responseView.setData(storeDashboardService.updateAOTStatusForStore(requestDTO));
        return ResponseEntity.status(HttpStatus.OK).body(responseView);
    }

    @GetMapping(path = "/demographic/data")
    public ResponseEntity<ResponseView<StoreRegionResponseDTO>> fetchDemographicDetails(@RequestParam(name = "hierarchyLevel") HierarchyLevel hierarchyLevel, @RequestParam("marketName") String marketName, @RequestParam(value = "regionName", required = false) String regionName) {
        ResponseView<StoreRegionResponseDTO> responseView = new ResponseView<>();
        responseView.setData(storeDashboardService.getDemograhicDetails(hierarchyLevel, marketName, regionName));
        return ResponseEntity.status(HttpStatus.OK).body(responseView);
    }

    /**
     * This API will add hardware details for a restaurant
     */
    @PostMapping(path = "/hardwaredetail")
    public ResponseEntity<ResponseView<StoreDetailResponseDTO>> addHarwareDetailForStore(@RequestBody StoreDetailRequestDTO requestDTO) {
        ResponseView<StoreDetailResponseDTO> responseView = new ResponseView<>();
        responseView.setData(storeDashboardService.insertHardwareDetails(requestDTO));
        return ResponseEntity.status(HttpStatus.CREATED).body(responseView);
    }

    @GetMapping(path = "/hardwaredetail")
    public ResponseEntity<ResponseView<StoreHardwareDetailResponseDTO>> fetchHardwareDetails(@RequestParam(value = "restaurant_no", required = false) String restaurantNo) {
        ResponseView<StoreHardwareDetailResponseDTO> responseView = new ResponseView<>();
        responseView.setData(storeDashboardService.getHardwareDetails(restaurantNo));
        return ResponseEntity.status(HttpStatus.OK).body(responseView);
    }

    @GetMapping(path = "/hardwaredata")
    public ResponseEntity<ResponseView<DeviceDetailResponseDTO>> fetchInstalledHardwareDetails(@RequestParam(name = "dataPrefix", required = false, defaultValue = "AXIS") String dataPrefix, @RequestParam(name = "offset", defaultValue = "0") Long offset, @RequestParam(name = "limit", defaultValue = "-1") Long limit) {
        ResponseView<DeviceDetailResponseDTO> responseView = new ResponseView<>();
        responseView.setData(storeDashboardService.getInstalledHardwareDetails(dataPrefix, offset, limit));
        return ResponseEntity.status(HttpStatus.OK).body(responseView);
    }

    @PatchMapping(path = "/usit")
    public ResponseEntity<ResponseView<USITPatchResponseDTO>> updateUSITTrackerData(@RequestBody USITPatchRequestDTO requestDTO) {
        ResponseView<USITPatchResponseDTO> responseView = new ResponseView<>();
        USITPatchResponseDTO responseDTO = storeDashboardService.insertUSITData(requestDTO);
        storeDashboardUtility.saveChangesForStoreDemographicUSIT(responseDTO.getStoreDemographicList());
        responseView.setData(responseDTO);
        return ResponseEntity.status(HttpStatus.OK).body(responseView);
    }


    @PatchMapping(path = "/restaurant/metadata")
    public ResponseEntity<ResponseView<USITRestaurantMetadataResponseDTO>> updateRestaurantMetaData(@RequestBody USITRestaurantMetadataRequestDTO requestDTO) {
        ResponseView<USITRestaurantMetadataResponseDTO> responseView = new ResponseView<>();
        responseView.setData(storeDashboardService.insertRestaurantMetadata(requestDTO));
        return ResponseEntity.status(HttpStatus.OK).body(responseView);
    }

    /*This API will accept the from and to date(PrepVisitDate) in request and will return back the
     * list of restarants numbers that can be onboarded as per the requested dates*/
    @GetMapping(path = "/restaurant/onboard")
    public ResponseEntity<ResponseView<StoreOnboardResponseDTO>> fetchRestaurantsForOnboarding(@RequestParam(name = "fromDate")Long fromDate,@RequestParam(name = "toDate")Long toDate) {
        ResponseView<StoreOnboardResponseDTO> responseView = new ResponseView<>();
        responseView.setData(storeDashboardService.getRestaurantsReadyForOnboarding(fromDate,toDate));
        return ResponseEntity.status(HttpStatus.OK).body(responseView);
    }

}
