package com.mcd.restaurant.dashboard;

import com.mcd.restaurant.dashboard.error.StoreDashboardBadRequestException;
import com.mcd.restaurant.dashboard.controller.view.request.StoreStatusChangeRequestDTO;
import com.mcd.restaurant.dashboard.controller.view.response.*;
import com.mcd.restaurant.dashboard.enums.HierarchyLevel;
import com.mcd.restaurant.dashboard.error.StoreDashboardException;
import com.mcd.restaurant.dashboard.service.StoreDashboardService;
import com.mcd.restaurant.dashboard.testUtils.StoreDashboardTestUtils;
import com.mcd.restaurant.dashboard.utils.csvhandler.CSVHierarchyHandler;
import com.mcd.restaurant.dashboard.utils.StoreDashboardUtility;
import com.mcd.restaurant.model.*;
import com.mcd.restaurant.repository.*;
import com.mcd.restaurant.repository.bulk.BREHierarchyNodeCRUDRepository;
import com.mcd.restaurant.repository.bulk.RestaurantBulkRepository;
import com.mcd.restaurant.repository.bulk.StoreDemographicCRUDRepository;
import io.crnk.core.exception.ResourceNotFoundException;
import io.crnk.core.queryspec.QuerySpec;
import io.crnk.core.resource.list.DefaultResourceList;
import io.crnk.core.resource.list.ResourceList;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.CsvSource;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;
import org.springframework.mock.web.MockMultipartFile;
import org.springframework.test.util.ReflectionTestUtils;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;

import static org.mockito.Mockito.when;

class StoreDashboardServiceTest {


    @BeforeEach
    void setUp() {
        MockitoAnnotations.initMocks(this);
    }

    @InjectMocks
    private StoreDashboardService storeDashboardService;
    @Mock
    private RestaurantViewRepository restaurantViewRepositoryMock;
    @Mock
    private StoreDashboardUtility storeDashboardUtilityMock;
    @InjectMocks
    private StoreDashboardUtility storeDashboardUtility;
    @Mock
    private HardwareDetailRepository hardwareDetailRepositoryMock;
    @InjectMocks
    private StoreDashboardTestUtils storeDashboardUtilityTest;
    @InjectMocks
    private BreHierarchyNodeRepository breHierarchyNodeRepository;
    @Mock
    private BreHierarchyNodeRepository breHierarchyNodeRepositoryMock;
    @Mock
    private CSVHierarchyHandler csvHierarchyHandlerMock;
    @InjectMocks
    private CSVHierarchyHandler csvHierarchyHandler;
    @Mock
    private ComponentRepository componentRepository;
    @Mock
    private StoreDataJobRepository storeDataJobRepositoryMock;
    @InjectMocks
    private StoreDataJobRepository storeDataJobRepository;
    @Mock
    private BREHierarchyNodeCRUDRepository breHierarchyNodeCRUDRepositoryMock;
    @Mock
    private StoreDemographicRepository storeDemographicRepositoryMock;
    @Mock
    private StoreDemographicCRUDRepository storeDemographicCRUDRepository;
    @Mock
    private RestaurantBulkRepository restaurantBulkRepositoryMock;
    @Mock
    private RestaurantRepository restaurantRepositoryMock;
    @Mock
    private MarketRepository marketRepository;


    @Test
    void testGetStoreDetailsSuccess() {
        Integer storeId = 1;
        RestaurantView view = new RestaurantView();
        view.setId(1);
        ResourceList<HardwareDetail> hardwareDetails = new DefaultResourceList<>();
        ReflectionTestUtils.setField(storeDashboardService, "storeDashboardUtility", storeDashboardUtility);
        Mockito.when(restaurantViewRepositoryMock.findOne(storeId, new QuerySpec(RestaurantView.class))).thenReturn(view);
        Mockito.when(hardwareDetailRepositoryMock.prepareQuerySpecFilterForHardwareDetail(storeId)).thenCallRealMethod();
        Mockito.when(hardwareDetailRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(hardwareDetails);
        Mockito.when(storeDashboardUtilityMock.buildDashboardDetailResponseDTO(Mockito.any(RestaurantView.class), Mockito.anyInt(), Mockito.anyInt(), Mockito.anyInt())).thenCallRealMethod();
        when(componentRepository.findAll(Mockito.any(QuerySpec.class))).thenReturn(new DefaultResourceList<>());
        when(componentRepository.prepareQuerySpecFilterForHardware(Mockito.anyInt())).thenCallRealMethod();
        when(componentRepository.prepareQuerySpecFilterForPlatformCount(Mockito.anyInt())).thenCallRealMethod();
        DashboardDetailResponseDTO dto = storeDashboardService.getStoreDetails(1);
        Assertions.assertNotNull(dto);
    }

    @Test
    void testGetStoreDetailsFailure() {
        Integer storeId = 1;
        RestaurantView view = new RestaurantView();
        view.setId(1);
        ResourceList<HardwareDetail> hardwareDetails = new DefaultResourceList<>();
        ReflectionTestUtils.setField(storeDashboardService, "storeDashboardUtility", storeDashboardUtility);
        Mockito.when(restaurantViewRepositoryMock.findOne(storeId, new QuerySpec(RestaurantView.class))).thenThrow(ResourceNotFoundException.class);
        Assertions.assertThrows(StoreDashboardException.class, () -> storeDashboardService.getStoreDetails(1));
    }

    @Test
    void testUpdateStoreHierarchyNodeWithIpAddress() throws IOException {
        ReflectionTestUtils.setField(storeDashboardService, "storeDashboardUtility", storeDashboardUtility);
        ReflectionTestUtils.setField(storeDashboardService, "csvHierarchyHandler", csvHierarchyHandler);
        MultipartFile file = storeDashboardUtilityTest.getMultipartValidFileForIpAddress();
        BreHierarchyNode breHierarchyNode = new BreHierarchyNode();
        breHierarchyNode.setHierarchyLevel(HierarchyLevel.MARKET.value());
        breHierarchyNode.setNodeName("US Country Office");
        breHierarchyNode.setId(1);
        ResourceList<BreHierarchyNode> list = new DefaultResourceList<>();
        list.add(breHierarchyNode);
        Mockito.when(breHierarchyNodeRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(list);
        Mockito.when(storeDataJobRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(new DefaultResourceList<>());
        Mockito.when(storeDataJobRepositoryMock.create(Mockito.any(StoreDataJob.class))).thenAnswer(i -> i.getArguments()[0]);
        Mockito.when(csvHierarchyHandlerMock.convertNetworkInfoCSVToBean(file)).thenCallRealMethod();
        Mockito.when(breHierarchyNodeRepositoryMock.prepareQuerySpecFilterForBreHierarchyNodeMarketName(HierarchyLevel.MARKET, "US Country Office")).thenCallRealMethod();
        Mockito.when(breHierarchyNodeRepositoryMock.prepareQuerySpecFilterForBreHierarchyNodeMarketId(Mockito.any(HierarchyLevel.class), Mockito.anyInt())).thenCallRealMethod();
        Mockito.when(storeDataJobRepositoryMock.saveJobWithNativeQuery(Mockito.any(StoreDataJob.class))).thenReturn(1);
        BREStoreIPAddressResponseDTO breStoreToNodeMappingResponeDTO = storeDashboardService.updateStoreHierarchyNodeWithIpAddress("US Country Office", file);
        Assertions.assertNotNull(breStoreToNodeMappingResponeDTO);
    }

    @Test
    void testUpdateStoreHierarchyNodeWithIpAddressfailure() throws IOException {
        ReflectionTestUtils.setField(storeDashboardService, "storeDashboardUtility", storeDashboardUtility);
        Mockito.when(storeDataJobRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(new DefaultResourceList<>());
        BreHierarchyNode breHierarchyNode1 = new BreHierarchyNode();
        MultipartFile file = storeDashboardUtilityTest.getMultipartValidFileForIpAddress();
        breHierarchyNode1.setHierarchyLevel(HierarchyLevel.MARKET.value());
        breHierarchyNode1.setNodeName("US Country Office");
        breHierarchyNode1.setId(1);
        BreHierarchyNode breHierarchyNode2 = new BreHierarchyNode();
        breHierarchyNode2.setHierarchyLevel(HierarchyLevel.MARKET.value());
        breHierarchyNode2.setNodeName("US Country Office");
        breHierarchyNode2.setId(2);
        ResourceList<BreHierarchyNode> list = new DefaultResourceList<>();
        list.add(breHierarchyNode1);
        list.add(breHierarchyNode2);
        Mockito.when(breHierarchyNodeRepositoryMock.prepareQuerySpecFilterForBreHierarchyNodeMarketName(HierarchyLevel.MARKET, "US Country Office")).thenCallRealMethod();
        Mockito.when(breHierarchyNodeRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(list);
        Mockito.when(breHierarchyNodeRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(list);
        Assertions.assertThrows(StoreDashboardException.class, () -> storeDashboardService.updateStoreHierarchyNodeWithIpAddress("US Country Office", file));
    }

    @Test
    void testUpdateStoreHierarchyNodeWithIpAddressfailureWithNoMarket() throws IOException {

        ReflectionTestUtils.setField(storeDashboardService, "storeDashboardUtility", storeDashboardUtility);
        Mockito.when(storeDataJobRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(new DefaultResourceList<>());
        MultipartFile file = storeDashboardUtilityTest.getMultipartValidFileForIpAddress();
        ResourceList<BreHierarchyNode> list = new DefaultResourceList<>();
        Mockito.when(breHierarchyNodeRepositoryMock.prepareQuerySpecFilterForBreHierarchyNodeMarketName(HierarchyLevel.MARKET, "US Country Office")).thenCallRealMethod();
        Mockito.when(breHierarchyNodeRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(list);
        Mockito.when(breHierarchyNodeRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(list);
        Assertions.assertThrows(StoreDashboardException.class, () -> storeDashboardService.updateStoreHierarchyNodeWithIpAddress("US Country Office", file));
    }


    @Test
    void testUploadDataToDataBaseOnGoingOperationFailure() throws IOException {
        ReflectionTestUtils.setField(storeDashboardService, "storeDashboardUtility", storeDashboardUtility);
        MultipartFile file = storeDashboardUtilityTest.getMultipartValidFileForHierarchyStructure();
        ResourceList<StoreDataJob> list = new DefaultResourceList<>();
        list.add(StoreDataJob.builder().build());
        Mockito.when(storeDataJobRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(list);
        Assertions.assertThrows(StoreDashboardException.class, () -> storeDashboardService.uploadDataToDataBase(file));
    }

    @Test
    void testUploadDataToDataBaseInvalidFileFailure() throws IOException {
        ReflectionTestUtils.setField(storeDashboardService, "storeDashboardUtility", storeDashboardUtility);
        MultipartFile file = new MockMultipartFile("hierarchy-structure.pdf", new byte[5]);
        Mockito.when(storeDataJobRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(new DefaultResourceList<>());
        Assertions.assertThrows(StoreDashboardException.class, () -> storeDashboardService.uploadDataToDataBase(file));
    }

    @Test
    void testUploadDataToDataBaseSuccess() throws IOException {
        ReflectionTestUtils.setField(storeDashboardService, "storeDashboardUtility", storeDashboardUtility);
        ReflectionTestUtils.setField(storeDashboardUtility, "csvHierarchyHandler", csvHierarchyHandler);
        MultipartFile file = storeDashboardUtilityTest.getMultipartValidFileForHierarchyStructure();
        Mockito.when(storeDataJobRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(new DefaultResourceList<>());
        Mockito.when(breHierarchyNodeRepositoryMock.prepareQuerySpecFilterForBreHierarchyNode(Mockito.any(HierarchyLevel.class))).thenCallRealMethod();
        Mockito.when(breHierarchyNodeRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(new DefaultResourceList<>());
        Mockito.when(breHierarchyNodeRepositoryMock.save(Mockito.any(BreHierarchyNode.class))).thenAnswer(i -> i.getArguments()[0]);
        Mockito.when(storeDataJobRepositoryMock.create(Mockito.any(StoreDataJob.class))).thenAnswer(i -> i.getArguments()[0]);
        Mockito.when(breHierarchyNodeCRUDRepositoryMock.saveAll(Mockito.anyList())).thenAnswer(i -> i.getArguments()[0]);
        Mockito.when(storeDataJobRepositoryMock.saveJobWithNativeQuery(Mockito.any(StoreDataJob.class))).thenReturn(1);
        BREHierarchyNodeDataUploadResponseDTO hierarchyNodeDataUploadResponseDTO = storeDashboardService.uploadDataToDataBase(file);
        Assertions.assertNotNull(hierarchyNodeDataUploadResponseDTO);
        Assertions.assertEquals(1, hierarchyNodeDataUploadResponseDTO.getMarketInserted());
        Assertions.assertEquals(1, hierarchyNodeDataUploadResponseDTO.getRegionInserted());
        Assertions.assertEquals(1, hierarchyNodeDataUploadResponseDTO.getCoopInserted());
        Assertions.assertEquals(1, hierarchyNodeDataUploadResponseDTO.getStoreInserted());
        Assertions.assertEquals(0, hierarchyNodeDataUploadResponseDTO.getStoreUpdated());

    }

    @Test
    void testUploadDemographicDataToDataBaseOnGoingOperationFailure() throws IOException {
        ReflectionTestUtils.setField(storeDashboardService, "storeDashboardUtility", storeDashboardUtility);
        MultipartFile file = storeDashboardUtilityTest.getMultipartValidFileForDemoGraphicStructure();
        ResourceList<StoreDataJob> list = new DefaultResourceList<>();
        list.add(StoreDataJob.builder().build());
        Mockito.when(storeDataJobRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(list);
        Assertions.assertThrows(StoreDashboardException.class, () -> storeDashboardService.uploadDemographicDataToDataBase(file));
    }

    @Test
    void testUploadDemographicDataToDataBaseInvalidFileFailure() throws IOException {
        ReflectionTestUtils.setField(storeDashboardService, "storeDashboardUtility", storeDashboardUtility);
        MultipartFile file = new MockMultipartFile("demographic-structure.pdf", new byte[5]);
        Mockito.when(storeDataJobRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(new DefaultResourceList<>());
        Assertions.assertThrows(StoreDashboardException.class, () -> storeDashboardService.uploadDemographicDataToDataBase(file));
    }

    @Test
    void testUploadDemographicDataToDataBaseNoMarketPresent() throws IOException {
        ReflectionTestUtils.setField(storeDashboardService, "storeDashboardUtility", storeDashboardUtility);
        MultipartFile file = storeDashboardUtilityTest.getInvalidMultipartValidFileForDemoGraphicStructure();
        Mockito.when(storeDataJobRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(new DefaultResourceList<>());
        ReflectionTestUtils.setField(storeDashboardUtility, "csvHierarchyHandler", csvHierarchyHandler);
        Assertions.assertThrows(StoreDashboardException.class, () -> storeDashboardService.uploadDemographicDataToDataBase(file));
    }


    @Test
    void testUploadDemographicDataToDataBaseSuccess() throws IOException {
        ReflectionTestUtils.setField(storeDashboardService, "storeDashboardUtility", storeDashboardUtility);
        ReflectionTestUtils.setField(storeDashboardService, "csvHierarchyHandler", csvHierarchyHandler);
        MultipartFile file = storeDashboardUtilityTest.getMultipartValidFileForDemoGraphicStructure();
        ResourceList<BreHierarchyNode> list = new DefaultResourceList<>();
        list.add(storeDashboardUtilityTest.getMarketNode());
        ResourceList<BreHierarchyNode> storeList = new DefaultResourceList<>();
        storeList.add(storeDashboardUtilityTest.getStoreNode());
        ResourceList<StoreDemographic> storeDemographics = new DefaultResourceList<>();
        storeDemographics.add(storeDashboardUtilityTest.getDemographicNode());
        QuerySpec marketQuery = breHierarchyNodeRepository.prepareQuerySpecFilterForBreHierarchyNodeMarketName(HierarchyLevel.MARKET, "US Country Office");
        QuerySpec existingStoreQuery = breHierarchyNodeRepository.prepareQuerySpecFilterForBreHierarchyNodeMarketId(HierarchyLevel.RESTAURANT, 3);
        Mockito.when(breHierarchyNodeRepositoryMock.prepareQuerySpecFilterForBreHierarchyNodeMarketName(HierarchyLevel.MARKET, "US Country Office")).thenReturn(marketQuery);
        Mockito.when(breHierarchyNodeRepositoryMock.prepareQuerySpecFilterForBreHierarchyNodeMarketId(HierarchyLevel.RESTAURANT, 3)).thenReturn(existingStoreQuery);
        Mockito.when(breHierarchyNodeRepositoryMock.prepareQuerySpecFilterForBreHierarchyNodeMarketId(Mockito.any(HierarchyLevel.class), Mockito.anyInt())).thenCallRealMethod();
        Mockito.when(storeDemographicRepositoryMock.prepareQuerySpecFilterForStoreDemographicMarketId(Mockito.any(HierarchyLevel.class), Mockito.anyInt())).thenCallRealMethod();
        Mockito.when(storeDemographicRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(storeDemographics);
        Mockito.when(breHierarchyNodeRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(new DefaultResourceList<>());
        Mockito.when(breHierarchyNodeRepositoryMock.findAll(marketQuery)).thenReturn(list);
        Mockito.when(breHierarchyNodeRepositoryMock.findAll(existingStoreQuery)).thenReturn(storeList);
        Mockito.when(storeDataJobRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(new DefaultResourceList<>());
        Mockito.when(storeDataJobRepositoryMock.create(Mockito.any(StoreDataJob.class))).thenAnswer(i -> i.getArguments()[0]);
        Mockito.when(storeDemographicCRUDRepository.saveAll(Mockito.anyList())).thenAnswer(i -> i.getArguments()[0]);
        Mockito.when(storeDataJobRepositoryMock.saveJobWithNativeQuery(Mockito.any(StoreDataJob.class))).thenReturn(1);
        BREStoreDemographicUploadResponseDTO hierarchyNodeDataUploadResponseDTO = storeDashboardService.uploadDemographicDataToDataBase(file);
        Assertions.assertNotNull(hierarchyNodeDataUploadResponseDTO);
        Assertions.assertEquals(0, hierarchyNodeDataUploadResponseDTO.getNewStoreDemograhicsAdded());
        Assertions.assertEquals(1, hierarchyNodeDataUploadResponseDTO.getUpdatedStoreDemographics());
    }


    @Test
    void testFetchHierarchyFilterDataSuccess() throws IOException {
        ResourceList<BreHierarchyNode> list = storeDashboardUtilityTest.getBreHierarchyNodesList();
        ResourceList<Restaurant> restaurants = storeDashboardUtilityTest.getRestaurantList();
        Mockito.when(breHierarchyNodeRepositoryMock.prepareQuerySpecFilterForBreHierarchyNode(Mockito.any(HierarchyLevel.class))).thenCallRealMethod();
        Mockito.when(restaurantRepositoryMock.prepareQuerySpecFilterForRestaurantWithBreHierarchyNodePresent()).thenCallRealMethod();
        Mockito.when(breHierarchyNodeRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(list);
        Mockito.when(restaurantRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(restaurants);
        HierarchySearchResponseDTO responseDTO = storeDashboardService.fetchHierarchyFilterData();
        Assertions.assertNotNull(responseDTO);
        Assertions.assertEquals(1, responseDTO.getMarkets().size());
        Assertions.assertEquals(1, responseDTO.getCoops().size());
        Assertions.assertEquals(1, responseDTO.getRegions().size());
        Assertions.assertEquals(1, responseDTO.getStores().size());
    }

    @Test
    void testFetchHierarchyFilterDataFailure() throws IOException {
        Mockito.when(breHierarchyNodeRepositoryMock.prepareQuerySpecFilterForBreHierarchyNode(Mockito.any(HierarchyLevel.class))).thenCallRealMethod();
        Mockito.when(restaurantRepositoryMock.prepareQuerySpecFilterForRestaurantWithBreHierarchyNodePresent()).thenCallRealMethod();
        Mockito.when(breHierarchyNodeRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenThrow(ResourceNotFoundException.class);
        Assertions.assertThrows(StoreDashboardException.class, () -> storeDashboardService.fetchHierarchyFilterData());
    }


    @Test
    void updateRestaurantWithHierarchyNodeIdSuccess() throws IOException {
        ReflectionTestUtils.setField(storeDashboardService, "storeDashboardUtility", storeDashboardUtility);
        String marketName = "US Country Office";
        Integer marketId = 1;
        ResourceList<BreHierarchyNode> list = new DefaultResourceList<>();
        list.add(storeDashboardUtilityTest.getMarketNode());
        ResourceList<BreHierarchyNode> storeList = new DefaultResourceList<>();
        storeList.add(storeDashboardUtilityTest.getStoreNode());
        ResourceList<Restaurant> restaurants = storeDashboardUtilityTest.getRestaurantList();
        QuerySpec marketQuery = breHierarchyNodeRepository.prepareQuerySpecFilterForBreHierarchyNodeMarketName(HierarchyLevel.MARKET, marketName);
        QuerySpec existingStoreQuery = breHierarchyNodeRepository.prepareQuerySpecFilterForBreHierarchyNodeMarketId(HierarchyLevel.RESTAURANT, marketId);
        Mockito.when(breHierarchyNodeRepositoryMock.prepareQuerySpecFilterForBreHierarchyNodeMarketName(HierarchyLevel.MARKET, marketName)).thenReturn(marketQuery);
        Mockito.when(restaurantRepositoryMock.prepareQuerySpecFilterForRestaurantByMarketName(marketId)).thenCallRealMethod();
        Mockito.when(breHierarchyNodeRepositoryMock.prepareQuerySpecFilterForBreHierarchyNodeMarketId(HierarchyLevel.RESTAURANT, marketId)).thenReturn(existingStoreQuery);
        Mockito.when(breHierarchyNodeRepositoryMock.findAll(marketQuery)).thenReturn(list);
        Mockito.when(restaurantRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(restaurants);
        Mockito.when(breHierarchyNodeRepositoryMock.findAll(existingStoreQuery)).thenReturn(storeList);
        Mockito.when(storeDataJobRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(new DefaultResourceList<>());
        Mockito.when(storeDataJobRepositoryMock.create(Mockito.any(StoreDataJob.class))).thenAnswer(i -> i.getArguments()[0]);
        Mockito.when(restaurantBulkRepositoryMock.saveAll(Mockito.anyList())).thenAnswer(i -> i.getArguments()[0]);
        Mockito.when(storeDataJobRepositoryMock.saveJobWithNativeQuery(Mockito.any(StoreDataJob.class))).thenReturn(1);
        BREStoreToNodeMappingResponeDTO responeDTO = storeDashboardService.updateRestaurantWithHierarchyNodeId(marketName, marketId);
        Assertions.assertNotNull(responeDTO);
        Assertions.assertEquals(1, responeDTO.getTotalRestaurantMapped().size());

    }

    @Test
    void updateRestaurantWithHierarchyNodeIdFailure() throws IOException {
        ReflectionTestUtils.setField(storeDashboardService, "storeDashboardUtility", storeDashboardUtility);
        String marketName = "US Country Office";
        Integer marketId = 1;
        ResourceList<BreHierarchyNode> list = new DefaultResourceList<>();
        list.add(storeDashboardUtilityTest.getMarketNode());
        list.add(storeDashboardUtilityTest.getMarketNode());
        QuerySpec marketQuery = breHierarchyNodeRepository.prepareQuerySpecFilterForBreHierarchyNodeMarketName(HierarchyLevel.MARKET, marketName);
        Mockito.when(breHierarchyNodeRepositoryMock.prepareQuerySpecFilterForBreHierarchyNodeMarketName(HierarchyLevel.MARKET, marketName)).thenReturn(marketQuery);
        Mockito.when(breHierarchyNodeRepositoryMock.findAll(marketQuery)).thenReturn(list);
        Mockito.when(storeDataJobRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(new DefaultResourceList<>());
        Mockito.when(storeDataJobRepositoryMock.create(Mockito.any(StoreDataJob.class))).thenAnswer(i -> i.getArguments()[0]);
        Mockito.when(restaurantBulkRepositoryMock.saveAll(Mockito.anyList())).thenAnswer(i -> i.getArguments()[0]);
        Mockito.when(storeDataJobRepositoryMock.saveJobWithNativeQuery(Mockito.any(StoreDataJob.class))).thenReturn(1);
        Assertions.assertThrows(StoreDashboardException.class, () -> storeDashboardService.updateRestaurantWithHierarchyNodeId(marketName, marketId));

    }

    @Test
    void updateRestaurantWithHierarchyNodeIdNoMarketFailure() throws IOException {
        ReflectionTestUtils.setField(storeDashboardService, "storeDashboardUtility", storeDashboardUtility);
        String marketName = "US Country Office";
        Integer marketId = 1;
        ResourceList<BreHierarchyNode> list = new DefaultResourceList<>();
        QuerySpec marketQuery = breHierarchyNodeRepository.prepareQuerySpecFilterForBreHierarchyNodeMarketName(HierarchyLevel.MARKET, marketName);
        Mockito.when(breHierarchyNodeRepositoryMock.prepareQuerySpecFilterForBreHierarchyNodeMarketName(HierarchyLevel.MARKET, marketName)).thenReturn(marketQuery);
        Mockito.when(breHierarchyNodeRepositoryMock.findAll(marketQuery)).thenReturn(list);
        Mockito.when(storeDataJobRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(new DefaultResourceList<>());
        Mockito.when(storeDataJobRepositoryMock.create(Mockito.any(StoreDataJob.class))).thenAnswer(i -> i.getArguments()[0]);
        Mockito.when(restaurantBulkRepositoryMock.saveAll(Mockito.anyList())).thenAnswer(i -> i.getArguments()[0]);
        Mockito.when(storeDataJobRepositoryMock.saveJobWithNativeQuery(Mockito.any(StoreDataJob.class))).thenReturn(1);
        Assertions.assertThrows(StoreDashboardException.class, () -> storeDashboardService.updateRestaurantWithHierarchyNodeId(marketName, marketId));

    }

    @ParameterizedTest
    @CsvSource({",US,8001", "5,,8001", "5,US,","5,US,801"})
    void testUpdateAOTStatusForStoreWithMissingRequest(String restaurantNo, String marketName, Integer storeStatus) {

        StoreStatusChangeRequestDTO requestDTO = StoreStatusChangeRequestDTO.builder().marketName(marketName).restaurantNo(restaurantNo).storeStatus(storeStatus).build();
        Assertions.assertThrows(StoreDashboardBadRequestException.class, () -> storeDashboardService.updateAOTStatusForStore(requestDTO));

    }

    @Test
    void testUpdateAOTStatusForStoreWithInvalidMarket() {
        StoreStatusChangeRequestDTO requestDTO = StoreStatusChangeRequestDTO.builder().marketName("ABC").restaurantNo("12").storeStatus(8001).build();
        ReflectionTestUtils.setField(storeDashboardService, "storeDashboardUtility", storeDashboardUtility);
        Mockito.when(marketRepository.findAll(Mockito.any(QuerySpec.class))).thenReturn(new DefaultResourceList<>());
        Mockito.when(marketRepository.prepareQuerySpecFilterForFilteringMarketByName(Mockito.anyString())).thenCallRealMethod();
        Assertions.assertThrows(StoreDashboardBadRequestException.class, () -> storeDashboardService.updateAOTStatusForStore(requestDTO));
    }

    @Test
    void testUpdateAOTStatusForStoreWithInvalidRestaurantNumber() {
        StoreStatusChangeRequestDTO requestDTO = StoreStatusChangeRequestDTO.builder().marketName("US").restaurantNo("123").storeStatus(8001).build();
        ResourceList<Market> list = storeDashboardUtilityTest.getMarkets();
        ReflectionTestUtils.setField(storeDashboardService, "storeDashboardUtility", storeDashboardUtility);
        Mockito.when(marketRepository.findAll(Mockito.any(QuerySpec.class))).thenReturn(list);
        Mockito.when(marketRepository.prepareQuerySpecFilterForFilteringMarketByName(Mockito.anyString())).thenCallRealMethod();

        Mockito.when(restaurantRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(new DefaultResourceList<>());
        Mockito.when(restaurantRepositoryMock.prepareQuerySpecFilterForRestaurantByRestaurantNumberAndMarketId(Mockito.anyString(), Mockito.anyInt())).thenCallRealMethod();

        Assertions.assertThrows(StoreDashboardBadRequestException.class, () -> storeDashboardService.updateAOTStatusForStore(requestDTO));
    }



    @Test
    void testUpdateAOTStatusForStoreSuccess() {
        StoreStatusChangeRequestDTO requestDTO = StoreStatusChangeRequestDTO.builder().marketName("US").restaurantNo("123").storeStatus(8001).build();
        ResourceList<Market> list = storeDashboardUtilityTest.getMarkets();
        ReflectionTestUtils.setField(storeDashboardService, "storeDashboardUtility", storeDashboardUtility);
        Mockito.when(marketRepository.findAll(Mockito.any(QuerySpec.class))).thenReturn(list);
        Mockito.when(marketRepository.prepareQuerySpecFilterForFilteringMarketByName(Mockito.anyString())).thenCallRealMethod();

        Mockito.when(restaurantRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(storeDashboardUtilityTest.getRestaurantList());
        Mockito.when(restaurantRepositoryMock.prepareQuerySpecFilterForRestaurantByRestaurantNumberAndMarketId(Mockito.anyString(), Mockito.anyInt())).thenCallRealMethod();

        Mockito.when(restaurantRepositoryMock.save(Mockito.any(Restaurant.class))).thenAnswer(i -> i.getArguments()[0]);

        StoreStatusChangeResponseDTO responseDTO = storeDashboardService.updateAOTStatusForStore(requestDTO);
        Assertions.assertEquals(true, responseDTO.isStatusChangeSuccess());
    }

    @ParameterizedTest
    @CsvSource({"RESTAURANT,US,", "RESTAURANT,ABC,Reg", "RESTAURANT,ABCD,Reg", "MARKET,ABCDE,Reg"})
    void testGetDemograhicDetailsBadRequest(HierarchyLevel hierarchyLevel, String marketName, String region) {
        ResourceList<BreHierarchyNode> list;
        if (marketName.equals("ABC"))
            list = new DefaultResourceList<>();
        else if (marketName.equals("ABCD")) {
            list = storeDashboardUtilityTest.getBreHierarchyNodesList();
            list.addAll(storeDashboardUtilityTest.getBreHierarchyNodesList());
        } else {
            list = storeDashboardUtilityTest.getBreHierarchyNodesList();
        }
        ReflectionTestUtils.setField(storeDashboardService, "storeDashboardUtility", storeDashboardUtility);
        Mockito.when(breHierarchyNodeRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(list);
        Mockito.when(breHierarchyNodeRepositoryMock.prepareQuerySpecFilterForBreHierarchyNodeMarketName(HierarchyLevel.MARKET, marketName)).thenCallRealMethod();
        Assertions.assertThrows(StoreDashboardBadRequestException.class, () -> storeDashboardService.getDemograhicDetails(hierarchyLevel, marketName, region));
    }

    @ParameterizedTest
    @CsvSource({"RESTAURANT,ABC,Reg"})
    void testGetDemograhicDetailsException(HierarchyLevel hierarchyLevel, String marketName, String region) {
        ResourceList<BreHierarchyNode> list = new DefaultResourceList<>();
        Assertions.assertThrows(StoreDashboardException.class, () -> storeDashboardService.getDemograhicDetails(hierarchyLevel, marketName, region));
    }

    @ParameterizedTest
    @CsvSource({"RESTAURANT,US,ABC", "REGION,US,"})
    void testGetDemograhicDetailsSuccess(HierarchyLevel hierarchyLevel, String marketName, String region) {
        ResourceList<BreHierarchyNode> dataList = new DefaultResourceList<>();
        ResourceList<Restaurant> restaurantResourceList = storeDashboardUtilityTest.getRestaurantList();
        dataList.addAll(storeDashboardUtilityTest.getBreHierarchyNodesList());
        ResourceList<BreHierarchyNode> list = storeDashboardUtilityTest.getBreHierarchyNodesList();
        ReflectionTestUtils.setField(storeDashboardService, "storeDashboardUtility", storeDashboardUtility);
        Mockito.when(breHierarchyNodeRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(list);
        Mockito.when(breHierarchyNodeRepositoryMock.prepareQuerySpecFilterForBreHierarchyNodeMarketName(HierarchyLevel.MARKET, marketName)).thenCallRealMethod();
        Mockito.when(breHierarchyNodeRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(dataList);
        Mockito.when(breHierarchyNodeRepositoryMock.prepareQuerySpecFilterForBreHierarchyNodeMarketId(HierarchyLevel.REGION, 1)).thenCallRealMethod();
        Mockito.when(breHierarchyNodeRepositoryMock.prepareQuerySpecFilterForRegionBreHierarchyNodeMarketId(HierarchyLevel.REGION, 1, region)).thenCallRealMethod();
        Mockito.when(restaurantRepositoryMock.prepareQuerySpecFilterForRestaurantByRegionAndMarket(1, 1)).thenCallRealMethod();
        Mockito.when(restaurantRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(restaurantResourceList);
        Assertions.assertNotNull(storeDashboardService.getDemograhicDetails(hierarchyLevel, marketName, region));
    }


}
