package com.mcd.restaurant.dashboard;

import com.mcd.restaurant.dashboard.controller.StoreDashboardController;
import com.mcd.restaurant.dashboard.controller.view.response.BREHierarchyNodeDataUploadResponseDTO;
import com.mcd.restaurant.dashboard.controller.view.response.BREStoreDemographicUploadResponseDTO;
import com.mcd.restaurant.dashboard.controller.view.response.BREStoreIPAddressResponseDTO;
import com.mcd.restaurant.dashboard.controller.view.response.DashboardDetailResponseDTO;
import com.mcd.restaurant.dashboard.service.StoreDashboardService;
import com.mcd.restaurant.dashboard.testUtils.StoreDashboardTestUtils;
import io.restassured.RestAssured;
import io.restassured.builder.RequestSpecBuilder;
import io.restassured.specification.RequestSpecification;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.web.server.LocalServerPort;
import org.springframework.restdocs.RestDocumentationContextProvider;
import org.springframework.restdocs.RestDocumentationExtension;
import org.springframework.restdocs.operation.preprocess.Preprocessors;
import org.springframework.restdocs.restassured3.RestAssuredRestDocumentation;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit.jupiter.SpringExtension;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;

import static org.hamcrest.CoreMatchers.*;

@ExtendWith({SpringExtension.class, RestDocumentationExtension.class})
@SpringBootTest(webEnvironment = SpringBootTest.WebEnvironment.RANDOM_PORT)
@ActiveProfiles("test")
class StoreDashboardControllerTest {
    @LocalServerPort
    int port;

    private RequestSpecification spec;
    @InjectMocks
    private StoreDashboardTestUtils storeDashboardTestUtils;
    @Mock
    private StoreDashboardService storeDashboardServiceMock;
    @InjectMocks
    private StoreDashboardController storeDashboardController;

    @BeforeEach
    public void setUp(RestDocumentationContextProvider restDocumentation) {
        RestAssured.port = port;

        this.spec = new RequestSpecBuilder()
                .addFilter(RestAssuredRestDocumentation.documentationConfiguration(restDocumentation)
                        .operationPreprocessors().withResponseDefaults(Preprocessors.prettyPrint()))
                .build();
    }

    @Test
    void getStoreDetailsError() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .get("/store-dashboard/details/2")
                .then().assertThat().statusCode(is(500));
    }

    @Test
    void addIpaddressToStoreSuccess() throws IOException {
        MultipartFile file = storeDashboardTestUtils.getMultipartValidFileForIpAddress();
        String marketName = "US Country Office";
        Mockito.when(storeDashboardServiceMock.updateStoreHierarchyNodeWithIpAddress(marketName, file)).thenReturn(new BREStoreIPAddressResponseDTO());
        Assertions.assertNotNull(storeDashboardController.addIpaddressToStore(marketName, file));
    }

    @Test
    void getStoreDetailsSuccess() {
        Mockito.when(storeDashboardServiceMock.getStoreDetails(Mockito.anyInt())).thenReturn(new DashboardDetailResponseDTO());
        Assertions.assertNotNull(storeDashboardController.getStoreDetails(1));
    }

    @Test
    void onBoardDataToBREHierarchyNode() throws IOException {
        MultipartFile file = storeDashboardTestUtils.getMultipartValidFileForHierarchyStructure();
        Mockito.when(storeDashboardServiceMock.uploadDataToDataBase(file)).thenReturn(new BREHierarchyNodeDataUploadResponseDTO());
        Assertions.assertNotNull(storeDashboardController.onBoardDataToBREHierarchyNode(file));
    }

    @Test
    void onBoardDemographicDataToBREHierarchyNode() throws IOException {
        MultipartFile file = storeDashboardTestUtils.getMultipartValidFileForDemoGraphicStructure();
        Mockito.when(storeDashboardServiceMock.uploadDemographicDataToDataBase(file)).thenReturn(new BREStoreDemographicUploadResponseDTO());
        Assertions.assertNotNull(storeDashboardController.onBoardDemographicDataToBREHierarchyNode(file));
    }

    @Test
    void getHierarchyFilterDataSuccess() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .get("/store-dashboard/hierarchy-filter")
                .then().assertThat().statusCode(is(200));
    }

    @Test
    void mapRestaurantwithHierarchyNodeSuccess() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .patch("/store-dashboard/restaurant-data/mapping?actualMarketName=US&dbMarketId=1")
                .then().assertThat().statusCode(is(200));
    }

    @Test
    void patchUpdateAOTStatusSucccess() {
        String jsonBody = "{\n" +
                "    \"restaurantNo\":\"101\",\n" +
                "    \"market\":\"US\", \n" +
                "    \"aotStatus\":\"8002\"\n" +
                "}";
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .patch("/store-dashboard/status/update")
                .then().assertThat().statusCode(is(200));
    }

    @Test
    void patchUpdateAOTStatusMissingParameter() {
        String jsonBody = "{\n" +
                "   \n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .patch("/store-dashboard/status/update")
                .then().assertThat().statusCode(is(400));
    }

    @Test
    void patchUpdateAOTStatusInvalidStatus() {
        String jsonBody = "{\n" +
                "    \"restaurantNo\":\"101\",\n" +
                "    \"market\":\"US\"\n" +
                "    \"aotStatus\":\"12\"\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .patch("/store-dashboard/status/update")
                .then().assertThat().statusCode(is(400));
    }

    @Test
    void fetchDemographicDetailsSuccess() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .get("/store-dashboard/demographic/data?hierarchyLevel=REGION&marketName=US")
                .then().assertThat().statusCode(is(200));
    }


}
