package com.mcd.restaurant;

import io.restassured.RestAssured;
import io.restassured.builder.RequestSpecBuilder;
import io.restassured.response.Response;
import io.restassured.specification.RequestSpecification;
import net.minidev.json.JSONObject;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.CsvSource;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.web.server.LocalServerPort;
import org.springframework.restdocs.RestDocumentationContextProvider;
import org.springframework.restdocs.RestDocumentationExtension;
import org.springframework.restdocs.operation.preprocess.Preprocessors;
import org.springframework.restdocs.restassured3.RestAssuredRestDocumentation;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit.jupiter.SpringExtension;

import static org.hamcrest.CoreMatchers.*;

@ExtendWith({SpringExtension.class, RestDocumentationExtension.class})
@SpringBootTest(webEnvironment = SpringBootTest.WebEnvironment.RANDOM_PORT)
@ActiveProfiles("test")
class RestaurantTests {

    private RequestSpecification spec;

    @LocalServerPort
    int port;

    @BeforeEach
    public void setUp(RestDocumentationContextProvider restDocumentation) {
        RestAssured.port = port;

        this.spec = new RequestSpecBuilder()
                .addFilter(RestAssuredRestDocumentation.documentationConfiguration(restDocumentation)
                        .operationPreprocessors().withResponseDefaults(Preprocessors.prettyPrint()))
                .build();
    }

    @Test
    void getRestaurantWhenIDIsInvalid() {
        RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/restaurants/123").then()
                .assertThat().statusCode(is(404));
    }

    @Test
    void getRestaurantWhenIDIsNull() {
        RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/restaurants/null").then()
                .assertThat().statusCode(is(500));
    }

    @Test
    void getRestaurantWhenIDIsMissing() {
        RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/restaurants/").then()
                .statusCode(is(200)).body("data.size()", not(0));
    }

    @Test
    void postRestaurantWhenInputJSONIsEmpty() {
        String jsonBody = "";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(jsonBody).when()
                .post("/restaurant_assets/restaurants").then().assertThat().statusCode(is(400));
    }

    @Test
    void postRestaurantWhenInputJSONIsInvalid() {
        String jsonBody = "{}";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(jsonBody).when()
                .post("/restaurant_assets/restaurants").then().assertThat().statusCode(is(400));
    }

    @Test
    void postRestaurantWhenInputJSONIsInvalidDataKeyMissing() {
        JSONObject body = new JSONObject();
        JSONObject relationShipJSON = new JSONObject();
        JSONObject marketJSON = new JSONObject();
        JSONObject ownerJSON = new JSONObject();
        JSONObject marketDataJSON = new JSONObject();
        JSONObject ownerOperatorJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Restaurant_Name_New");
        // fill data JSON
        body.put("type", "restaurants");
        body.put("attributes", attributeJSON);
        // fill market and owner operator json
        marketDataJSON.put("id", "1");
        marketDataJSON.put("type", "markets");
        ownerOperatorJSON.put("id", "1");
        ownerOperatorJSON.put("type", "owner_operators");

        marketJSON.put("data", marketDataJSON);
        ownerJSON.put("data", ownerOperatorJSON);
        relationShipJSON.put("market", marketJSON);
        relationShipJSON.put("ownerOperator", ownerJSON);

        body.put("relationships", relationShipJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/restaurants").then().assertThat().statusCode(is(400));
    }

    @Test
    void postRestaurantWhenOwnerRelationshipIsMissing() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject relationShipJSON = new JSONObject();
        JSONObject marketJSON = new JSONObject();
        JSONObject marketDataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Restaurant_Name_New");
        // fill data JSON
        dataJSON.put("type", "restaurants");
        dataJSON.put("attributes", attributeJSON);
        // fill market and owner operator json
        marketDataJSON.put("id", "1");
        marketDataJSON.put("type", "markets");

        marketJSON.put("data", marketDataJSON);
        relationShipJSON.put("market", marketJSON);
        dataJSON.put("relationships", relationShipJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/restaurants").then().assertThat().statusCode(is(201));
    }

    @Test
    void postRestaurantWhenOwnerIdIsMissing() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject relationShipJSON = new JSONObject();
        JSONObject marketJSON = new JSONObject();
        JSONObject ownerJSON = new JSONObject();
        JSONObject marketDataJSON = new JSONObject();
        JSONObject ownerOperatorJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Restaurant_Name_New");
        // fill data JSON
        dataJSON.put("type", "restaurants");
        dataJSON.put("attributes", attributeJSON);
        // fill market and owner operator json
        marketDataJSON.put("id", "1");
        marketDataJSON.put("type", "markets");
        ownerOperatorJSON.put("type", "owner_operators");

        marketJSON.put("data", marketDataJSON);
        ownerJSON.put("data", ownerOperatorJSON);
        relationShipJSON.put("market", marketJSON);
        relationShipJSON.put("ownerOperator", ownerJSON);

        dataJSON.put("relationships", relationShipJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/restaurants").then().assertThat().statusCode(is(500));
    }

    @ParameterizedTest
    @CsvSource({"abc,500", "null,500", ",500"})
    void postRestaurantWhenOwnerIdIsEmpty(String id, int code) {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject relationShipJSON = new JSONObject();
        JSONObject marketJSON = new JSONObject();
        JSONObject ownerJSON = new JSONObject();
        JSONObject marketDataJSON = new JSONObject();
        JSONObject ownerOperatorJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Restaurant_Name_New");
        // fill data JSON
        dataJSON.put("type", "restaurants");
        dataJSON.put("attributes", attributeJSON);
        // fill market and owner operator json
        marketDataJSON.put("id", "1");
        marketDataJSON.put("type", "markets");
        ownerOperatorJSON.put("id", id);
        ownerOperatorJSON.put("type", "owner_operators");

        marketJSON.put("data", marketDataJSON);
        ownerJSON.put("data", ownerOperatorJSON);
        relationShipJSON.put("market", marketJSON);
        relationShipJSON.put("ownerOperator", ownerJSON);

        dataJSON.put("relationships", relationShipJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/restaurants").then().assertThat().statusCode(is(code));
    }

    @ParameterizedTest
    @CsvSource({"12,404", "abc,500", "null,500", ",500"})
    void postRestaurantWhenMarketIdIsEmpty(String id, int code) {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject relationShipJSON = new JSONObject();
        JSONObject marketJSON = new JSONObject();
        JSONObject ownerJSON = new JSONObject();
        JSONObject marketDataJSON = new JSONObject();
        JSONObject ownerOperatorJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Restaurant_Name_New");
        // fill data JSON
        dataJSON.put("type", "restaurants");
        dataJSON.put("attributes", attributeJSON);
        // fill market and owner operator json
        marketDataJSON.put("id", id);
        marketDataJSON.put("type", "markets");
        ownerOperatorJSON.put("id", "1");
        ownerOperatorJSON.put("type", "owner_operators");

        marketJSON.put("data", marketDataJSON);
        ownerJSON.put("data", ownerOperatorJSON);
        relationShipJSON.put("market", marketJSON);
        relationShipJSON.put("ownerOperator", ownerJSON);

        dataJSON.put("relationships", relationShipJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/restaurants").then().assertThat().statusCode(is(code));
    }


    @ParameterizedTest
    @CsvSource({"12,12,404", "abc,xyz,500", "null,null,500", ",,500"})
    void postRestaurantWhenMarketAndOwnerIDAreInvalid(String marketId,String ownerId,int code) {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject relationShipJSON = new JSONObject();
        JSONObject marketJSON = new JSONObject();
        JSONObject ownerJSON = new JSONObject();
        JSONObject marketDataJSON = new JSONObject();
        JSONObject ownerOperatorJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Restaurant_Name_New");
        // fill data JSON
        dataJSON.put("type", "restaurants");
        dataJSON.put("attributes", attributeJSON);
        // fill market and owner operator json
        marketDataJSON.put("id", marketId);
        marketDataJSON.put("type", "markets");
        ownerOperatorJSON.put("id", ownerId);
        ownerOperatorJSON.put("type", "owner_operators");

        marketJSON.put("data", marketDataJSON);
        ownerJSON.put("data", ownerOperatorJSON);
        relationShipJSON.put("market", marketJSON);
        relationShipJSON.put("ownerOperator", ownerJSON);

        dataJSON.put("relationships", relationShipJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/restaurants").then().assertThat().statusCode(is(code));
    }


    @ParameterizedTest
    @CsvSource({"12,404", "abc,500", "null,500", ",500"})
    void patchRestaurantWhenRestaurantIdIsDoesNotExists(String id, int code) {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject relationShipJSON = new JSONObject();
        JSONObject marketJSON = new JSONObject();
        JSONObject ownerJSON = new JSONObject();
        JSONObject marketDataJSON = new JSONObject();
        JSONObject ownerOperatorJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Restaurant_Name_New");
        // fill data JSON
        dataJSON.put("type", "restaurants");
        dataJSON.put("attributes", attributeJSON);
        // fill market and owner operator json
        marketDataJSON.put("id", "1");
        marketDataJSON.put("type", "markets");
        ownerOperatorJSON.put("id", "1");
        ownerOperatorJSON.put("type", "owner_operators");

        marketJSON.put("data", marketDataJSON);
        ownerJSON.put("data", ownerOperatorJSON);
        relationShipJSON.put("market", marketJSON);
        relationShipJSON.put("ownerOperator", ownerJSON);

        dataJSON.put("relationships", relationShipJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .patch("/restaurant_assets/restaurants/"+id).then().assertThat().statusCode(is(code));
    }


    @Test
    void patchRestaurantWhenRestaurantIdIsMising() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject relationShipJSON = new JSONObject();
        JSONObject marketJSON = new JSONObject();
        JSONObject ownerJSON = new JSONObject();
        JSONObject marketDataJSON = new JSONObject();
        JSONObject ownerOperatorJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Restaurant_Name_New");
        // fill data JSON
        dataJSON.put("type", "restaurants");
        dataJSON.put("attributes", attributeJSON);
        // fill market and owner operator json
        marketDataJSON.put("id", "1");
        marketDataJSON.put("type", "markets");
        ownerOperatorJSON.put("id", "1");
        ownerOperatorJSON.put("type", "owner_operators");

        marketJSON.put("data", marketDataJSON);
        ownerJSON.put("data", ownerOperatorJSON);
        relationShipJSON.put("market", marketJSON);
        relationShipJSON.put("ownerOperator", ownerJSON);

        dataJSON.put("relationships", relationShipJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .patch("/restaurant_assets/restaurants/").then().assertThat().statusCode(is(400));
    }


    @Test
    void patchRestaurantWhenInputJSONIsEmptyString() {
        String body = "";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body).when()
                .patch("/restaurant_assets/restaurants/1").then().assertThat().statusCode(is(400));
    }

    @Test
    void patchRestaurantWhenInputJSONIsEmptyAndRestaurantIdIsInvalid() {
        String body = "";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body).when()
                .patch("/restaurant_assets/restaurants/abc").then().assertThat().statusCode(is(500));
    }
}
