package com.mcd.restaurant;

import com.mcd.restaurant.repository.MarketRepository;
import io.restassured.RestAssured;
import io.restassured.builder.RequestSpecBuilder;
import io.restassured.specification.RequestSpecification;
import net.minidev.json.JSONObject;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.CsvSource;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.web.server.LocalServerPort;
import org.springframework.restdocs.RestDocumentationContextProvider;
import org.springframework.restdocs.RestDocumentationExtension;
import org.springframework.restdocs.operation.preprocess.Preprocessors;
import org.springframework.restdocs.restassured3.RestAssuredRestDocumentation;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit.jupiter.SpringExtension;

import static org.hamcrest.CoreMatchers.*;

@ExtendWith({SpringExtension.class, RestDocumentationExtension.class})
@SpringBootTest(webEnvironment = SpringBootTest.WebEnvironment.RANDOM_PORT)
@ActiveProfiles("test")
class MarketTests {

    private RequestSpecification spec;

    @Autowired
    private MarketRepository marketRepository;

    @LocalServerPort
    int port;

    @BeforeEach
    public void setUp(RestDocumentationContextProvider restDocumentation) {
        RestAssured.port = port;

        this.spec = new RequestSpecBuilder()
                .addFilter(RestAssuredRestDocumentation.documentationConfiguration(restDocumentation)
                        .operationPreprocessors().withResponseDefaults(Preprocessors.prettyPrint()))
                .build();
    }

    @Test
    void getMarketWhenIDIsInvalid() {
        RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/markets/123").then()
                .assertThat().statusCode(is(404));
    }

    @Test
    void getMarketWhenIDIsNull() {
        RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/markets/null").then()
                .assertThat().statusCode(is(500));
    }

    @Test
    void getMarketWhenIDIsMissing() {
        RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/markets/").then()
                .statusCode(is(200)).body("data.size()", not(0));
    }

    @Test
    void postMarketWhenInputJSONIsEmpty() {
        String jsonBody = "";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(jsonBody).when()
                .post("/restaurant_assets/markets").then().assertThat().statusCode(is(400));
    }

    @Test
    void postMarketWhenInputJSONIsInvalid() {
        String jsonBody = "{}";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(jsonBody).when()
                .post("/restaurant_assets/markets").then().assertThat().statusCode(is(400));
    }

    @Test
    void postMarketWhenInputJSONIsInvalidDataKeyMissing() {
        JSONObject body = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Market_2");
        attributeJSON.put("type", "Market_Type_New");
        // fill data JSON
        body.put("type", "markets");
        body.put("attributes", attributeJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/markets").then().assertThat().statusCode(is(400));
    }

    @Test
     void postMarketTypeAndNameMissing() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/markets").then().assertThat().statusCode(is(422));
    }

    @Test
    void postMarketWhenTypeNameBothMissing() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        // fill data JSON
        dataJSON.put("attributes", attributeJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/markets").then().assertThat().statusCode(is(422));
    }

    @Test
    void postMarketWhenInternalTypeIsNull() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "abc");
        attributeJSON.put("type", null);
        // fill data JSON
        dataJSON.put("type", "markets");
        dataJSON.put("attributes", attributeJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/markets").then().assertThat().statusCode(is(201));
    }

    @Test
    void postMarketWhenInternalTypeIsEmpty() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "abc");
        attributeJSON.put("type", "");
        // fill data JSON
        dataJSON.put("type", "markets");
        dataJSON.put("attributes", attributeJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/markets").then().assertThat().statusCode(is(201));
    }


    @ParameterizedTest
    @CsvSource({"12,404", "abc,500", "null,500", ",500"})
    void patchMarketWhenMarketIdIsInvalid(String id, int code) {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Market_2");
        attributeJSON.put("type", "Market_Type_New");
        // fill data JSON
        dataJSON.put("type", "markets");
        dataJSON.put("attributes", attributeJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .patch("/restaurant_assets/markets/"+id).then().assertThat().statusCode(is(code));
    }

    @Test
    void patchMarketWhenMarketIdIsMising() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Market_2");
        attributeJSON.put("type", "Market_Type_New");
        // fill data JSON
        dataJSON.put("type", "markets");
        dataJSON.put("attributes", attributeJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .patch("/restaurant_assets/markets/").then().assertThat().statusCode(is(400));
    }


    @Test
    void patchMarketWhenInputJSONIsEmptyString() {
        String body = "";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .patch("/restaurant_assets/markets/1").then().assertThat().statusCode(is(400));
    }

    @Test
    void patchMarketWhenInputJSONIsEmptyAndMarketIdIsInvalid() {
        String body = "";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .patch("/restaurant_assets/markets/abc").then().assertThat().statusCode(is(500));
    }
}
