package com.mcd.restaurant;

import io.restassured.RestAssured;
import io.restassured.builder.RequestSpecBuilder;
import io.restassured.specification.RequestSpecification;
import net.minidev.json.JSONObject;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.CsvSource;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.web.server.LocalServerPort;
import org.springframework.restdocs.RestDocumentationContextProvider;
import org.springframework.restdocs.RestDocumentationExtension;
import org.springframework.restdocs.operation.preprocess.Preprocessors;
import org.springframework.restdocs.restassured3.RestAssuredRestDocumentation;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit.jupiter.SpringExtension;

import static org.hamcrest.CoreMatchers.is;

@ExtendWith({SpringExtension.class, RestDocumentationExtension.class})
@SpringBootTest(webEnvironment = SpringBootTest.WebEnvironment.RANDOM_PORT)
@ActiveProfiles("test")
class ComponentTests {

    private RequestSpecification spec;

    @LocalServerPort
    int port;

    @BeforeEach
    public void setUp(RestDocumentationContextProvider restDocumentation) {
        RestAssured.port = port;

        this.spec = new RequestSpecBuilder()
                .addFilter(RestAssuredRestDocumentation.documentationConfiguration(restDocumentation).operationPreprocessors()
                        .withResponseDefaults(Preprocessors.prettyPrint()))
                .build();
    }


    @Test
    void getComponentWhenIDIsInvalid() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .when().get("/restaurant_assets/components/123")
                .then().assertThat().statusCode(is(404));
    }

    @Test
    void getComponentWhenIDIsNull() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .when().get("/restaurant_assets/components/null")
                .then().assertThat().statusCode(is(500));
    }

    @Test
     void getComponentWhenIDIsMissing() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .when().get("/restaurant_assets/components/")
                .then().statusCode(is(200));
    }

    @Test
    void postComponentWhenInputJSONIsEmpty() {
        String jsonBody = "";

        RestAssured.given(this.spec)
                .contentType("application/vnd.api+json")
                .body(jsonBody)
                .when().post("/restaurant_assets/components")
                .then().assertThat().statusCode(is(400));
    }

    @Test
    void postComponentWhenInputJSONIsInvalid() {
        String jsonBody = "{}";

        RestAssured.given(this.spec)
                .contentType("application/vnd.api+json")
                .body(jsonBody)
                .when().post("/restaurant_assets/components")
                .then().assertThat().statusCode(is(400));
    }

    @Test
    void postComponentWhenInputJSONIsInvalidDataKeyMissing() {
        JSONObject body = new JSONObject();
        JSONObject relationShipJSON = new JSONObject();
        JSONObject vendorJSON = new JSONObject();
        JSONObject typeJSON = new JSONObject();
        JSONObject restJSON = new JSONObject();
        JSONObject restDataJSON = new JSONObject();
        JSONObject vendorDataJSON = new JSONObject();
        JSONObject typeDataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Restaurant_Name_New");
        attributeJSON.put("serialNumber", "344209859");
        attributeJSON.put("reportedVersion", "1.1");
        attributeJSON.put("installDate", "2019-08-20T01:14:14");
        attributeJSON.put("parentComponent", "0");
        attributeJSON.put("warranty", "2019-07-10T01:14:04");
        attributeJSON.put("modelNumber", 134);
        //fill data JSON
        body.put("type", "components");
        body.put("attributes", attributeJSON);
        //fill market and owner operator json
        vendorDataJSON.put("id", "1");
        vendorDataJSON.put("type", "vendors");
        typeDataJSON.put("id", "1");
        typeDataJSON.put("type", "component_type");
        restDataJSON.put("id", "1");
        restDataJSON.put("type", "restaurants");

        typeJSON.put("data", typeDataJSON);
        vendorJSON.put("data", vendorDataJSON);
        restJSON.put("data", restDataJSON);
        relationShipJSON.put("type", typeJSON);
        relationShipJSON.put("vendor", vendorJSON);
        relationShipJSON.put("restaurant", restJSON);

        body.put("relationships", relationShipJSON);

        RestAssured.given(this.spec)
                .contentType("application/vnd.api+json")
                .body(body.toString())
                .when().post("/restaurant_assets/components")
                .then().assertThat().statusCode(is(400));
    }

    @Test
    void postComponentWhenComponentIdIsMissing() {
        JSONObject body = new JSONObject();
        JSONObject relationShipJSON = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject vendorJSON = new JSONObject();
        JSONObject typeJSON = new JSONObject();
        JSONObject restJSON = new JSONObject();
        JSONObject restDataJSON = new JSONObject();
        JSONObject vendorDataJSON = new JSONObject();
        JSONObject typeDataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Restaurant_Name_New");
        attributeJSON.put("serialNumber", "344209859");
        attributeJSON.put("reportedVersion", "1.1");
        attributeJSON.put("installDate", "2019-08-20T01:14:14");
        attributeJSON.put("parentComponent", "0");
        attributeJSON.put("warranty", "2019-07-10T01:14:04");
        attributeJSON.put("modelNumber", 134);
        //fill data JSON
        dataJSON.put("type", "components");
        dataJSON.put("attributes", attributeJSON);
        //fill market and owner operator json
        vendorDataJSON.put("id", "1");
        vendorDataJSON.put("type", "vendors");
        typeDataJSON.put("id", "1");
        typeDataJSON.put("type", "component_type");
        restDataJSON.put("type", "restaurants");

        typeJSON.put("data", typeDataJSON);
        vendorJSON.put("data", vendorDataJSON);
        restJSON.put("data", restDataJSON);
        relationShipJSON.put("type", typeJSON);
        relationShipJSON.put("vendor", vendorJSON);
        relationShipJSON.put("restaurant", restJSON);

        dataJSON.put("relationships", relationShipJSON);
        body.put("data", dataJSON);


        RestAssured.given(this.spec)
                .contentType("application/vnd.api+json")
                .body(body.toString())
                .when().post("/restaurant_assets/components")
                .then().assertThat().statusCode(is(500));
    }

    @ParameterizedTest
    @CsvSource({"abc,500","null,500",",500"})
    void postComponentWhenComponentIdIsEmpty(String id,int code) {
        JSONObject body = new JSONObject();
        JSONObject relationShipJSON = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject vendorJSON = new JSONObject();
        JSONObject typeJSON = new JSONObject();
        JSONObject restJSON = new JSONObject();
        JSONObject restDataJSON = new JSONObject();
        JSONObject vendorDataJSON = new JSONObject();
        JSONObject typeDataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Restaurant_Name_New");
        attributeJSON.put("serialNumber", "344209859");
        attributeJSON.put("reportedVersion", "1.1");
        attributeJSON.put("installDate", "2019-08-20T01:14:14");
        attributeJSON.put("parentComponent", "0");
        attributeJSON.put("warranty", "2019-07-10T01:14:04");
        attributeJSON.put("modelNumber", 134);
        //fill data JSON
        dataJSON.put("type", "components");
        dataJSON.put("attributes", attributeJSON);
        //fill market and owner operator json
        vendorDataJSON.put("id", "1");
        vendorDataJSON.put("type", "vendors");
        typeDataJSON.put("id", "1");
        typeDataJSON.put("type", "component_type");
        restDataJSON.put("id", id);
        restDataJSON.put("type", "restaurants");

        typeJSON.put("data", typeDataJSON);
        vendorJSON.put("data", vendorDataJSON);
        restJSON.put("data", restDataJSON);
        relationShipJSON.put("type", typeJSON);
        relationShipJSON.put("vendor", vendorJSON);
        relationShipJSON.put("restaurant", restJSON);

        dataJSON.put("relationships", relationShipJSON);
        body.put("data", dataJSON);


        RestAssured.given(this.spec)
                .contentType("application/vnd.api+json")
                .body(body.toString())
                .when().post("/restaurant_assets/components")
                .then().assertThat().statusCode(is(code));
    }

//    @Test
//    void postComponentWhenComponentIdIsNull() {
//        JSONObject body = new JSONObject();
//        JSONObject relationShipJSON = new JSONObject();
//        JSONObject dataJSON = new JSONObject();
//        JSONObject vendorJSON = new JSONObject();
//        JSONObject typeJSON = new JSONObject();
//        JSONObject restJSON = new JSONObject();
//        JSONObject restDataJSON = new JSONObject();
//        JSONObject vendorDataJSON = new JSONObject();
//        JSONObject typeDataJSON = new JSONObject();
//        JSONObject attributeJSON = new JSONObject();
//        attributeJSON.put("name", "Restaurant_Name_New");
//        attributeJSON.put("serialNumber", "344209859");
//        attributeJSON.put("reportedVersion", "1.1");
//        attributeJSON.put("installDate", "2019-08-20T01:14:14");
//        attributeJSON.put("parentComponent", "0");
//        attributeJSON.put("warranty", "2019-07-10T01:14:04");
//        attributeJSON.put("modelNumber", 134);
//        //fill data JSON
//        dataJSON.put("type", "components");
//        dataJSON.put("attributes", attributeJSON);
//        //fill market and owner operator json
//        vendorDataJSON.put("id", "1");
//        vendorDataJSON.put("type", "vendors");
//        typeDataJSON.put("id", "1");
//        typeDataJSON.put("type", "component_type");
//        restDataJSON.put("id", null);
//        restDataJSON.put("type", "restaurants");
//
//        typeJSON.put("data", typeDataJSON);
//        vendorJSON.put("data", vendorDataJSON);
//        restJSON.put("data", restDataJSON);
//        relationShipJSON.put("type", typeJSON);
//        relationShipJSON.put("vendor", vendorJSON);
//        relationShipJSON.put("restaurant", restJSON);
//
//        dataJSON.put("relationships", relationShipJSON);
//        body.put("data", dataJSON);
//
//
//        RestAssured.given(this.spec)
//                .contentType("application/vnd.api+json")
//                .body(body.toString())
//                .when().post("/restaurant_assets/components")
//                .then().assertThat().statusCode(is(500));
//    }

//    @Test
//    void postComponentWhenComponentIdIsInvalid() {
//        JSONObject body = new JSONObject();
//        JSONObject relationShipJSON = new JSONObject();
//        JSONObject dataJSON = new JSONObject();
//        JSONObject vendorJSON = new JSONObject();
//        JSONObject typeJSON = new JSONObject();
//        JSONObject restJSON = new JSONObject();
//        JSONObject restDataJSON = new JSONObject();
//        JSONObject vendorDataJSON = new JSONObject();
//        JSONObject typeDataJSON = new JSONObject();
//        JSONObject attributeJSON = new JSONObject();
//        attributeJSON.put("name", "Restaurant_Name_New");
//        attributeJSON.put("serialNumber", "344209859");
//        attributeJSON.put("reportedVersion", "1.1");
//        attributeJSON.put("installDate", "2019-08-20T01:14:14");
//        attributeJSON.put("parentComponent", "0");
//        attributeJSON.put("warranty", "2019-07-10T01:14:04");
//        attributeJSON.put("modelNumber", 134);
//        //fill data JSON
//        dataJSON.put("type", "components");
//        dataJSON.put("attributes", attributeJSON);
//        //fill market and owner operator json
//        vendorDataJSON.put("id", "1");
//        vendorDataJSON.put("type", "vendors");
//        typeDataJSON.put("id", "1");
//        typeDataJSON.put("type", "component_type");
//        restDataJSON.put("id", "adc");
//        restDataJSON.put("type", "restaurants");
//
//        typeJSON.put("data", typeDataJSON);
//        vendorJSON.put("data", vendorDataJSON);
//        restJSON.put("data", restDataJSON);
//        relationShipJSON.put("type", typeJSON);
//        relationShipJSON.put("vendor", vendorJSON);
//        relationShipJSON.put("restaurant", restJSON);
//
//        dataJSON.put("relationships", relationShipJSON);
//        body.put("data", dataJSON);
//
//
//        RestAssured.given(this.spec)
//                .contentType("application/vnd.api+json")
//                .body(body.toString())
//                .when().post("/restaurant_assets/components")
//                .then().assertThat().statusCode(is(500));
//    }

    @ParameterizedTest
    @CsvSource({"abc,500","null,500",",500"})
    void postComponentWhenVendorIdIsMissing(String id,int code) {
        JSONObject body = new JSONObject();
        JSONObject relationShipJSON = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject vendorJSON = new JSONObject();
        JSONObject typeJSON = new JSONObject();
        JSONObject restJSON = new JSONObject();
        JSONObject restDataJSON = new JSONObject();
        JSONObject vendorDataJSON = new JSONObject();
        JSONObject typeDataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Restaurant_Name_New");
        attributeJSON.put("serialNumber", "344209859");
        attributeJSON.put("reportedVersion", "1.1");
        attributeJSON.put("installDate", "2019-08-20T01:14:14");
        attributeJSON.put("parentComponent", "0");
        attributeJSON.put("warranty", "2019-07-10T01:14:04");
        attributeJSON.put("modelNumber", 134);
        //fill data JSON
        dataJSON.put("type", "components");
        dataJSON.put("attributes", attributeJSON);
        //fill market and owner operator json
        vendorDataJSON.put("id", id);
        vendorDataJSON.put("type", "vendors");
        typeDataJSON.put("id", "1");
        typeDataJSON.put("type", "component_type");
        restDataJSON.put("id", "1");
        restDataJSON.put("type", "restaurants");

        typeJSON.put("data", typeDataJSON);
        vendorJSON.put("data", vendorDataJSON);
        restJSON.put("data", restDataJSON);
        relationShipJSON.put("type", typeJSON);
        relationShipJSON.put("vendor", vendorJSON);
        relationShipJSON.put("restaurant", restJSON);

        dataJSON.put("relationships", relationShipJSON);
        body.put("data", dataJSON);


        RestAssured.given(this.spec)
                .contentType("application/vnd.api+json")
                .body(body.toString())
                .when().post("/restaurant_assets/components")
                .then().assertThat().statusCode(is(code));
    }

//    @Test
//    void postComponentWhenVendorIdIsEmpty() {
//        JSONObject body = new JSONObject();
//        JSONObject relationShipJSON = new JSONObject();
//        JSONObject dataJSON = new JSONObject();
//        JSONObject vendorJSON = new JSONObject();
//        JSONObject typeJSON = new JSONObject();
//        JSONObject restJSON = new JSONObject();
//        JSONObject restDataJSON = new JSONObject();
//        JSONObject vendorDataJSON = new JSONObject();
//        JSONObject typeDataJSON = new JSONObject();
//        JSONObject attributeJSON = new JSONObject();
//        attributeJSON.put("name", "Restaurant_Name_New");
//        attributeJSON.put("serialNumber", "344209859");
//        attributeJSON.put("reportedVersion", "1.1");
//        attributeJSON.put("installDate", "2019-08-20T01:14:14");
//        attributeJSON.put("parentComponent", "0");
//        attributeJSON.put("warranty", "2019-07-10T01:14:04");
//        attributeJSON.put("modelNumber", 134);
//        //fill data JSON
//        dataJSON.put("type", "components");
//        dataJSON.put("attributes", attributeJSON);
//        //fill market and owner operator json
//        vendorDataJSON.put("id", "");
//        vendorDataJSON.put("type", "vendors");
//        typeDataJSON.put("id", "1");
//        typeDataJSON.put("type", "component_type");
//        restDataJSON.put("id", "1");
//        restDataJSON.put("type", "restaurants");
//
//        typeJSON.put("data", typeDataJSON);
//        vendorJSON.put("data", vendorDataJSON);
//        restJSON.put("data", restDataJSON);
//        relationShipJSON.put("type", typeJSON);
//        relationShipJSON.put("vendor", vendorJSON);
//        relationShipJSON.put("restaurant", restJSON);
//
//        dataJSON.put("relationships", relationShipJSON);
//        body.put("data", dataJSON);
//
//        RestAssured.given(this.spec)
//                .contentType("application/vnd.api+json")
//                .body(body.toString())
//                .when().post("/restaurant_assets/components")
//                .then().assertThat().statusCode(is(500));
//    }
//
//    public void postComponentWhenVendorIdIsNull() {
//        JSONObject body = new JSONObject();
//        JSONObject relationShipJSON = new JSONObject();
//        JSONObject dataJSON = new JSONObject();
//        JSONObject vendorJSON = new JSONObject();
//        JSONObject typeJSON = new JSONObject();
//        JSONObject restJSON = new JSONObject();
//        JSONObject restDataJSON = new JSONObject();
//        JSONObject vendorDataJSON = new JSONObject();
//        JSONObject typeDataJSON = new JSONObject();
//        JSONObject attributeJSON = new JSONObject();
//        attributeJSON.put("name", "Restaurant_Name_New");
//        attributeJSON.put("serialNumber", "344209859");
//        attributeJSON.put("reportedVersion", "1.1");
//        attributeJSON.put("installDate", "2019-08-20T01:14:14");
//        attributeJSON.put("parentComponent", "0");
//        attributeJSON.put("warranty", "2019-07-10T01:14:04");
//        attributeJSON.put("modelNumber", 134);
//        //fill data JSON
//        dataJSON.put("type", "components");
//        dataJSON.put("attributes", attributeJSON);
//        //fill market and owner operator json
//        vendorDataJSON.put("id", "");
//        vendorDataJSON.put("type", "vendors");
//        typeDataJSON.put("id", "1");
//        typeDataJSON.put("type", "component_type");
//        restDataJSON.put("id", "1");
//        restDataJSON.put("type", "restaurants");
//
//        typeJSON.put("data", typeDataJSON);
//        vendorJSON.put("data", vendorDataJSON);
//        restJSON.put("data", restDataJSON);
//        relationShipJSON.put("type", typeJSON);
//        relationShipJSON.put("vendor", vendorJSON);
//        relationShipJSON.put("restaurant", restJSON);
//
//        dataJSON.put("relationships", relationShipJSON);
//        body.put("data", dataJSON);
//
//
//        RestAssured.given(this.spec)
//                .contentType("application/vnd.api+json")
//                .body(body.toString())
//                .when().post("/restaurant_assets/components")
//                .then().assertThat().statusCode(is(500));
//    }
//
//    @Test
//    void postComponentWhenVendorIdIsInvalid() {
//        JSONObject body = new JSONObject();
//        JSONObject relationShipJSON = new JSONObject();
//        JSONObject dataJSON = new JSONObject();
//        JSONObject vendorJSON = new JSONObject();
//        JSONObject typeJSON = new JSONObject();
//        JSONObject restJSON = new JSONObject();
//        JSONObject restDataJSON = new JSONObject();
//        JSONObject vendorDataJSON = new JSONObject();
//        JSONObject typeDataJSON = new JSONObject();
//        JSONObject attributeJSON = new JSONObject();
//        attributeJSON.put("name", "Restaurant_Name_New");
//        attributeJSON.put("serialNumber", "344209859");
//        attributeJSON.put("reportedVersion", "1.1");
//        attributeJSON.put("installDate", "2019-08-20T01:14:14");
//        attributeJSON.put("parentComponent", "0");
//        attributeJSON.put("warranty", "2019-07-10T01:14:04");
//        attributeJSON.put("modelNumber", 134);
//        //fill data JSON
//        dataJSON.put("type", "components");
//        dataJSON.put("attributes", attributeJSON);
//        //fill market and owner operator json
//        vendorDataJSON.put("id", "1an");
//        vendorDataJSON.put("type", "vendors");
//        typeDataJSON.put("id", "1");
//        typeDataJSON.put("type", "component_type");
//        restDataJSON.put("id", "1");
//        restDataJSON.put("type", "restaurants");
//
//        typeJSON.put("data", typeDataJSON);
//        vendorJSON.put("data", vendorDataJSON);
//        restJSON.put("data", restDataJSON);
//        relationShipJSON.put("type", typeJSON);
//        relationShipJSON.put("vendor", vendorJSON);
//        relationShipJSON.put("restaurant", restJSON);
//
//        dataJSON.put("relationships", relationShipJSON);
//        body.put("data", dataJSON);
//
//
//        RestAssured.given(this.spec)
//                .contentType("application/vnd.api+json")
//                .body(body.toString())
//                .when().post("/restaurant_assets/components")
//                .then().assertThat().statusCode(is(500));
//    }

    @Test
    void postComponentWhenRestaurantAndComponentIDAreInvalid() {
        JSONObject body = new JSONObject();
        JSONObject relationShipJSON = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject vendorJSON = new JSONObject();
        JSONObject typeJSON = new JSONObject();
        JSONObject restJSON = new JSONObject();
        JSONObject restDataJSON = new JSONObject();
        JSONObject vendorDataJSON = new JSONObject();
        JSONObject typeDataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Restaurant_Name_New");
        attributeJSON.put("serialNumber", "344209859");
        attributeJSON.put("reportedVersion", "1.1");
        attributeJSON.put("installDate", "2019-08-20T01:14:14");
        attributeJSON.put("parentComponent", "0");
        attributeJSON.put("warranty", "2019-07-10T01:14:04");
        attributeJSON.put("modelNumber", 134);
        //fill data JSON
        dataJSON.put("type", "components");
        dataJSON.put("attributes", attributeJSON);
        //fill market and owner operator json
        vendorDataJSON.put("id", "1");
        vendorDataJSON.put("type", "vendors");
        typeDataJSON.put("id", "ax");
        typeDataJSON.put("type", "component_type");
        restDataJSON.put("id", "ss");
        restDataJSON.put("type", "restaurants");

        typeJSON.put("data", typeDataJSON);
        vendorJSON.put("data", vendorDataJSON);
        restJSON.put("data", restDataJSON);
        relationShipJSON.put("type", typeJSON);
        relationShipJSON.put("vendor", vendorJSON);
        relationShipJSON.put("restaurant", restJSON);

        dataJSON.put("relationships", relationShipJSON);
        body.put("data", dataJSON);


        RestAssured.given(this.spec)
                .contentType("application/vnd.api+json")
                .body(body.toString())
                .when().post("/restaurant_assets/components")
                .then().assertThat().statusCode(is(500));
    }

    @Test
    void postComponentWhenRestaurantAndComponentIdAreEmpty() {
        JSONObject body = new JSONObject();
        JSONObject relationShipJSON = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject vendorJSON = new JSONObject();
        JSONObject typeJSON = new JSONObject();
        JSONObject restJSON = new JSONObject();
        JSONObject restDataJSON = new JSONObject();
        JSONObject vendorDataJSON = new JSONObject();
        JSONObject typeDataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Restaurant_Name_New");
        attributeJSON.put("serialNumber", "344209859");
        attributeJSON.put("reportedVersion", "1.1");
        attributeJSON.put("installDate", "2019-08-20T01:14:14");
        attributeJSON.put("parentComponent", "0");
        attributeJSON.put("warranty", "2019-07-10T01:14:04");
        attributeJSON.put("modelNumber", 134);
        //fill data JSON
        dataJSON.put("type", "components");
        dataJSON.put("attributes", attributeJSON);
        //fill market and owner operator json
        vendorDataJSON.put("id", "1");
        vendorDataJSON.put("type", "vendors");
        typeDataJSON.put("id", "");
        typeDataJSON.put("type", "component_type");
        restDataJSON.put("id", "");
        restDataJSON.put("type", "restaurants");

        typeJSON.put("data", typeDataJSON);
        vendorJSON.put("data", vendorDataJSON);
        restJSON.put("data", restDataJSON);
        relationShipJSON.put("type", typeJSON);
        relationShipJSON.put("vendor", vendorJSON);
        relationShipJSON.put("restaurant", restJSON);

        dataJSON.put("relationships", relationShipJSON);
        body.put("data", dataJSON);


        RestAssured.given(this.spec)
                .contentType("application/vnd.api+json")
                .body(body.toString())
                .when().post("/restaurant_assets/components")
                .then().assertThat().statusCode(is(500));
    }

    @Test
    void postComponentWhenRestaurantAndComponentIdAreMissing() {
        JSONObject body = new JSONObject();
        JSONObject relationShipJSON = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject vendorJSON = new JSONObject();
        JSONObject typeJSON = new JSONObject();
        JSONObject restJSON = new JSONObject();
        JSONObject restDataJSON = new JSONObject();
        JSONObject vendorDataJSON = new JSONObject();
        JSONObject typeDataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Restaurant_Name_New");
        attributeJSON.put("serialNumber", "344209859");
        attributeJSON.put("reportedVersion", "1.1");
        attributeJSON.put("installDate", "2019-08-20T01:14:14");
        attributeJSON.put("parentComponent", "0");
        attributeJSON.put("warranty", "2019-07-10T01:14:04");
        attributeJSON.put("modelNumber", 134);
        //fill data JSON
        dataJSON.put("type", "components");
        dataJSON.put("attributes", attributeJSON);
        //fill market and owner operator json
        vendorDataJSON.put("id", "1");
        vendorDataJSON.put("type", "vendors");
        typeDataJSON.put("type", "component_type");
        restDataJSON.put("type", "restaurants");

        typeJSON.put("data", typeDataJSON);
        vendorJSON.put("data", vendorDataJSON);
        restJSON.put("data", restDataJSON);
        relationShipJSON.put("type", typeJSON);
        relationShipJSON.put("vendor", vendorJSON);
        relationShipJSON.put("restaurant", restJSON);

        dataJSON.put("relationships", relationShipJSON);
        body.put("data", dataJSON);


        RestAssured.given(this.spec)
                .contentType("application/vnd.api+json")
                .body(body.toString())
                .when().post("/restaurant_assets/components")
                .then().assertThat().statusCode(is(500));
    }

    @ParameterizedTest
    @CsvSource({"40,404","abc,500","null,500",",500"})
    void patchComponentWhenComponentIdIsDoesNotExists(String id, int code) {

        JSONObject body = new JSONObject();
        JSONObject relationShipJSON = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject vendorJSON = new JSONObject();
        JSONObject typeJSON = new JSONObject();
        JSONObject restJSON = new JSONObject();
        JSONObject restDataJSON = new JSONObject();
        JSONObject vendorDataJSON = new JSONObject();
        JSONObject typeDataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Restaurant_Name_New");
        attributeJSON.put("serialNumber", "344209859");
        attributeJSON.put("reportedVersion", "1.1");
        attributeJSON.put("installDate", "2019-08-20T01:14:14");
        attributeJSON.put("parentComponent", "0");
        attributeJSON.put("warranty", "2019-07-10T01:14:04");
        attributeJSON.put("modelNumber", 134);
        //fill data JSON
        dataJSON.put("type", "components");
        dataJSON.put("attributes", attributeJSON);
        //fill market and owner operator json
        vendorDataJSON.put("id", "1");
        vendorDataJSON.put("type", "vendors");
        typeDataJSON.put("id", "1");
        typeDataJSON.put("type", "component_type");
        restDataJSON.put("id", "1");
        restDataJSON.put("type", "restaurants");

        typeJSON.put("data", typeDataJSON);
        vendorJSON.put("data", vendorDataJSON);
        restJSON.put("data", restDataJSON);
        relationShipJSON.put("type", typeJSON);
        relationShipJSON.put("vendor", vendorJSON);
        relationShipJSON.put("restaurant", restJSON);

        dataJSON.put("relationships", relationShipJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec)
                .contentType("application/vnd.api+json")
                .body(body.toString())
                .when().patch("/restaurant_assets/components/"+id)
                .then().assertThat().statusCode(is(code));
    }


    @Test
    void patchComponentWhenComponentIdIsMising() {

        JSONObject body = new JSONObject();
        JSONObject relationShipJSON = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject vendorJSON = new JSONObject();
        JSONObject typeJSON = new JSONObject();
        JSONObject restJSON = new JSONObject();
        JSONObject restDataJSON = new JSONObject();
        JSONObject vendorDataJSON = new JSONObject();
        JSONObject typeDataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Restaurant_Name_New");
        attributeJSON.put("serialNumber", "344209859");
        attributeJSON.put("reportedVersion", "1.1");
        attributeJSON.put("installDate", "2019-08-20T01:14:14");
        attributeJSON.put("parentComponent", "0");
        attributeJSON.put("warranty", "2019-07-10T01:14:04");
        attributeJSON.put("modelNumber", 134);
        //fill data JSON
        dataJSON.put("type", "components");
        dataJSON.put("attributes", attributeJSON);
        //fill market and owner operator json
        vendorDataJSON.put("id", "1");
        vendorDataJSON.put("type", "vendors");
        typeDataJSON.put("id", "1");
        typeDataJSON.put("type", "component_type");
        restDataJSON.put("id", "1");
        restDataJSON.put("type", "restaurants");

        typeJSON.put("data", typeDataJSON);
        vendorJSON.put("data", vendorDataJSON);
        restJSON.put("data", restDataJSON);
        relationShipJSON.put("type", typeJSON);
        relationShipJSON.put("vendor", vendorJSON);
        relationShipJSON.put("restaurant", restJSON);

        dataJSON.put("relationships", relationShipJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec)
                .contentType("application/vnd.api+json")
                .body(body.toString())
                .when().patch("/restaurant_assets/components/")
                .then().assertThat().statusCode(is(400));
    }


    @Test
    void patchComponentWhenInputJSONIsEmptyString() {
        String body = "";

        RestAssured.given(this.spec)
                .contentType("application/vnd.api+json")
                .body(body)
                .when().patch("/restaurant_assets/components/null")
                .then().assertThat().statusCode(is(500));
    }

    @Test
    void patchComponentWhenInputJSONIsEmptyAndComponentIdIsInvalid() {
        String body = "";

        RestAssured.given(this.spec)
                .contentType("application/vnd.api+json")
                .body(body)
                .when().patch("/restaurant_assets/components/abc")
                .then().assertThat().statusCode(is(500));
    }
}
