package com.mcd.restaurant;

import io.restassured.RestAssured;
import io.restassured.builder.RequestSpecBuilder;
import io.restassured.specification.RequestSpecification;
import net.minidev.json.JSONObject;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.CsvSource;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.web.server.LocalServerPort;
import org.springframework.restdocs.RestDocumentationContextProvider;
import org.springframework.restdocs.RestDocumentationExtension;
import org.springframework.restdocs.operation.preprocess.Preprocessors;
import org.springframework.restdocs.restassured3.RestAssuredRestDocumentation;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit.jupiter.SpringExtension;

import static org.hamcrest.CoreMatchers.is;

@ExtendWith({SpringExtension.class, RestDocumentationExtension.class})
@SpringBootTest(webEnvironment = SpringBootTest.WebEnvironment.RANDOM_PORT)
@ActiveProfiles("test")
class ComponentPropsTests {

    private RequestSpecification spec;

    @LocalServerPort
    int port;

    @BeforeEach
    public void setUp(RestDocumentationContextProvider restDocumentation) {
        RestAssured.port = port;

        this.spec = new RequestSpecBuilder()
                .addFilter(RestAssuredRestDocumentation.documentationConfiguration(restDocumentation)
                        .operationPreprocessors().withResponseDefaults(Preprocessors.prettyPrint()))
                .build();
    }

    @Test
    void getComponentPropsWhenIDIsInvalid() {
        RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/component_props/123")
                .then().assertThat().statusCode(is(404));
    }

    @Test
    void getComponentPropsWhenIDIsNull() {
        RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/component_props/null")
                .then().assertThat().statusCode(is(500));
    }

    @Test
    void getComponentPropsWhenIDIsMissing() {
        RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/component_props/").then()
                .statusCode(is(200));
    }

    @Test
    void postComponentPropsWhenInputJSONIsEmpty() {
        String jsonBody = "";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(jsonBody).when()
                .post("/restaurant_assets/component_props").then().assertThat().statusCode(is(400));
    }

    @Test
    void postComponentPropsWhenInputJSONIsInvalid() {
        String jsonBody = "{}";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(jsonBody).when()
                .post("/restaurant_assets/component_props").then().assertThat().statusCode(is(400));
    }

    @Test
    void postComponentPropsWhenInputJSONIsInvalidDataKeyMissing() {
        JSONObject body = new JSONObject();
        JSONObject relationShipJSON = new JSONObject();
        JSONObject componentJSON = new JSONObject();
        JSONObject typeJSON = new JSONObject();
        JSONObject componentDataJSON = new JSONObject();
        JSONObject typeDataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("propertyName", "Prop_New");
        attributeJSON.put("propertyValue", "344209859");
        // fill data JSON
        body.put("type", "component_props");
        body.put("attributes", attributeJSON);
        // fill market and owner operator json
        componentDataJSON.put("id", "1");
        componentDataJSON.put("type", "components");
        typeDataJSON.put("id", "1");
        typeDataJSON.put("type", "component_properties_type");

        typeJSON.put("data", typeDataJSON);
        componentJSON.put("data", componentDataJSON);
        relationShipJSON.put("type", typeJSON);
        relationShipJSON.put("component", componentJSON);

        body.put("relationships", relationShipJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/component_props").then().assertThat().statusCode(is(400));
    }

    @Test
    void postComponentPropsWhenComponentIdIsMissing() {
        JSONObject body = new JSONObject();
        JSONObject relationShipJSON = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject componentJSON = new JSONObject();
        JSONObject typeJSON = new JSONObject();
        JSONObject componentDataJSON = new JSONObject();
        JSONObject typeDataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("propertyName", "Prop_New");
        attributeJSON.put("propertyValue", "344209859");
        // fill data JSON
        dataJSON.put("type", "component_props");
        dataJSON.put("attributes", attributeJSON);

        componentDataJSON.put("type", "components");
        typeDataJSON.put("id", "1");
        typeDataJSON.put("type", "component_properties_type");

        typeJSON.put("data", typeDataJSON);
        componentJSON.put("data", componentDataJSON);
        relationShipJSON.put("type", typeJSON);
        relationShipJSON.put("component", componentJSON);

        dataJSON.put("relationships", relationShipJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/component_props").then().assertThat().statusCode(is(500));
    }

    @ParameterizedTest
    @CsvSource({"17,404", "null,500", ",500"})
    void postComponentPropsWhenComponentIdIsEmpty(String id, int code) {
        JSONObject body = new JSONObject();
        JSONObject relationShipJSON = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject componentJSON = new JSONObject();
        JSONObject typeJSON = new JSONObject();
        JSONObject componentDataJSON = new JSONObject();
        JSONObject typeDataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("propertyName", "Prop_New");
        attributeJSON.put("propertyValue", "344209859");
        // fill data JSON
        dataJSON.put("type", "component_props");
        dataJSON.put("attributes", attributeJSON);
        // fill market and owner operator json
        componentDataJSON.put("id", id);
        componentDataJSON.put("type", "components");
        typeDataJSON.put("id", "1");
        typeDataJSON.put("type", "component_properties_type");

        typeJSON.put("data", typeDataJSON);
        componentJSON.put("data", componentDataJSON);
        relationShipJSON.put("type", typeJSON);
        relationShipJSON.put("component", componentJSON);

        dataJSON.put("relationships", relationShipJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/component_props").then().assertThat().statusCode(is(code));
    }

    @Test
    void postComponentPropsWhenPropertyIdIsEmpty() {
        JSONObject body = new JSONObject();
        JSONObject relationShipJSON = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject componentJSON = new JSONObject();
        JSONObject typeJSON = new JSONObject();
        JSONObject componentDataJSON = new JSONObject();
        JSONObject typeDataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("propertyName", "Prop_New");
        attributeJSON.put("propertyValue", "344209859");
        // fill data JSON
        dataJSON.put("type", "component_props");
        dataJSON.put("attributes", attributeJSON);
        componentDataJSON.put("id", "1");
        componentDataJSON.put("type", "components");
        typeDataJSON.put("id", "");
        typeDataJSON.put("type", "component_properties_type");

        typeJSON.put("data", typeDataJSON);
        componentJSON.put("data", componentDataJSON);
        relationShipJSON.put("type", typeJSON);
        relationShipJSON.put("component", componentJSON);

        dataJSON.put("relationships", relationShipJSON);
        body.put("data", dataJSON);
        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/component_props").then().assertThat().statusCode(is(500));
    }

    public void postComponentPropsWhenPropertyIdIsNull() {
        JSONObject body = new JSONObject();
        JSONObject relationShipJSON = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject componentJSON = new JSONObject();
        JSONObject typeJSON = new JSONObject();
        JSONObject componentDataJSON = new JSONObject();
        JSONObject typeDataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("propertyName", "Prop_New");
        attributeJSON.put("propertyValue", "344209859");
        // fill data JSON
        dataJSON.put("type", "component_props");
        dataJSON.put("attributes", attributeJSON);
        componentDataJSON.put("id", "1");
        componentDataJSON.put("type", "components");
        typeDataJSON.put("id", null);
        typeDataJSON.put("type", "component_properties_type");

        typeJSON.put("data", typeDataJSON);
        componentJSON.put("data", componentDataJSON);
        relationShipJSON.put("type", typeJSON);
        relationShipJSON.put("component", componentJSON);

        dataJSON.put("relationships", relationShipJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/component_props").then().assertThat().statusCode(is(500));
    }

    @Test
    void postComponentPropsWhenPropIdIsInvalid() {
        JSONObject body = new JSONObject();
        JSONObject relationShipJSON = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject componentJSON = new JSONObject();
        JSONObject typeJSON = new JSONObject();
        JSONObject componentDataJSON = new JSONObject();
        JSONObject typeDataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("propertyName", "Prop_New");
        attributeJSON.put("propertyValue", "344209859");
        // fill data JSON
        dataJSON.put("type", "component_props");
        dataJSON.put("attributes", attributeJSON);
        componentDataJSON.put("id", "1");
        componentDataJSON.put("type", "components");
        typeDataJSON.put("id", "23");
        typeDataJSON.put("type", "component_properties_type");

        typeJSON.put("data", typeDataJSON);
        componentJSON.put("data", componentDataJSON);
        relationShipJSON.put("type", typeJSON);
        relationShipJSON.put("component", componentJSON);

        dataJSON.put("relationships", relationShipJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/component_props").then().assertThat().statusCode(is(404));
    }

    @ParameterizedTest
    @CsvSource({"abc,500", "null,500", ",500"})
    void patchComponentPropsWhenComponentPropIdIsInvalid(String id, int code) {
        JSONObject body = new JSONObject();
        JSONObject relationShipJSON = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject componentJSON = new JSONObject();
        JSONObject typeJSON = new JSONObject();
        JSONObject componentDataJSON = new JSONObject();
        JSONObject typeDataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("propertyName", "Prop_New");
        attributeJSON.put("propertyValue", "344209859");
        // fill data JSON
        dataJSON.put("type", "component_props");
        dataJSON.put("attributes", attributeJSON);
        componentDataJSON.put("id", "1");
        componentDataJSON.put("type", "components");
        typeDataJSON.put("id", "1");
        typeDataJSON.put("type", "component_properties_type");

        typeJSON.put("data", typeDataJSON);
        componentJSON.put("data", componentDataJSON);
        relationShipJSON.put("type", typeJSON);
        relationShipJSON.put("component", componentJSON);

        dataJSON.put("relationships", relationShipJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .patch("/restaurant_assets/component_props/" + id).then().assertThat().statusCode(is(code));
    }

    @Test
    void patchComponentPropsWhenComponentPropIdIsMising() {
        JSONObject body = new JSONObject();
        JSONObject relationShipJSON = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject componentJSON = new JSONObject();
        JSONObject typeJSON = new JSONObject();
        JSONObject componentDataJSON = new JSONObject();
        JSONObject typeDataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("propertyName", "Prop_New");
        attributeJSON.put("propertyValue", "344209859");
        // fill data JSON
        dataJSON.put("type", "component_props");
        dataJSON.put("attributes", attributeJSON);
        componentDataJSON.put("id", "1");
        componentDataJSON.put("type", "components");
        typeDataJSON.put("id", "1");
        typeDataJSON.put("type", "component_properties_type");

        typeJSON.put("data", typeDataJSON);
        componentJSON.put("data", componentDataJSON);
        relationShipJSON.put("type", typeJSON);
        relationShipJSON.put("component", componentJSON);

        dataJSON.put("relationships", relationShipJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .patch("/restaurant_assets/component_props/").then().assertThat().statusCode(is(400));
    }


    @Test
    void patchComponentPropsWhenInputJSONIsEmptyString() {
        String body = "";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body).when()
                .patch("/restaurant_assets/component_props/null").then().assertThat().statusCode(is(500));
    }

    @Test
    void patchComponentPropsWhenInputJSONIsEmptyAndPropIdIsInvalid() {
        String body = "";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body).when()
                .patch("/restaurant_assets/component_props/abc").then().assertThat().statusCode(is(500));
    }
}
