package com.mcd.restaurant.repository;

import com.mcd.restaurant.dashboard.utils.StoreDashboardConstants;
import com.mcd.restaurant.deployment.controller.view.request.DeployComponentRequest;
import com.mcd.restaurant.deployment.service.DeploymentType;
import com.mcd.restaurant.model.Component;
import io.crnk.core.queryspec.FilterOperator;
import io.crnk.core.queryspec.FilterSpec;
import io.crnk.core.queryspec.PathSpec;
import io.crnk.core.queryspec.QuerySpec;
import io.crnk.core.resource.list.ResourceList;
import io.crnk.data.jpa.JpaEntityRepositoryBase;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@org.springframework.stereotype.Component
public class ComponentRepository extends JpaEntityRepositoryBase<Component, Integer> {
    public static final String RESTAURANT_ID="restaurant.id";
    public static final String REPORTED_VERSION="reportedVersion";
    public ComponentRepository() {
        super(Component.class);
    }

    /**
     * Function to get all docker components based on the applicable filters.
     *
     * @param deployComponent target deployComponent
     * @param restaurantIds
     * @return Applicable components list
     */
    public ResourceList<Component> getFilteredDockerComponents(DeployComponentRequest deployComponent, Set<Integer> restaurantIds) {
        QuerySpec componentSpecs = new QuerySpec(Component.class);
        componentSpecs.addFilter(PathSpec.of("name").filter(FilterOperator.EQ, deployComponent.getComponentName()));
        componentSpecs.addFilter(PathSpec.of("type.name").filter(FilterOperator.EQ, DeploymentType.docker.name()));
        Optional.ofNullable(deployComponent.getUpdateFromVersion()).ifPresent(v ->
                componentSpecs.addFilter(PathSpec.of(REPORTED_VERSION).filter(FilterOperator.EQ, v)));
        componentSpecs.addFilter(new FilterSpec(PathSpec.of(RESTAURANT_ID), FilterOperator.EQ, restaurantIds));
        return this.findAll(componentSpecs);
    }

    /**
     * Function to get all crd components based on the applicable filters.
     *
     * @param deployComponent target deployComponent
     * @param restaurantIds
     * @return Applicable components list
     */
    public ResourceList<Component> getFilteredCrdComponents(DeployComponentRequest deployComponent, Set<Integer> restaurantIds) {
        QuerySpec componentSpecs = new QuerySpec(Component.class);
        componentSpecs.addFilter(PathSpec.of("name").filter(FilterOperator.EQ, deployComponent.getComponentName()));
        componentSpecs.addFilter(PathSpec.of("type.name").filter(FilterOperator.EQ, DeploymentType.crd.name()));
        Optional.ofNullable(deployComponent.getUpdateFromVersion()).ifPresent(v ->
                componentSpecs.addFilter(PathSpec.of(REPORTED_VERSION).filter(FilterOperator.EQ, v)));
        componentSpecs.addFilter(new FilterSpec(PathSpec.of(RESTAURANT_ID), FilterOperator.EQ, restaurantIds));
        return this.findAll(componentSpecs);
    }

    /**
     * Function to get all firmware components based on the applicable filters.
     *
     * @param deployComponent target deployComponent
     * @return Applicable components list
     */
    public ResourceList<Component> getFilteredFirmwareComponents(DeployComponentRequest deployComponent, Set<Integer> restaurantIds) {
        QuerySpec componentSpecs = new QuerySpec(Component.class);
        componentSpecs.addFilter(PathSpec.of("vendor.name").filter(FilterOperator.EQ, deployComponent.getVendor()));
        componentSpecs.addFilter(PathSpec.of("modelNumber").filter(FilterOperator.EQ, deployComponent.getModelNumber()));
        Optional.ofNullable(deployComponent.getUpdateFromVersion()).ifPresent(v ->
                componentSpecs.addFilter(PathSpec.of(REPORTED_VERSION).filter(FilterOperator.EQ, v)));
        componentSpecs.addFilter(new FilterSpec(PathSpec.of(RESTAURANT_ID), FilterOperator.EQ, restaurantIds));
        return this.findAll(componentSpecs);
    }

    public QuerySpec prepareQuerySpecFilterForHardware(Integer restaurantId) {
        QuerySpec querySpec = new QuerySpec(com.mcd.restaurant.model.Component.class);
        FilterSpec filterDockerSpec = new FilterSpec(PathSpec.of(StoreDashboardConstants.TYPE_NAME), FilterOperator.EQ, "DEVICE");
        FilterSpec filterStoreSpec = new FilterSpec(PathSpec.of(StoreDashboardConstants.RESTAURANT_ID), FilterOperator.EQ, restaurantId);
        FilterSpec filterFinalType = FilterSpec.and(Stream.of(filterDockerSpec, filterStoreSpec).collect(Collectors.toList()));
        querySpec.setFilters(Stream.of(filterFinalType).collect(Collectors.toList()));
        return querySpec;
    }

    public QuerySpec prepareQuerySpecFilterForPlatformCount(Integer restaurantId) {
        QuerySpec querySpec = new QuerySpec(com.mcd.restaurant.model.Component.class);
        FilterSpec filterDockerSpec = new FilterSpec(PathSpec.of(StoreDashboardConstants.TYPE_NAME), FilterOperator.EQ, "DOCKER");
        FilterSpec filterGreenGrassSpec = new FilterSpec(PathSpec.of(StoreDashboardConstants.TYPE_NAME), FilterOperator.EQ, "GREENGRASS");
        FilterSpec filterSpecType = FilterSpec.or(Stream.of(filterDockerSpec, filterGreenGrassSpec).collect(Collectors.toList()));
        FilterSpec filterStoreSpec = new FilterSpec(PathSpec.of(StoreDashboardConstants.RESTAURANT_ID), FilterOperator.EQ, restaurantId);
        FilterSpec filterFinalType = FilterSpec.and(Stream.of(filterSpecType, filterStoreSpec).collect(Collectors.toList()));
        querySpec.setFilters(Stream.of(filterFinalType).collect(Collectors.toList()));
        return querySpec;
    }

    public List<Component> getAllComponentByRestaurant(int restaurantId, List<String> applicationName)
    {
        QuerySpec querySpec = new QuerySpec(Component.class);
        FilterSpec filterRestaurantId = new FilterSpec(PathSpec.of(StoreDashboardConstants.RESTAURANT_ID), FilterOperator.EQ, restaurantId);
        FilterSpec filterType = new FilterSpec(PathSpec.of(StoreDashboardConstants.TYPE_NAME), FilterOperator.EQ, DeploymentType.docker.name());
        FilterSpec filterApplication = new FilterSpec(PathSpec.of("name"), FilterOperator.EQ, applicationName);
        FilterSpec filterFinalType = FilterSpec.and(Stream.of(filterRestaurantId, filterType, filterApplication).collect(Collectors.toList()));
        querySpec.setFilters(Stream.of(filterFinalType).collect(Collectors.toList()));
        return this.findAll(querySpec);
    }
}
