package com.mcd.restaurant.reip.services;

import com.mcd.restaurant.common.MapperUtils;
import com.mcd.restaurant.model.*;
import com.mcd.restaurant.reip.error.RestaurantEdgeIOTBadRequestException;
import com.mcd.restaurant.reip.error.RestaurantEdgeIOTException;
import com.mcd.restaurant.reip.view.request.DeviceAdapterOnboardingRequestDTO;
import com.mcd.restaurant.reip.view.request.VendorOnboardingRequestDTO;
import com.mcd.restaurant.reip.view.response.DeviceAdapterOnboardingResponseDTO;
import com.mcd.restaurant.reip.view.response.VendorOnboardingResponseDTO;
import com.mcd.restaurant.repository.*;
import io.crnk.core.queryspec.FilterOperator;
import io.crnk.core.queryspec.FilterSpec;
import io.crnk.core.queryspec.PathSpec;
import io.crnk.core.queryspec.QuerySpec;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Service
public class RestaurantEdgeIOTService {
    @Autowired
    private RestaurantRepository restaurantRepository;

    @Autowired
    private ReipVendorRepository reipVendorRepository;
    @Autowired
    private ReipDeviceTypesRepository reipDeviceTypesRepository;
    @Autowired
    private ReipDeviceTypePropsRepository reipDeviceTypePropsRepository;
    @Autowired
    private ReipDeviceInstancesRepository reipDeviceInstancesRepository;
    @Autowired
    private ReipDeviceInstancePropsRepository deviceInstancePropsRepository;
    @Autowired
    private ReipDeviceAdapterMappingRepository reipDeviceAdapterMappingRepository;
    @Autowired
    private ReipPropListRepository reipPropListRepository;
    @Autowired
    private MapperUtils mapperUtils;
    @Autowired
    private RestaurantEdgeIOTValidator restaurantEdgeIOTValidator;

    @Transactional
    public List<DeviceAdapterOnboardingResponseDTO> onboardDevices(DeviceAdapterOnboardingRequestDTO requestDTO) {
        try {
            restaurantEdgeIOTValidator.deviceAdapterOnboardingRequestDTOValidator(requestDTO);
            List<DeviceAdapterOnboardingResponseDTO> responseDTO = new ArrayList<>();
            Restaurant restaurant = restaurantRepository.findOne(requestDTO.getStoreId(), new QuerySpec(Restaurant.class));
            requestDTO.getDevices().forEach(device -> {
                ReipDeviceTypes deviceType = reipDeviceTypesRepository.findOne(device.getDeviceTypeId(), new QuerySpec(ReipDeviceTypes.class));
                List<ReipDeviceTypeProps> reipDeviceTypeProps = reipDeviceTypePropsRepository.findAll(prepareQuerySpecFilterForDeviceTypePropsByDeviceType(deviceType.getId()));
                ReipDeviceInstances deviceInstance = ReipDeviceInstances.builder()
                        .name(device.getName())
                        .reipDeviceTypes(deviceType)
                        .createdBy(device.getCreatedBy())
                        .modelNumber(device.getModelNumber())
                        .status(device.getStatus())
                        .serialNumber(device.getSerialNumber())
                        .firmwareVersion(device.getFirmwareVersion())
                        .restaurant(restaurant)
                        .build();
                ReipDeviceInstances finalDeviceInstance = reipDeviceInstancesRepository.save(deviceInstance);

                responseDTO.add(mapperUtils.map(finalDeviceInstance, DeviceAdapterOnboardingResponseDTO.class));
                if (deviceType.getIsAdapter()) {
                    reipDeviceTypeProps.forEach(prop -> {
                        ReipDeviceAdapterMapping reipDeviceAdapterMapping = ReipDeviceAdapterMapping.builder().createdBy(device.getCreatedBy())
                                .reipAdapterInstance(finalDeviceInstance)
                                .reipDeviceTypeProps(prop)
                                .build();
                        reipDeviceAdapterMappingRepository.save(reipDeviceAdapterMapping);
                    });
                } else {
                    if(!deviceType.getIsMQTTEnabled() && device.getConnectorId() != 0) {
                        List<ReipDeviceAdapterMapping> deviceTypeProps = reipDeviceAdapterMappingRepository.findAll(prepareQuerySpecFilterForDeviceAdapterMappingByDeviceInstanceAndDeviceTypePropId(device.getAdapterId(), device.getConnectorId()));
                        deviceTypeProps.forEach(prop -> {
                            prop.setReipDeviceInstance(finalDeviceInstance);
                            prop.setUpdatedBy(device.getCreatedBy());
                            reipDeviceAdapterMappingRepository.save(prop);
                        });
                    }
                }

                device.getProps().forEach(prop -> {
                    ReipDeviceInstanceProps deviceInstanceProp = ReipDeviceInstanceProps.builder()
                            .propName(prop.getPropName())
                            .propValue(prop.getPropValue())
                            .reipDeviceInstances(finalDeviceInstance)
                            .build();
                    deviceInstancePropsRepository.save(deviceInstanceProp);
                });
            });
            return responseDTO;

        } catch (RestaurantEdgeIOTBadRequestException ex) {
            throw ex;
        } catch (Exception e) {
            throw new RestaurantEdgeIOTException(StringUtils.isBlank(e.getMessage()) ? "Some Exception Occured" : e.getMessage());
        }
    }

    @Transactional
    public List<VendorOnboardingResponseDTO> onboardVendors(VendorOnboardingRequestDTO requestDTO) {
        try {
            restaurantEdgeIOTValidator.vendorOnboardingRequestDTOValidator(requestDTO);
            List<VendorOnboardingResponseDTO> responseDTO = new ArrayList<>();
            ReipVendor reipVendor = ReipVendor.builder()
                    .name(requestDTO.getVendorName())
                    .contact_name(requestDTO.getVendorContact().getName())
                    .email_id(requestDTO.getVendorContact().getEmail())
                    .phone_number(requestDTO.getVendorContact().getPhone())
                    .contactNotes(requestDTO.getVendorContact().getContactNotes())
                    .vendorNotes(requestDTO.getVendorNotes())
                    .createdBy(requestDTO.getCreatedBy())

                    .build();
            ReipVendor finalReipVendor = reipVendorRepository.save(reipVendor);
            responseDTO.add(mapperUtils.map(finalReipVendor, VendorOnboardingResponseDTO.class));
            requestDTO.getDeviceTypes().forEach(deviceType -> {
                if(deviceType.getCategory().equalsIgnoreCase("device")){
                    ReipDeviceTypes reipDeviceTypes = ReipDeviceTypes.builder()
                            .isAdapter(false)
                            .typeName(deviceType.getDeviceType())
                            .isMQTTEnabled(deviceType.isMQTTEnabled())
                            .isOTAEnabled(deviceType.isOTAEnabled())
                            .createdBy(requestDTO.getCreatedBy())
                            .reipVendor(finalReipVendor)
                            .modelNumber(deviceType.getModelNumber())
                            .firmwareVersion(deviceType.getFirmwareVersion())
                            .build();
                    ReipDeviceTypes finalReipDeviceTypes = reipDeviceTypesRepository.save(reipDeviceTypes);
                    deviceType.getAttributes().forEach(attribute -> {
                        ReipDeviceTypeProps reipDeviceTypeProps = ReipDeviceTypeProps.builder()
                                .name(attribute)
                                .isConnector(false)
                                .reipDeviceTypes(finalReipDeviceTypes)
                                .build();
                        reipDeviceTypePropsRepository.save(reipDeviceTypeProps);
                    });
                } else {
                    ReipDeviceTypes reipDeviceTypes = ReipDeviceTypes.builder()
                            .isAdapter(true)
                            .typeName(deviceType.getDeviceType())
                            .createdBy(requestDTO.getCreatedBy())
                            .reipVendor(finalReipVendor)
                            .modelNumber(deviceType.getModelNumber())
                            .firmwareVersion(deviceType.getFirmwareVersion())
                            .build();
                    ReipDeviceTypes finalReipDeviceTypes = reipDeviceTypesRepository.save(reipDeviceTypes);
                    deviceType.getAttributes().forEach(attribute -> {
                        ReipDeviceTypeProps reipDeviceTypeProps = ReipDeviceTypeProps.builder()
                                .name(attribute)
                                .isConnector(false)
                                .reipDeviceTypes(finalReipDeviceTypes)
                                .build();
                        reipDeviceTypePropsRepository.save(reipDeviceTypeProps);
                    });
                    deviceType.getConnectors().forEach(connector -> {
                        ReipDeviceTypeProps reipDeviceTypeProps = ReipDeviceTypeProps.builder()
                                .name(connector)
                                .isConnector(true)
                                .reipDeviceTypes(finalReipDeviceTypes)
                                .build();
                        reipDeviceTypePropsRepository.save(reipDeviceTypeProps);
                    });
                }
            });
            return responseDTO;

        } catch (RestaurantEdgeIOTBadRequestException ex) {
            throw ex;
        } catch (Exception e) {
            throw new RestaurantEdgeIOTException(StringUtils.isBlank(e.getMessage()) ? "Some Exception Occured" : e.getMessage());
        }
    }

//    @Transactional
//    public List<VendorOnboardingResponseDTO> editVendor(VendorOnboardingRequestDTO requestDTO) {
//        try {
//            return responseDTO;
//        } catch (RestaurantEdgeIOTBadRequestException ex) {
//            throw ex;
//        } catch (Exception e) {
//            throw new RestaurantEdgeIOTException(StringUtils.isBlank(e.getMessage()) ? "Some Exception Occured" : e.getMessage());
//        }
//    }
    public QuerySpec prepareQuerySpecFilterForDeviceTypePropsByDeviceType(Integer deviceTypeId) {
        QuerySpec querySpec = new QuerySpec(ReipDeviceTypeProps.class);
        FilterSpec filterDeviceTypeSpec = new FilterSpec(PathSpec.of("reipDeviceTypes.id"), FilterOperator.EQ, deviceTypeId);
        FilterSpec filterDeviceTypePropSpec = new FilterSpec(PathSpec.of("isConnector"), FilterOperator.EQ, true);
        querySpec.setFilters(Stream.of(filterDeviceTypeSpec,filterDeviceTypePropSpec).collect(Collectors.toList()));
        return querySpec;
    }

    public QuerySpec prepareQuerySpecFilterForDeviceAdapterMappingByDeviceInstanceAndDeviceTypePropId(Integer deviceInstanceId, Integer deviceTypePropId) {
        QuerySpec querySpec = new QuerySpec(ReipDeviceAdapterMapping.class);
        FilterSpec filterDeviceInstanceSpec = new FilterSpec(PathSpec.of("reipAdapterInstance.id"), FilterOperator.EQ, deviceInstanceId);
        FilterSpec filterDeviceTypePropSpec = new FilterSpec(PathSpec.of("reipDeviceTypeProps.id"), FilterOperator.EQ, deviceTypePropId);
        FilterSpec filterFinalType = FilterSpec.and(Stream.of(filterDeviceInstanceSpec, filterDeviceTypePropSpec).collect(Collectors.toList()));
        querySpec.setFilters(Stream.of(filterFinalType).collect(Collectors.toList()));
        return querySpec;
    }
}
