package com.mcd.restaurant.deployment.service;


import com.amazonaws.services.lambda.AWSLambdaAsync;
import com.amazonaws.services.lambda.model.InvokeRequest;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.mcd.restaurant.common.Constants;
import com.mcd.restaurant.common.MapperUtils;
import com.mcd.restaurant.bundledrelease.controller.view.response.SQSApplicationResponse;
import com.mcd.restaurant.deployment.controller.view.beans.DeploymentStatisticsSegment;
import com.mcd.restaurant.deployment.controller.view.request.*;
import com.mcd.restaurant.deployment.controller.view.response.BulkInsertComponentResponseDTO;
import com.mcd.restaurant.deployment.controller.view.response.ComponentDetailsResponseDTO;
import com.mcd.restaurant.deployment.controller.view.response.DeployGraphResponseDTO;
import com.mcd.restaurant.deployment.controller.view.response.DeploymentGroupDetailsResponseDTO;
import com.mcd.restaurant.deployment.controller.view.response.UpdateComponentResponseDTO;
import com.mcd.restaurant.deployment.error.DeploymentBadRequestException;
import com.mcd.restaurant.deployment.error.DeploymentError;
import com.mcd.restaurant.model.*;
import com.mcd.restaurant.repository.*;
import com.mcd.restaurant.repository.bulk.ComponentBulkCRUDRepository;

import io.crnk.core.queryspec.QuerySpec;
import io.crnk.core.queryspec.FilterSpec;
import io.crnk.core.queryspec.PathSpec;
import io.crnk.core.queryspec.FilterOperator;
import io.crnk.core.queryspec.SortSpec;
import io.crnk.core.queryspec.Direction;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.nio.ByteBuffer;
import java.sql.Timestamp;
import java.time.ZoneOffset;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Component
@Slf4j
public class DeploymentUtils {


    @Autowired
    private DeploymentGroupRepository deploymentGroupRepository;
    @Autowired
    private ComponentTypeRepository componentTypeRepository;
    @Autowired
    private DeploymentTagRepository deploymentTagRepository;
    @Autowired
    private ComponentBulkCRUDRepository componentBulkCRUDRepository;
    @Autowired
    private MapperUtils mapperUtils;
    @Autowired
    private ProductApplicationRepository productApplicationRepository;
    @Autowired
    private ComponentRepository componentRepository;
    @Autowired
    private RestaurantRepository restaurantRepository;
    @Autowired
    private MarketRepository marketRepository;
    @Autowired
    private PrePostDeploymentRepository prePostDeploymentRepository;
    @Autowired
    private AWSLambdaAsync awsLambdaAsync;
    @Value("${aws.lambda.url}")
    String awsPrePostLambdaUrl;
    @Value("${aws.sqs.queue.account.id}")
    String accountId;


    public QuerySpec prepareQuerySpecFilterForArchival(Long archiveBeforeTime, Long limit, Integer offset) {
        QuerySpec querySpec = new QuerySpec(DeploymentGroup.class);
        FilterSpec filterSpecDate = new FilterSpec(PathSpec.of(Constants.CREATED), FilterOperator.LE, new Timestamp(archiveBeforeTime));
        FilterSpec filterStatus = new FilterSpec(PathSpec.of(Constants.DEPLOYMENT_HISTORY_STATUS), FilterOperator.NEQ, DeploymentStatus.Scheduled.value());
        FilterSpec finalFilter = FilterSpec.and(Stream.of(filterStatus, filterSpecDate).collect(Collectors.toList()));
        SortSpec sortSpec = PathSpec.of("id").sort(Direction.DESC);
        querySpec.setFilters(Stream.of(finalFilter).collect(Collectors.toList()));
        querySpec.setSort(Stream.of(sortSpec).collect(Collectors.toList()));
        querySpec.setOffset(offset);
        querySpec.setLimit(limit);
        return querySpec;
    }

    public QuerySpec prepareQuerySpecFilterForSuggestion(Class<?> classtype, String searchOn, String searchParam) {
        FilterSpec filterSpecVersion = new FilterSpec(PathSpec.of(searchOn), FilterOperator.LIKE, "%" + searchParam + "%");
        QuerySpec querySpec = new QuerySpec(classtype);
        querySpec.setFilters(Stream.of(filterSpecVersion).collect(Collectors.toList()));
        return querySpec;
    }


    @Transactional
    public DeployGraphResponseDTO analyseDeploymentResultsDay(Long startTime, Long endTime) {
        List<DeploymentGroup> progressDeploymentGroup = deploymentGroupRepository.findAll(deploymentGroupRepository.prepareQuerySpecFilterForGraph(DeploymentStatus.InProgress.value(), startTime, endTime));
        List<DeploymentGroup> failedDeploymentGroup = deploymentGroupRepository.findAll(deploymentGroupRepository.prepareQuerySpecFilterForGraph(DeploymentStatus.Failed.value(), startTime, endTime));
        List<DeploymentGroup> cancelledDeploymentGroup = deploymentGroupRepository.findAll(deploymentGroupRepository.prepareQuerySpecFilterForGraph(DeploymentStatus.Cancelled.value(), startTime, endTime));
        List<DeploymentGroup> completeDeploymentGroup = deploymentGroupRepository.findAll(deploymentGroupRepository.prepareQuerySpecFilterForGraph(DeploymentStatus.Completed.value(), startTime, endTime));
        completeDeploymentGroup.removeIf(group -> failedDeploymentGroup.contains(group) || progressDeploymentGroup.contains(group) || cancelledDeploymentGroup.contains(group));
        Map<Integer, Integer> completedSegmentmap = new HashMap<>();
        Map<Integer, Integer> failedSegmentmap = new HashMap<>();
        int totalCount = 0;
        for (int i = 0; i < DeployGraphSegment.DAY.value(); i++) {
            final int hourCounter = i;
            List<DeploymentGroup> completeDeploymentPerHour = completeDeploymentGroup.stream().filter(j -> hourCounter == j.getCreated().toInstant().atOffset(ZoneOffset.UTC).toLocalDateTime().getHour()).collect(Collectors.toList());
            List<DeploymentGroup> failedDeploymentPerHour = failedDeploymentGroup.stream().filter(j -> hourCounter == j.getCreated().toInstant().atOffset(ZoneOffset.UTC).toLocalDateTime().getHour()).collect(Collectors.toList());
            completedSegmentmap.put(i, completeDeploymentPerHour.size());
            failedSegmentmap.put(i, failedDeploymentPerHour.size());
            totalCount += completeDeploymentPerHour.size() + failedDeploymentPerHour.size();
            completeDeploymentGroup.removeAll(completeDeploymentPerHour);
            failedDeploymentGroup.removeAll(failedDeploymentPerHour);
        }
        return DeployGraphResponseDTO.builder().failedDeployments(failedSegmentmap).completedDeployments(completedSegmentmap).totalDeployment(totalCount).build();
    }

    @Transactional
    public DeploymentStatisticsSegment analyseDeploymentResults(Long startTime, Long endTime) {
        List<DeploymentGroup> progressDeploymentGroup = deploymentGroupRepository.findAll(deploymentGroupRepository.prepareQuerySpecFilterForGraph(DeploymentStatus.InProgress.value(), startTime, endTime));
        ArrayList<Integer> progressRecordMap = (ArrayList<Integer>) progressDeploymentGroup.stream().map(DeploymentGroup::getId).distinct().collect(Collectors.toList());
        progressDeploymentGroup.clear();

        List<DeploymentGroup> failedDeploymentGroup = deploymentGroupRepository.findAll(deploymentGroupRepository.prepareQuerySpecFilterForGraph(DeploymentStatus.Failed.value(), startTime, endTime));
        ArrayList<Integer> failedRecordMap = (ArrayList<Integer>) failedDeploymentGroup.stream().map(DeploymentGroup::getId).distinct().collect(Collectors.toList());
        failedDeploymentGroup.clear();
        List<DeploymentGroup> cancelledDeploymentGroup = deploymentGroupRepository.findAll(deploymentGroupRepository.prepareQuerySpecFilterForGraph(DeploymentStatus.Cancelled.value(), startTime, endTime));
        ArrayList<Integer> cancelledRecordMap = (ArrayList<Integer>) cancelledDeploymentGroup.stream().map(DeploymentGroup::getId).distinct().collect(Collectors.toList());
        failedDeploymentGroup.clear();
        List<DeploymentGroup> completeDeploymentGroup = deploymentGroupRepository.findAll(deploymentGroupRepository.prepareQuerySpecFilterForGraph(DeploymentStatus.Completed.value(), startTime, endTime));
        ArrayList<Integer> completedRecordMap = (ArrayList<Integer>) completeDeploymentGroup.stream().map(DeploymentGroup::getId).distinct().collect(Collectors.toList());
        completeDeploymentGroup.clear();

        completedRecordMap.removeIf(i -> failedRecordMap.contains(i) || progressRecordMap.contains(i) || cancelledRecordMap.contains(i));
        return DeploymentStatisticsSegment.builder().completed(completedRecordMap.size()).failed(failedRecordMap.size()).build();
    }

    public void deploymentStatusUpdateRequestValidator(UpdateDeploymentStatusRequestDTO requestDTO) {
        if (StringUtils.isBlank(requestDTO.getMarketName())) {
            throw new DeploymentBadRequestException(Constants.MARKET_NAME_IS_REQUIRED);
        }
        if (StringUtils.isBlank(requestDTO.getApplicationName())) {
            throw new DeploymentBadRequestException(Constants.APPLICATION_NAME_IS_REQUIRED);
        }
        if (StringUtils.isBlank(requestDTO.getApplicationVersion())) {
            throw new DeploymentBadRequestException(Constants.APPLICATION_VERSION_IS_REQUIRED);
        }
        if (StringUtils.isBlank(requestDTO.getRestaurantNo())) {
            throw new DeploymentBadRequestException(Constants.RESTAURANT_NO_IS_REQUIRED);
        }
        if (requestDTO.getDeploymentGroupId() == null) {
            throw new DeploymentBadRequestException(Constants.DEPLOYMENT_GROUP_ID_IS_REQUIRED);
        }
        if (requestDTO.getStatus() == null) {
            throw new DeploymentBadRequestException(Constants.STATUS_IS_REQUIRED);
        }
    }

    public com.mcd.restaurant.model.Component handleMissingComponent(UpdateComponentRequestDTO updateComponentRequestDTO, UpdateComponentResponseDTO updateComponentResponseDTO, Restaurant restaurant) {
        com.mcd.restaurant.model.Component component;
        ComponentType type = fetchComponentType("docker");
        component = new com.mcd.restaurant.model.Component();
        setNewComponentFields(updateComponentRequestDTO, restaurant, component, type);
        updateComponentResponseDTO.setIsNewComponentInserted(true);
        return component;
    }

    public com.mcd.restaurant.model.Component handleMissingComponentForDeployment(UpdateComponentRequestDTO updateComponentRequestDTO, Restaurant restaurant, ComponentType type) {
        com.mcd.restaurant.model.Component component = new com.mcd.restaurant.model.Component();
        setNewComponentFields(updateComponentRequestDTO, restaurant, component, type);
        return component;
    }

    private void setNewComponentFields(UpdateComponentRequestDTO updateComponentRequestDTO, Restaurant restaurant, com.mcd.restaurant.model.Component component, ComponentType type) {
        component.setRestaurant(restaurant);
        component.setName(updateComponentRequestDTO.getApplicationName());
        component.setReportedVersion(updateComponentRequestDTO.getApplicationVersion());
        component.setType(type);
        component.setInstallDate(new Timestamp(System.currentTimeMillis()));
    }

    public ComponentType fetchComponentType(String name) {
        List<ComponentType> types = componentTypeRepository.getComponentTypeByName(name);
        if (!CollectionUtils.isEmpty(types)) {
            return types.get(0);
        } else {
            throw new DeploymentError("ComponentType not found");
        }
    }

    @Async
    @Transactional
    public void asyncDeletePOstArchival(PostArchivalDeleteRequestDTO requestDTO) {
        requestDTO.getDeploymentGroupId().stream().forEach(i -> deploymentGroupRepository.delete(i));
        if (!CollectionUtils.isEmpty(requestDTO.getDeploymentTagId())) {
            requestDTO.getDeploymentTagId().stream().forEach(i -> deploymentTagRepository.delete(i));
        }
    }

    public void bundleDeploymentRequestValidator(BundleDeployRequest request) {
        if (request.getDeploymentGroupId() == null) {
            throw new DeploymentBadRequestException(Constants.DEPLOYMENT_GROUP_ID_IS_REQUIRED);
        }
    }

    public void validateRequest(BulkInsertComponentRequestDTO componentDTO) {
        if (StringUtils.isBlank(componentDTO.getMarketName()) || StringUtils.isBlank(componentDTO.getStoreName()) || CollectionUtils.isEmpty(componentDTO.getComponents())) {
            throw new DeploymentBadRequestException(
                    Constants.APPLICATION_NAME_APPLICATION_VERSION_RESTAURANT_NAME_AND_MARKET_NAME_ARE_MANDATORY_FOR_THIS_OPERATION);
        } else {
            componentDTO.getComponents().stream()
                    .forEach(comp ->
                    {
                        if (StringUtils.isBlank(comp.getApplicationName()) || StringUtils.isBlank(comp.getApplicationVersion())) {
                            throw new DeploymentBadRequestException(
                                    Constants.APPLICATION_NAME_APPLICATION_VERSION_RESTAURANT_NAME_AND_MARKET_NAME_ARE_MANDATORY_FOR_THIS_OPERATION);
                        }
                    });
        }
    }

    public List<com.mcd.restaurant.model.Component> bulkInsertComponentsData(List<ComponentDetailRequestDTO> components, Restaurant restaurant, List<com.mcd.restaurant.model.Component> existingComponents) {
        List<com.mcd.restaurant.model.Component> componentData;
        ComponentType type = fetchComponentType("docker");
        componentData = new ArrayList<>();
        insertComponent(components, restaurant, componentData, type, existingComponents);
        return componentData;
    }

    public void insertComponent(List<ComponentDetailRequestDTO> components, Restaurant restaurant, List<com.mcd.restaurant.model.Component> componentData, ComponentType type, List<com.mcd.restaurant.model.Component> existingComponents) {
        componentData.addAll(components.stream()
                .map(application -> {
                    com.mcd.restaurant.model.Component component = new com.mcd.restaurant.model.Component();
                    component.setInstallDate(application.getInstallDate() == null ? new Timestamp(System.currentTimeMillis()) : application.getInstallDate());
                    component.setName(application.getApplicationName());
                    component.setReportedVersion(application.getApplicationVersion());
                    component.setType(type);
                    component.setRestaurant(restaurant);
                    return component;
                })
                .collect(Collectors.toList()));
        componentData.addAll(existingComponents);
        componentBulkCRUDRepository.saveAll(componentData);
    }

    public BulkInsertComponentResponseDTO bulkInsertComponentResponse(List<com.mcd.restaurant.model.Component> prepareResponse, Restaurant restaurant, String marketName) {
        List<ComponentDetailsResponseDTO> deploymentGroupArchivalResponseDTOList = mapperUtils.map(prepareResponse,
                ComponentDetailsResponseDTO.class);

        return BulkInsertComponentResponseDTO.builder()
                .restaurantId(restaurant.getId())
                .restaurantName(restaurant.getName())
                .marketName(marketName)
                .components(deploymentGroupArchivalResponseDTOList)
                .build();
    }

    public List<String> fetchAllProductApplications() {
        return productApplicationRepository.getAllApplication()
                .stream()
                .map(ProductApplications::getApplicationName)
                .distinct()
                .collect(Collectors.toList());
    }

    public List<String> fetchAllVersionForApplication(String applicationName) {
        return productApplicationRepository.getAllApplicationByName(applicationName)
                .stream()
                .map(ProductApplications::getApplicationVersion)
                .distinct()
                .collect(Collectors.toList());
    }

    public List<ProductApplications> getUniqueProductApplicationsInDeployment(List<DeploymentHistory> deploymentHistoryList) {
        List<String> applicationName = new ArrayList<>();
        List<ProductApplications> applicationsList = new ArrayList<>();
        List<FilterSpec> filterSpecList = new ArrayList<>();
        QuerySpec querySpec = new QuerySpec(ProductApplications.class);
        deploymentHistoryList.forEach(history -> {
            FilterSpec filterNameSpec = new FilterSpec(PathSpec.of("applicationName"), FilterOperator.EQ, history.getApplicationName());
            FilterSpec filterVersionSpec = new FilterSpec(PathSpec.of("applicationVersion"), FilterOperator.EQ, history.getApplicationVersion());
            filterSpecList.add(FilterSpec.and(Stream.of(filterNameSpec, filterVersionSpec).collect(Collectors.toList())));
        });
        SortSpec sortSpec = PathSpec.of("updated").sort(Direction.DESC);
        querySpec.addFilter(FilterSpec.or(filterSpecList));
        querySpec.setSort(Stream.of(sortSpec).collect(Collectors.toList()));
        productApplicationRepository.findAll(querySpec).forEach(app -> {
            if (!applicationName.contains(app.getApplicationName())) {
                applicationsList.add(app);
                applicationName.add(app.getApplicationName());
            }
        });
        return applicationsList;
    }

    public void validateInputForFetchingApplicationDetails(SuggestionType suggestionType, String applicationName) {
        if (!suggestionType.equals(SuggestionType.APPLICATION_NAME) && !suggestionType.equals(SuggestionType.APPLICATION_VERSION)) {
            throw new DeploymentBadRequestException(String.format(Constants.INVALID_OPERATION_REQUESTED, suggestionType));
        }
        if (suggestionType.equals(SuggestionType.APPLICATION_VERSION) && StringUtils.isBlank(applicationName)) {
            throw new DeploymentBadRequestException(String.format(Constants.APPLICATION_NAME_IS_REQUIRED_FOR_THE_REQUESTED_OPERATION, suggestionType));
        }
    }

    public Map<String, List<SQSApplicationResponse>> updateApplicationDetailsGroupedByProduct(List<ProductApplications> applicationsList, List<DeploymentHistory> deploymentHistoryListFull) {
        Map<String, List<SQSApplicationResponse>> applicationNameSpace = applicationsList.stream().collect(Collectors.groupingBy(app -> app.getProductVersions().getProducts().getName(), Collectors.collectingAndThen(Collectors.toList(), list ->
                list.stream().map(i -> SQSApplicationResponse.builder().applicationName(i.getApplicationName()).applicationVersion(i.getApplicationVersion()).applicationUrl(i.getApplicationUrl()).build()).collect(Collectors.toList())
        )));
        Optional<DeploymentHistory> imageHistory = deploymentHistoryListFull
                .stream()
                .filter(history -> history.getApplicationName().equalsIgnoreCase(Constants.IMAGE_CACHE))
                .findFirst();
        if (imageHistory.isPresent()) {
            DeploymentHistory history = imageHistory.get();
            applicationNameSpace.put(Constants.IMAGE_CACHE, Stream.of(SQSApplicationResponse
                    .builder()
                    .applicationName(history.getApplicationName())
                    .applicationVersion(history.getApplicationVersion())
                    .build())
                    .collect(Collectors.toList()));
        }
        return applicationNameSpace;
    }

    public DeploymentGroupDetailsResponseDTO buildDeploymentDetailResponseDTO(List<DeploymentHistory> deploymentHistoryList, List<RestaurantView> restaurantViewList, Map<String, List<SQSApplicationResponse>> applicationNameSpace) {
        Map<String, String> products;
        DeploymentGroupDetailsResponseDTO responseDTO = DeploymentGroupDetailsResponseDTO.builder().isProductDeployment(false).build();
        Map<String, Set<String>> storeList = restaurantViewList.stream()
                .collect(Collectors.groupingBy(RestaurantView::getRegionName, Collectors.mapping(RestaurantView::getRestaurantNo, Collectors.toSet())));
        if (!CollectionUtils.isEmpty(deploymentHistoryList) && deploymentHistoryList.get(0).getProductVersions() != null) {
            responseDTO.setIsProductDeployment(true);
        }
        if (responseDTO.getIsProductDeployment().equals(Boolean.TRUE)) {
            List<ProductVersions> versionsList = deploymentHistoryList.stream().map(DeploymentHistory::getProductVersions).distinct().collect(Collectors.toList());
            products = versionsList.stream().collect(Collectors.toMap(i -> i.getProducts().getName(), ProductVersions::getVersionName));
            responseDTO.setProducts(products);
        }
        responseDTO.setMarketName(restaurantViewList.get(0).getMarketName());
        responseDTO.setApplications(applicationNameSpace);
        responseDTO.setStoreList(storeList);
        return responseDTO;
    }

    public List<com.mcd.restaurant.model.Component> fetchExistingComponents(List<String> components, Restaurant restaurant) {
        components.forEach(component -> {
            if (Collections.frequency(components, component) > 1) {
                throw new DeploymentBadRequestException(Constants.UNIQUE_APPLICATION_NAME_IS_MANDATORY_FOR_THIS_OPERATION);
            }
        });
        return componentRepository.getAllComponentByRestaurant(restaurant.getId(), components);
    }


    public Restaurant validateMarketAndRestaurant(String marketName, String storeName) {
        List<Market> markets = marketRepository.findAll(marketRepository.prepareQuerySpecFilterForFilteringMarketByName(marketName));
        if (CollectionUtils.isEmpty(markets)) {
            throw new DeploymentBadRequestException(Constants.NO_MARKET_PRESENT_WITH_THE_NAME + marketName);
        }
        Integer marketId = markets.get(0).getId();

        List<Restaurant> restaurants = restaurantRepository.findAll(restaurantRepository.prepareQuerySpecFilterForRestaurantByRestaurantNumberAndMarketId(storeName, marketId));
        if (CollectionUtils.isEmpty(restaurants)) {
            throw new DeploymentBadRequestException(String.format(Constants.NO_RESTAURANT_PRESENT_WITH_THE_NAME_S_FOR_THE_MARKET_ID, storeName, marketId));
        }
        return restaurants.get(0);
    }

    public PrePostDeploymentModel getPrePostModelById(Integer id) {
        return prePostDeploymentRepository.findOne(id, new QuerySpec(PrePostDeploymentModel.class));
    }

    public DeploymentGroup deploymentCheckRequestValidator(DeploymentCheckRequestDTO requestDTO) {
        if (requestDTO.getDeploymentGroupId() == null) {
            throw new DeploymentBadRequestException("DeploymentGroupId is required");
        }

        if (StringUtils.isBlank(requestDTO.getInitiatedBy())) {
            throw new DeploymentBadRequestException("InitiatedBy is required");
        }
        if (requestDTO.getType() == null) {
            throw new DeploymentBadRequestException("DeploymentCheckType is required");
        }
        DeploymentGroup group = deploymentGroupRepository.findOne(requestDTO.getDeploymentGroupId(), new QuerySpec(DeploymentGroup.class));
        if (!group.getType().equalsIgnoreCase("Bundle")) {
            throw new DeploymentBadRequestException("Pre-Deployment check is only applicable for bundle deployment");
        }
        if (!group.getStatus().equalsIgnoreCase(DeploymentStatus.InProgress.value())) {
            throw new DeploymentBadRequestException("Pre-Deployment check is only applicable for ongoing bundle deployment");
        }
        if (!CollectionUtils.isEmpty(prePostDeploymentRepository.getByDeploymentGroupIdAndStatus(group.getId(), DeploymentStatus.InProgress.value()))) {
            throw new DeploymentBadRequestException("A pre-deployment check is already in-progress. Please try after sometime");
        }
        if (prePostDeploymentRepository.getByDeploymentGroupIdAndStatus(group.getId(), DeploymentStatus.Completed.value()).size() >= 5) {
            throw new DeploymentBadRequestException("Maximum pre-deployment check limit of 5 has been reached");
        }
        return group;
    }

    public PrePostDeploymentModel prepareAndSavePrePostDeploymentDetails(DeploymentGroup group, DeploymentCheckRequestDTO deploymentCheckRequestDTO) {
        PrePostDeploymentModel model = PrePostDeploymentModel.builder()
                .deploymentGroup(group)
                .initiatedBy(deploymentCheckRequestDTO.getInitiatedBy())
                .type(deploymentCheckRequestDTO.getType().name())
                .status(deploymentCheckRequestDTO.getStatus().value())
                .build();
        return prePostDeploymentRepository.save(model);
    }

    public void callPrePostLambda(PrePostDeploymentModel model, DeploymentGroup group) throws JsonProcessingException {
        try {
            String functionInput = createLambdaRequestBody(model, group);
            String lambdaUrl = awsPrePostLambdaUrl + "-" + getEnvironment(accountId);
            log.info("Lambda URL " + lambdaUrl);
            InvokeRequest req = new InvokeRequest()
                    .withFunctionName(lambdaUrl)
                    .withPayload(ByteBuffer.wrap(functionInput.getBytes()));
            awsLambdaAsync.invokeAsync(req);
        } catch (Exception ex) {
            log.error("Caught an AmazonServiceException, which means " +
                    "your request made it to Amazon lambda, but was " +
                    "rejected with an error response for some reason" + ex.getMessage());
            throw ex;
        }
    }

    public String createLambdaRequestBody(PrePostDeploymentModel model, DeploymentGroup group) throws JsonProcessingException {
        try {
            Set<String> restaurantNos;
            Map<String, Object> messageBodyMap = new HashMap<>();
            Map<String, Map<String, Object>> fullBody = new HashMap<>();
            restaurantNos = group.getDeploymentHistory().stream().map(history -> history.getRestaurant().getName()).collect(Collectors.toSet());
            messageBodyMap.put("selectedRestaurantNumbers", restaurantNos);
            messageBodyMap.put("deploymentId", group.getId());
            messageBodyMap.put("assetId", model.getId());
            messageBodyMap.put("initiatedBy", model.getInitiatedBy());
            fullBody.put("body", messageBodyMap);
            return new ObjectMapper().writeValueAsString(fullBody);
        } catch (JsonProcessingException e) {
            log.error(e.getMessage());
            throw e;
        }
    }

    public String getEnvironment(String accountId) {
        if (StringUtils.isBlank(accountId)) {
            return "dev";
        }
        if (accountId.equalsIgnoreCase("524430043955"))
            return "prod";
        if (accountId.equalsIgnoreCase("593265675765"))
            return "int";
        if (accountId.equalsIgnoreCase("688810906228"))
            return "stg";
        else
            return "dev";
    }

}

