package com.mcd.restaurant.deployment.service;


import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectInputStream;
import com.amazonaws.util.IOUtils;
import com.mcd.restaurant.bundledrelease.controller.view.response.SQSApplicationResponse;
import com.mcd.restaurant.bundledrelease.enums.BundleType;
import com.mcd.restaurant.bundledrelease.enums.BundledReleaseStatus;
import com.mcd.restaurant.bundledrelease.error.BundleReleaseInvalidRequestException;
import com.mcd.restaurant.common.MapperUtils;
import com.mcd.restaurant.deployment.controller.view.beans.DeploymentHelper;
import com.mcd.restaurant.deployment.controller.view.beans.DeploymentStatisticsSegment;
import com.mcd.restaurant.deployment.controller.view.beans.DownloadReportHelper;
import com.mcd.restaurant.deployment.controller.view.request.*;
import com.mcd.restaurant.deployment.controller.view.response.*;
import com.mcd.restaurant.deployment.error.ComponentNotApplicableError;
import com.mcd.restaurant.deployment.error.DeploymentBadRequestException;
import com.mcd.restaurant.deployment.error.DeploymentError;
import com.mcd.restaurant.model.*;
import com.mcd.restaurant.repository.*;
import com.mcd.restaurant.repository.bulk.ComponentBulkCRUDRepository;
import io.crnk.core.queryspec.QuerySpec;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import com.mcd.restaurant.common.Constants;

import javax.transaction.Transactional;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.mcd.restaurant.deployment.service.DeploymentBuilder.*;

/**
 * Service to handle all deployment transaction scenarios
 */
@Slf4j
@Service
public class DeployService {

    private DeploymentGroupRepository deploymentGroupRepository;
    private ComponentRepository componentRepository;
    private DeploymentTagRepository deploymentTagRepository;
    private RestaurantRepository restaurantRepository;
    @Autowired
    private DeploymentUtils deploymentUtils;
    private DeploymentHistoryRepository deploymentHistoryRepository;
    private ComponentBulkCRUDRepository componentBulkCRUDRepository;
    @Autowired
    private ProductVersionRepository productVersionRepository;
    @Autowired
    private MapperUtils mapperUtils;
    @Autowired
    private ComponentTypeRepository componentTypeRepository;
    @Autowired
    private MarketRepository marketRepository;
    @Autowired
    private RestaurantViewRepository restaurantViewRepository;
    @Autowired
    private BundleReleaseRepository bundleReleaseRepository;
    @Autowired
    private AmazonS3 s3Client;
    @Value("${aws.sqs.queue.account.id}")
    String accountId;

    public DeployService(DeploymentGroupRepository deploymentGroupRepository, ComponentRepository componentRepository,
                         DeploymentTagRepository deploymentTagRepository, RestaurantRepository restaurantRepository, DeploymentHistoryRepository deploymentHistoryRepository, ComponentBulkCRUDRepository componentBulkCRUDRepository) {
        this.deploymentGroupRepository = deploymentGroupRepository;
        this.componentRepository = componentRepository;
        this.deploymentTagRepository = deploymentTagRepository;
        this.restaurantRepository = restaurantRepository;
        this.deploymentHistoryRepository = deploymentHistoryRepository;
        this.componentBulkCRUDRepository = componentBulkCRUDRepository;
    }

    /**
     * This method is responsible for validating components and restaurants data and will save all
     * Deployment request if all data provided is valid.
     *
     * @param request target DeployRequest
     * @return generated deployment ids
     */
    @Transactional
    public DeployResponse saveAllDeployments(DeployRequest request) {
        try {
            BundleRelease bundleRelease = null;
            Map<Integer, ProductVersions> versionsMap = new HashMap<>();
            List<DeploymentHistory> deploymentHistoryList = new ArrayList<>();
            deploymentRequestValidator(request);
            ComponentType type = deploymentUtils.fetchComponentType(request.getDeploymentType().name());
            List<Restaurant> selectedRestaurants = restaurantRepository.findAll(restaurantRepository.prepareQuerySpecFilterForRestaurantByIds(request.getRestaurantIds()));
            if (selectedRestaurants.size() != request.getRestaurantIds().size()) {
                throw new DeploymentBadRequestException(Constants.INVALID_RESTAURANTS_PASSED_FOR_DEPLOYMENT);
            }
            List<DeployComponentRequest> deployComponents = !CollectionUtils.isEmpty(request.getDeployComponents()) ? request.getDeployComponents() : new ArrayList<>();
            if (Boolean.TRUE.equals(request.getIsProductDeployment())) {
                prepareApplicationRequestFromProducts(request, deployComponents, versionsMap);
            }
            if (Boolean.TRUE.equals(request.getIsBundleDeployment())) {
                bundleRelease = bundleReleaseRepository.findOne(request.getBundleId(), new QuerySpec(BundleRelease.class));
                if (bundleRelease.getStatus().equals(BundledReleaseStatus.STAGED.value()) || bundleRelease.getStatus().equals(BundledReleaseStatus.FAILED.value())) {
                    throw new BundleReleaseInvalidRequestException(String.format(Constants.BUNDLE_IS_HENCE_CAN_NOT_BE_DISPLAYED, bundleRelease.getStatus()));
                }
            }
            Map<Integer, Restaurant> selectedRestaurantMap = selectedRestaurants.stream().collect(Collectors.toMap(Restaurant::getId, restaurant -> restaurant));
            DeploymentHelper helper = DeploymentHelper.builder()
                    .deploymentType(request.getDeploymentType())
                    .isProductDeployment(request.getIsProductDeployment())
                    .selectedRestaurants(selectedRestaurants)
                    .versionsMap(versionsMap)
                    .selectedRestaurantMap(selectedRestaurantMap)
                    .type(type).build();
            DeploymentTag deploymentTag = deploymentTagRepository.save(buildDeploymentTag(request.getTagName()));
            DeploymentGroup deploymentGroup = buildDeploymentGroup(request, deploymentTag, versionsMap, bundleRelease);
            request.getDeployComponents().forEach(application ->
                    deploymentHistoryList.addAll(createDeploymentHistoryRecord(application, deploymentGroup, helper))
            );
            deploymentGroupRepository.save(addHistoryToDeploymentGroup(deploymentGroup, parentgroup -> deploymentHistoryList));

            return DeployResponse.build(deploymentTag.getId(), deploymentTag.getName(), deploymentGroup.getId(), request);
        } catch (ComponentNotApplicableError | DeploymentBadRequestException e) {
            throw e;
        } catch (Exception ex) {
            throw new DeploymentError(ex.getMessage());
        }
    }

    private void deploymentRequestValidator(DeployRequest request) {
        if (Boolean.FALSE.equals(request.getIsProductDeployment()) && CollectionUtils.isEmpty(request.getDeployComponents())) {
            throw new ComponentNotApplicableError(Constants.INVALID_COMPONENT_REQUEST);
        }
        if (Boolean.TRUE.equals(request.getIsBundleDeployment()) && request.getBundleId() == null) {
            throw new DeploymentBadRequestException(Constants.BUNDLE_IS_REQUIRED_FOR_BUNDLE_DEPLOYMENT);
        }
        if (!CollectionUtils.isEmpty(request.getDeployComponents())) {
            request.getDeployComponents().forEach(i -> {
                if (i.getProductVersionId() != null) {
                    throw new DeploymentBadRequestException(Constants.PRODUCT_DEPLOYMENT_DETAILS_CAN_ONLY_BE_PASSED_IN_DEPLOYMENT_LIST);
                }
            });
        }
        if (Boolean.TRUE.equals(request.getIsProductDeployment()) && CollectionUtils.isEmpty(request.getDeployProducts())) {
            throw new ComponentNotApplicableError(Constants.INVALID_PRODUCT_REQUEST);
        }
    }

    private void prepareApplicationRequestFromProducts(DeployRequest request, List<DeployComponentRequest> productDeployComponents, Map<Integer, ProductVersions> versionsMap) {
        request.getDeployProducts().forEach(i -> {
            if (request.getDeploymentType() != DeploymentType.docker) {
                throw new DeploymentBadRequestException(Constants.PRODUCT_DEPLOYMENT_IS_APPLICABLE_FOR_DOCKER_APPLICATION_ONLY);
            }
            ProductVersions versions = productVersionRepository.findOne(i, new QuerySpec(ProductVersions.class));
            versionsMap.put(versions.getId(), versions);
            productDeployComponents.addAll(versions.getProductApplications().stream().map(application ->
                    DeployComponentRequest.builder()
                            .componentName(application.getApplicationName()).updateToVersion(application.getApplicationVersion()).productVersionId(i).build()
            ).collect(Collectors.toList()));
        });
        request.setDeployComponents(productDeployComponents);
    }

    private List<DeploymentHistory> createDeploymentHistoryRecord(DeployComponentRequest deployComponent, DeploymentGroup parentGroup, DeploymentHelper helper) {
        log.info(String.format("INFO: Initiate deployment for %s, type: %s, from: %s, to: %s",
                deployComponent.getComponentName(), helper.getDeploymentType(),
                deployComponent.getUpdateFromVersion(), deployComponent.getUpdateToVersion()));
        List<Component> components = getEligibleComponents(deployComponent, helper);
        return getDeploymentHistories(components, parentGroup, deployComponent, helper);
    }

    private List<Component> getEligibleComponents(DeployComponentRequest deployComponent, DeploymentHelper helper) {
        switch (helper.getDeploymentType()) {
            case docker:
                return componentRepository.getFilteredDockerComponents(deployComponent, helper.getSelectedRestaurantMap().keySet());
            case firmware:
                return componentRepository.getFilteredFirmwareComponents(deployComponent, helper.getSelectedRestaurantMap().keySet());
            case crd:
                return componentRepository.getFilteredCrdComponents(deployComponent, helper.getSelectedRestaurantMap().keySet());
            default:
                return Collections.emptyList();
        }
    }

    private List<DeploymentHistory> getDeploymentHistories(List<Component> components,
                                                           DeploymentGroup deploymentGroup,
                                                           DeployComponentRequest deployComponent, DeploymentHelper helper) {
        if (CollectionUtils.isEmpty(components) && helper.getDeploymentType().equals(DeploymentType.firmware)) {
            throw new ComponentNotApplicableError(String.format(
                    "The component [%s] with version [%s] is not applicable in requested restaurant",
                    deployComponent.getComponentName(), deployComponent.getUpdateFromVersion()));
        }
        List<Component> newComponents = new ArrayList<>();
        if (!helper.getDeploymentType().equals(DeploymentType.firmware)) {
            List<Integer> restaurantWithComponent = components.stream().map(i -> i.getRestaurant().getId()).collect(Collectors.toList());
            helper.getSelectedRestaurantMap().keySet().forEach(restaurant -> {
                if (!restaurantWithComponent.contains(restaurant)) {
                    Restaurant restaurantModel = Optional.ofNullable(helper.getSelectedRestaurantMap().get(restaurant)).orElseThrow(() -> new DeploymentBadRequestException("Restaurant with id" + Integer.toString(restaurant) + "does not exist"));
                    UpdateComponentRequestDTO requestDTO = UpdateComponentRequestDTO
                            .builder()
                            .applicationName(deployComponent.getComponentName())
                            .build();
                    newComponents.add(deploymentUtils.handleMissingComponentForDeployment(requestDTO, restaurantModel, helper.getType()));
                }
            });
            componentBulkCRUDRepository.saveAll(newComponents).forEach(components::add);
        }
        ProductVersions versions = null;
        if (helper.getIsProductDeployment() != null && helper.getIsProductDeployment() && deployComponent.getProductVersionId() != null) {
            versions = helper.getVersionsMap().get(deployComponent.getProductVersionId());
            versions.setIsDeployed(Boolean.TRUE.toString());
            productVersionRepository.save(versions);
        }
        Set<Integer> restaurants = new HashSet<>();
        ProductVersions finalVersions = versions;
        List<DeploymentHistory> deploymentHistories = components.stream()
                .map(it -> {
                    restaurants.add(it.getRestaurant().getId());
                    return buildDeploymentHistory(deploymentGroup, it, deployComponent, finalVersions, helper);
                })
                .collect(Collectors.toList());
        if (restaurants.size() < helper.getSelectedRestaurantMap().keySet().size()) {
            log.debug(String.format("INFO: Restaurants applicable %d", restaurants.size()));
            throw new ComponentNotApplicableError(String.format(
                    "The component [%s] with version [%s] is not applicable in some of the requested restaurants",
                    deployComponent.getComponentName(), deployComponent.getUpdateFromVersion()));
        }
        return deploymentHistories;

    }

    @Transactional
    public StatusChangeResponse changeDeploymentStatus(int id, DeploymentStatus deploymentStatus) {
        DeploymentGroup deploymentGroup = deploymentGroupRepository.findOne(id, new QuerySpec(DeploymentGroup.class));
        deploymentGroup.getDeploymentHistory().forEach(it -> {
            if (!it.getStatus().equalsIgnoreCase(DeploymentStatus.ALREADYPRESENT.value()))
                it.setStatus(deploymentStatus.value());
        });
        deploymentGroupRepository.save(deploymentGroup);
        return StatusChangeResponse.build(deploymentStatus);
    }

    public SuggestionResponseDTO getSuggestionList(SuggestionType suggestionType, String searchParam) {
        QuerySpec querySpec;
        List<String> suggestionList = new ArrayList<>();
        switch (suggestionType) {
            case ID:
                querySpec = deploymentUtils.prepareQuerySpecFilterForSuggestion(DeploymentGroup.class, SuggestionType.ID.value(), searchParam);
                suggestionList.addAll(deploymentGroupRepository.findAll(querySpec).stream().map(i ->
                        Integer.toString(i.getId())
                ).distinct().collect(Collectors.toList()));
                break;

            case STORE_NAME:
                querySpec = deploymentUtils.prepareQuerySpecFilterForSuggestion(Restaurant.class, SuggestionType.STORE_NAME.value(), searchParam);
                suggestionList.addAll(restaurantRepository.findAll(querySpec).stream().map(Restaurant::getName
                ).distinct().collect(Collectors.toList()));
                break;

            case DEPLOYMENT_NAME:
                querySpec = deploymentUtils.prepareQuerySpecFilterForSuggestion(DeploymentTag.class, SuggestionType.DEPLOYMENT_NAME.value(), searchParam);
                suggestionList.addAll(deploymentTagRepository.findAll(querySpec).stream().map(DeploymentTag::
                        getName
                ).distinct().collect(Collectors.toList()));
                break;
            case RESTAURANT_NO:
                querySpec = deploymentUtils.prepareQuerySpecFilterForSuggestion(Restaurant.class, SuggestionType.RESTAURANT_NO.value(), searchParam);
                suggestionList.addAll(restaurantRepository.findAll(querySpec).stream().map(Restaurant::
                        getName
                ).distinct().collect(Collectors.toList()));
                break;
            case DEPLOYED_BY:
                querySpec = deploymentUtils.prepareQuerySpecFilterForSuggestion(DeploymentGroup.class, SuggestionType.DEPLOYED_BY.value(), searchParam);
                suggestionList.addAll(deploymentGroupRepository.findAll(querySpec).stream().map(DeploymentGroup::
                        getDeployedBy
                ).distinct().collect(Collectors.toList()));
                break;
            default:
                throw new DeploymentError(Constants.UNKNOWN_SUGGESTION_TYPE);


        }
        return SuggestionResponseDTO.builder().suggestionList(suggestionList).build();
    }


    @Transactional
    public ArchivalDataResponseDTO getDeploymentGroupForArchival(Long archiveBeforeTime, Long limit, Integer offset) {
        List<DeploymentGroup> deploymentGroupList = deploymentGroupRepository.findAll(deploymentUtils.prepareQuerySpecFilterForArchival(archiveBeforeTime, limit, offset));
        List<DeploymentGroupArchivalResponseDTO> deploymentGroupArchivalResponseDTOList = mapperUtils.map(deploymentGroupList, DeploymentGroupArchivalResponseDTO.class);
        deploymentGroupArchivalResponseDTOList.parallelStream().forEach(group ->
                {
                    group.getDeploymentHistory().stream().forEach(history -> {
                        history.setComponentId(history.getComponent().getId());
                        history.setRestaurantId(history.getRestaurant().getId());
                        history.setDeploymentGroupId(history.getDeployment_group().getId());
                        history.setRestaurant(null);
                        history.setDeployment_group(null);
                        history.setComponent(null);
                        if (history.getProductVersions() != null) {
                            history.setProductVersionId(Integer.toString(history.getProductVersions().getId()));
                        }
                    });
                    group.setBundleId(group.getBundleRelease() != null ? group.getBundleRelease().getId() : null);
                    group.setBundleRelease(null);
                }
        );

        return ArchivalDataResponseDTO.builder().deploymentGroupList(deploymentGroupArchivalResponseDTOList).build();
    }

    @Transactional
    public ArchivalDeleteResponse deleteDeploymentRecordsAfterArchival(PostArchivalDeleteRequestDTO requestDTO) {
        if (CollectionUtils.isEmpty(requestDTO.getDeploymentGroupId())) {
            throw new DeploymentError(Constants.ID_LIST_CONNOT_BE_EMPTY);
        }
        try {
            //deployment history records will get deleted along with groups
            log.info(Constants.INITIATING_DELETION_OF_RECORDS_POST_ARCHIVAL + System.currentTimeMillis());
            deploymentUtils.asyncDeletePOstArchival(requestDTO);
            return ArchivalDeleteResponse.builder().isDeleted(true).build();
        } catch (Exception e) {
            log.error(Constants.EXCEPTION_OCCURED_WHILE_DELETING_THE_RECORDS_AFTER_ARCHIVAL);
            log.error(e.getMessage());
            return ArchivalDeleteResponse.builder().isDeleted(false).build();
        }
    }


    @Transactional
    public DeployGraphResponseDTO prepareStatisticsForDeployGraph(Long startDate, Long endDate, DeployGraphSegment
            deployGraphSegment) {
        try {

            boolean immediateStop = true;
            LocalDateTime segmentEndTime;
            Integer counter = 0;
            Integer totalCount = 0;
            Map<Integer, Integer> completedSegmentmap = new HashMap<>();
            Map<Integer, Integer> failedSegmentmap = new HashMap<>();

            LocalDateTime startTime = LocalDateTime.ofInstant(Instant.ofEpochMilli(startDate), ZoneId.of("UTC"));
            LocalDateTime endTime = LocalDateTime.ofInstant(Instant.ofEpochMilli(endDate), ZoneId.of("UTC"));

            while (counter < deployGraphSegment.value() && immediateStop) {
                switch (deployGraphSegment) {
                    case DAY:
                        return deploymentUtils.analyseDeploymentResultsDay(startTime.toEpochSecond(ZoneOffset.UTC) * 1000, endTime.toEpochSecond(ZoneOffset.UTC) * 1000);
                    case WEEK:
                        segmentEndTime = startTime.plusDays(deployGraphSegment.getSegment());
                        break;
                    case MONTH:
                        if (counter == 0)
                            segmentEndTime = startTime.plusDays(deployGraphSegment.getSegment());
                        else
                            segmentEndTime = startTime.plusDays(deployGraphSegment.getSegment() + 1l);
                        break;
                    default:
                        throw new DeploymentError(Constants.UNKNOWN_GRAPH_SEGMENT);
                }
                if (segmentEndTime.isAfter(endTime)) {
                    segmentEndTime = endTime;
                    immediateStop = false;
                }
                DeploymentStatisticsSegment statistics = deploymentUtils.analyseDeploymentResults(startTime.toEpochSecond(ZoneOffset.UTC) * 1000, segmentEndTime.toEpochSecond(ZoneOffset.UTC) * 1000);
                completedSegmentmap.put(counter, statistics.getCompleted());
                failedSegmentmap.put(counter, statistics.getFailed());
                totalCount = totalCount + statistics.getCompleted() + statistics.getFailed();
                startTime = segmentEndTime;
                counter++;
            }
            return DeployGraphResponseDTO.builder().failedDeployments(failedSegmentmap).completedDeployments(completedSegmentmap).totalDeployment(totalCount).build();
        } catch (Exception e) {
            throw new DeploymentError(e.getMessage());
        }
    }

    @Transactional
    public UpdateComponentResponseDTO modifyComponentRecordsForRestaurant(UpdateComponentRequestDTO
                                                                                  updateComponentRequestDTO) {
        try {
            if (StringUtils.isBlank(updateComponentRequestDTO.getApplicationName()) || StringUtils.isBlank(updateComponentRequestDTO.getApplicationVersion()) || StringUtils.isBlank(updateComponentRequestDTO.getRestaurantNo()) || StringUtils.isBlank(updateComponentRequestDTO.getMarketName())) {
                throw new DeploymentBadRequestException(Constants.APPLICATION_NAME_APPLICATION_VERSION_RESTAURANT_NAME_MARKET_NAME_IS_MANDATORY);
            }
            UpdateComponentResponseDTO updateComponentResponseDTO = UpdateComponentResponseDTO.builder().isNewComponentInserted(false).build();
            Component component;
            Restaurant restaurant = deploymentUtils.validateMarketAndRestaurant(updateComponentRequestDTO.getMarketName(), updateComponentRequestDTO.getRestaurantNo());
            List<Component> components = componentRepository.getFilteredDockerComponents(DeployComponentRequest
                    .builder()
                    .componentName(updateComponentRequestDTO.getApplicationName())
                    .build(), Stream.of(restaurant.getId()).collect(Collectors.toSet()));
            if (CollectionUtils.isEmpty(components)) {
                component = deploymentUtils.handleMissingComponent(updateComponentRequestDTO, updateComponentResponseDTO, restaurant);
            } else {
                component = components.get(0);
                component.setReportedVersion(updateComponentRequestDTO.getApplicationVersion());
            }

            component = componentRepository.save(component);
            updateComponentResponseDTO.setComponentId(component.getId());
            updateComponentResponseDTO.setIsVersionUpdated(true);
            return updateComponentResponseDTO;
        } catch (DeploymentBadRequestException ex) {
            throw ex;
        } catch (Exception e) {
            throw new DeploymentError(e.getMessage());
        }
    }


    @Transactional
    public DeploymentGroupDetailsResponseDTO getDeploymentDetails(Integer deploymentGroupId) {
        try {
            List<ProductApplications> applicationsList = new ArrayList<>();
            List<DeploymentHistory> deploymentHistoryListFull = deploymentHistoryRepository.getHistoryByGroupId(deploymentGroupId);
            List<DeploymentHistory> deploymentHistoryList = deploymentHistoryListFull.stream()
                    .filter(history -> !history.getApplicationName().equalsIgnoreCase(Constants.IMAGE_CACHE))
                    .collect(Collectors.toList());
            if (CollectionUtils.isEmpty(deploymentHistoryListFull)) {
                throw new DeploymentBadRequestException(Constants.NO_DEPLOYMENT_RECORDS_PRESENT_FOR_THE_GIVEN_ID);
            }
            Set<Integer> restaurantSet = deploymentHistoryListFull
                    .stream()
                    .map(history ->
                            history.getRestaurant().getId()
                    ).collect(Collectors.toSet());
            if (!CollectionUtils.isEmpty(deploymentHistoryList)) {
                applicationsList = deploymentUtils.getUniqueProductApplicationsInDeployment(deploymentHistoryList);
            }
            List<RestaurantView> restaurantViewList = restaurantViewRepository.getRestaurantDetailsById(restaurantSet);
            Map<String, List<SQSApplicationResponse>> applicationNameSpace = deploymentUtils.updateApplicationDetailsGroupedByProduct(applicationsList, deploymentHistoryListFull);
            return deploymentUtils.buildDeploymentDetailResponseDTO(deploymentHistoryList, restaurantViewList, applicationNameSpace);
        } catch (DeploymentBadRequestException ex) {
            throw ex;
        } catch (Exception e) {
            throw new DeploymentError(e.getMessage());
        }
    }


    @Transactional
    public StatusChangeResponse updateDeploymentHistoryStatus(UpdateDeploymentStatusRequestDTO requestDTO) {
        try {
            deploymentUtils.deploymentStatusUpdateRequestValidator(requestDTO);
            Restaurant restaurant = deploymentUtils.validateMarketAndRestaurant(requestDTO.getMarketName(), requestDTO.getRestaurantNo());
            List<DeploymentHistory> deploymentHistoryList = deploymentHistoryRepository.getHistoryByGroupIdAndRestaurantIdAndApplicationNameAndApplicationVersion(requestDTO.getDeploymentGroupId(), restaurant.getId(), requestDTO.getApplicationName(), requestDTO.getApplicationVersion());
            if (CollectionUtils.isEmpty(deploymentHistoryList)) {
                throw new DeploymentBadRequestException(Constants.NO_DEPLOYMENT_RECORD_FOUND);
            }
            DeploymentHistory deploymentHistory = deploymentHistoryList.get(0);
            deploymentHistory.setStatus(requestDTO.getStatus().value());
            deploymentHistoryRepository.save(deploymentHistory);
            return StatusChangeResponse.build(requestDTO.getStatus());
        } catch (DeploymentBadRequestException ex) {
            throw ex;
        } catch (Exception e) {
            throw new DeploymentError(e.getMessage());
        }
    }

    @Transactional
    public DeployResponse saveAllBundleDeployments(BundleDeployRequest bundleDeployRequest) {
        try {
            deploymentUtils.bundleDeploymentRequestValidator(bundleDeployRequest);
            Map<Integer, ProductVersions> versionsMap = new HashMap<>();
            List<DeployComponentRequest> componentRequests = new ArrayList<>();
            List<DeploymentHistory> deploymentHistoryList = new ArrayList<>();
            DeploymentGroup group = deploymentGroupRepository.findOne(bundleDeployRequest.getDeploymentGroupId(), new QuerySpec(DeploymentGroup.class));
            BundleRelease bundleRelease = group.getBundleRelease();
            DeployRequest request = mapperUtils.map(bundleDeployRequest, DeployRequest.class);
            request.setDeploymentType(DeploymentType.docker);
            if (bundleRelease.getType().equalsIgnoreCase(BundleType.PRODUCTS.value())) {
                request.setIsProductDeployment(Boolean.TRUE);
            } else {
                request.setIsProductDeployment(Boolean.FALSE);
            }
            bundleRelease.getBundleReleaseAppVersions().forEach(application -> {
                if (bundleRelease.getType().equalsIgnoreCase(BundleType.PRODUCTS.value())) {
                    versionsMap.put(application.getProductVersions().getId(), application.getProductVersions());
                    componentRequests.add(DeployComponentRequest.builder().productVersionId(application.getProductVersions().getId()).updateToVersion(application.getVersion()).componentName(application.getAppName()).build());
                } else {
                    componentRequests.add(DeployComponentRequest.builder().updateToVersion(application.getVersion()).componentName(application.getAppName()).build());
                }
            });
            request.setDeployComponents(componentRequests);
            List<Restaurant> selectedRestaurants = new ArrayList<>(group.getDeploymentHistory().stream().map(DeploymentHistory::getRestaurant).collect(Collectors.toSet()));
            ComponentType type = deploymentUtils.fetchComponentType(request.getDeploymentType().name());
            Map<Integer, Restaurant> selectedRestaurantMap = selectedRestaurants.stream().collect(Collectors.toMap(Restaurant::getId, restaurant -> restaurant));
            request.setRestaurantIds(selectedRestaurantMap.keySet());
            DeploymentHelper helper = DeploymentHelper.builder()
                    .deploymentType(request.getDeploymentType())
                    .isProductDeployment(request.getIsProductDeployment())
                    .selectedRestaurants(selectedRestaurants)
                    .versionsMap(versionsMap)
                    .selectedRestaurantMap(selectedRestaurantMap)
                    .type(type).build();
            request.getDeployComponents().forEach(application ->
                    deploymentHistoryList.addAll(createDeploymentHistoryRecord(application, group, helper))
            );
            group.getDeploymentHistory().addAll(deploymentHistoryList);
            deploymentGroupRepository.save(group);
            return DeployResponse.build(group.getDeploymentTag().getId(), group.getDeploymentTag().getName(), group.getId(), request);
        } catch (ComponentNotApplicableError | DeploymentBadRequestException e) {
            throw e;
        } catch (Exception ex) {
            throw new DeploymentError(ex.getMessage());
        }

    }

    @Transactional
    public BulkInsertComponentResponseDTO insertComponentData(BulkInsertComponentRequestDTO componentDTO) {
        try {
            deploymentUtils.validateRequest(componentDTO);
            Restaurant restaurant = deploymentUtils.validateMarketAndRestaurant(componentDTO.getMarketName(), componentDTO.getStoreName());
            List<String> components = componentDTO.getComponents().stream().map(ComponentDetailRequestDTO::getApplicationName).collect(Collectors.toList());
            List<Component> existingComponents = deploymentUtils.fetchExistingComponents(components, restaurant);
            Map<String, String> mapRequest = componentDTO.getComponents().stream().collect(Collectors.toMap(ComponentDetailRequestDTO::getApplicationName, ComponentDetailRequestDTO::getApplicationVersion));
            List<String> existingComponentName = new ArrayList<>();
            existingComponents.forEach(component -> {
                if (mapRequest.containsKey(component.getName())) {
                    component.setReportedVersion(mapRequest.get(component.getName()));
                    existingComponentName.add(component.getName());
                }
            });
            List<ComponentDetailRequestDTO> newComponents = componentDTO.getComponents().stream().filter(component -> !existingComponentName.contains(component.getApplicationName())).collect(Collectors.toList());
            List<com.mcd.restaurant.model.Component> prepareResponse = deploymentUtils
                    .bulkInsertComponentsData(newComponents, restaurant, existingComponents);
            return deploymentUtils.bulkInsertComponentResponse(prepareResponse, restaurant, componentDTO.getMarketName());

        } catch (DeploymentBadRequestException ex) {
            throw ex;
        } catch (Exception e) {
            throw new DeploymentError(e.getMessage());
        }
    }


    @Transactional
    public SuggestionResponseDTO getApplicationForDeployment(SuggestionType suggestionType, String applicationName) {
        SuggestionResponseDTO responseDTO = SuggestionResponseDTO.builder().build();
        deploymentUtils.validateInputForFetchingApplicationDetails(suggestionType, applicationName);
        try {
            if (suggestionType.equals(SuggestionType.APPLICATION_NAME)) {

                responseDTO.setSuggestionList(deploymentUtils.fetchAllProductApplications());
            } else if (suggestionType.equals(SuggestionType.APPLICATION_VERSION)) {
                responseDTO.setSuggestionList(deploymentUtils.fetchAllVersionForApplication(applicationName));
            }
            return responseDTO;
        } catch (Exception ex) {
            throw new DeploymentError(ex.getMessage());
        }
    }

    @Transactional
    public DeploymentCheckResponseDTO saveDeploymentCheckDetails(DeploymentCheckRequestDTO deploymentCheckRequestDTO) {
        try {
            DeploymentGroup group = deploymentUtils.deploymentCheckRequestValidator(deploymentCheckRequestDTO);
            PrePostDeploymentModel model = deploymentUtils.prepareAndSavePrePostDeploymentDetails(group, deploymentCheckRequestDTO);
            deploymentUtils.callPrePostLambda(model, group);
            return DeploymentCheckResponseDTO.build(model);
        } catch (DeploymentBadRequestException ex) {
            throw ex;
        } catch (Exception e) {
            throw new DeploymentError(e.getMessage());
        }
    }

    @Transactional
    public DownloadReportHelper downloadFile(Integer id) {
        try {
            PrePostDeploymentModel model = deploymentUtils.getPrePostModelById(id);
            if (model.getStatus().equalsIgnoreCase(DeploymentStatus.Completed.value())) {
                String bucketName = getBucketName();
                S3Object s3Object = s3Client.getObject(bucketName, model.getReportPath());
                S3ObjectInputStream inputStream = s3Object.getObjectContent();
                byte[] content = IOUtils.toByteArray(inputStream);
                return DownloadReportHelper.builder().content(content).fileName(model.getReportPath()).build();
            } else {
                throw new DeploymentBadRequestException("Report is not Generated. Current Status is: " + model.getStatus());
            }
        } catch (DeploymentBadRequestException e) {
            throw e;
        } catch (Exception e) {
            throw new DeploymentError(e.getMessage());
        }
    }

    public String getBucketName() {
        String env = deploymentUtils.getEnvironment(accountId);
        return "bre-" + env + "-pre-post-check";

    }
}