package com.mcd.restaurant.common;

import com.mcd.restaurant.service.ApiLoggingService;
import io.crnk.core.engine.document.ErrorData;
import io.crnk.core.engine.document.ErrorDataBuilder;
import io.crnk.core.engine.error.ErrorResponse;
import io.crnk.core.engine.error.ExceptionMapper;
import io.crnk.core.engine.http.HttpStatus;
import io.crnk.core.exception.BadRequestException;
import io.crnk.core.exception.CrnkMappableException;
import io.crnk.core.exception.ForbiddenException;
import io.crnk.core.exception.InternalServerErrorException;
import io.crnk.core.exception.MethodNotAllowedException;
import io.crnk.core.exception.ResourceNotFoundException;
import io.crnk.core.exception.UnauthorizedException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Iterator;

/**
 * ExceptionMapper for {@link CrnkMappableException} in order to add error id.
 */
@Component
public class AssetExceptionMapper implements ExceptionMapper<CrnkMappableException> {

    @Autowired
    private ApiLoggingService apiLoggingService;

    @Override
    public ErrorResponse toErrorResponse(CrnkMappableException exception) {
        String errorId = logError(exception);
        ErrorData errorData = exception.getErrorData();
        String message = exception.getHttpStatus() >= 500 ?
                "There was an issue with processing your request. Please try after sometime." :
                errorData.getDetail();
        ErrorDataBuilder builder = ErrorData.builder()
                .setId(errorId)
                .setTitle(errorData.getTitle())
                .setCode(ErrorCode.EXCEPTION_MAPPER_ASSET)
                .setStatus(errorData.getStatus())
                .setDetail(message);
        return ErrorResponse.builder()
                .setStatus(exception.getHttpStatus())
                .setSingleErrorData(builder.build())
                .build();
    }

    private String logError(CrnkMappableException exception) {
        if (exception.getHttpStatus() >= 500 && exception.getHttpStatus() < 600) {
            return apiLoggingService.logError(exception);
        } else {
            return apiLoggingService.logWarn(exception);
        }
    }

    @Override
    public CrnkMappableException fromErrorResponse(ErrorResponse errorResponse) {
        String message = getMessage(errorResponse);

        int httpStatus = errorResponse.getHttpStatus();
        if (httpStatus == HttpStatus.FORBIDDEN_403) {
            return new ForbiddenException(message);
        }
        if (httpStatus == HttpStatus.METHOD_NOT_ALLOWED_405) {
            return new MethodNotAllowedException(message);
        }
        if (httpStatus == HttpStatus.UNAUTHORIZED_401) {
            return new UnauthorizedException(message);
        }
        if (httpStatus == HttpStatus.NOT_FOUND_404) {
            return new ResourceNotFoundException(message);
        }
        if (httpStatus == HttpStatus.BAD_REQUEST_400) {
            return new BadRequestException(message);
        }
        if (httpStatus == HttpStatus.INTERNAL_SERVER_ERROR_500) {
            return new InternalServerErrorException(message);
        }
        throw new IllegalStateException(errorResponse.toString());
    }

    private String getMessage(ErrorResponse errorResponse) {
        Iterator<ErrorData> errors = errorResponse.getErrors().iterator();
        String message = null;
        if (errors.hasNext()) {
            ErrorData data = errors.next();
            message = data.getDetail();
            if (message == null) {
                message = data.getTitle();
            }
            if (message == null) {
                message = data.getCode();
            }
        }
        return message;
    }

    @Override
    public boolean accepts(ErrorResponse errorResponse) {
        int httpStatus = errorResponse.getHttpStatus();
        return httpStatus == HttpStatus.NOT_FOUND_404 || httpStatus == HttpStatus.METHOD_NOT_ALLOWED_405 ||
                httpStatus == HttpStatus.BAD_REQUEST_400 || httpStatus == HttpStatus.FORBIDDEN_403
                || httpStatus == HttpStatus.UNAUTHORIZED_401 || httpStatus == HttpStatus.INTERNAL_SERVER_ERROR_500;
    }
}
