package com.mcd.restaurant.bundledrelease.service;

import com.mcd.restaurant.bundledrelease.controller.view.request.BundledAppVersionRequest;
import com.mcd.restaurant.bundledrelease.controller.view.request.BundledReleaseRequest;
import com.mcd.restaurant.bundledrelease.controller.view.request.UpdateBundleDeploymentDetailsRequest;
import com.mcd.restaurant.bundledrelease.controller.view.request.UpdateBundleReleaseRequest;
import com.mcd.restaurant.bundledrelease.controller.view.response.BundleReleaseGenericResponse;
import com.mcd.restaurant.bundledrelease.controller.view.response.BundledReleaseAppVersionResponse;
import com.mcd.restaurant.bundledrelease.controller.view.response.BundledReleaseResponse;
import com.mcd.restaurant.bundledrelease.controller.view.response.SQSApplicationResponse;
import com.mcd.restaurant.bundledrelease.enums.BundledReleaseStatus;
import com.mcd.restaurant.bundledrelease.error.BundleNotFoundException;
import com.mcd.restaurant.bundledrelease.error.BundleReleaseGenericException;
import com.mcd.restaurant.bundledrelease.error.BundleReleaseInvalidRequestException;
import com.mcd.restaurant.bundledrelease.utils.BundleReleaseUtility;
import com.mcd.restaurant.model.BundleRelease;
import com.mcd.restaurant.model.BundleReleaseAppVersion;
import com.mcd.restaurant.model.ProductApplications;
import com.mcd.restaurant.model.ProductVersions;
import com.mcd.restaurant.repository.BundleReleaseAppVersionRepository;
import com.mcd.restaurant.repository.BundleReleaseRepository;
import com.mcd.restaurant.repository.ProductApplicationRepository;
import com.mcd.restaurant.repository.ProductVersionRepository;
import io.crnk.core.engine.internal.utils.StringUtils;
import io.crnk.core.exception.ResourceNotFoundException;
import io.crnk.core.queryspec.QuerySpec;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import javax.transaction.Transactional;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.ArrayList;
import static com.mcd.restaurant.bundledrelease.service.BundledReleaseBuilder.buildBundleRelease;
import static com.mcd.restaurant.bundledrelease.service.BundledReleaseBuilder.buildBundleReleaseAppVersion;

/**
 * Service to handle all Bundle Release transaction scenarios
 */
@Slf4j
@Service
public class BundledReleaseService {

    private BundleReleaseRepository bundleReleaseRepository;
    private BundleReleaseAppVersionRepository bundleReleaseAppVersionRepository;
    private static final String SOME_EXCEPTION_OCCURRED = "Some Exception occurred";
    @Autowired
    private BundleReleaseUtility bundleReleaseUtility;
    @Autowired
    private ProductVersionRepository productVersionRepository;
    @Autowired
    private ProductApplicationRepository productApplicationRepository;

    public BundledReleaseService(BundleReleaseRepository bundleReleaseRepository, BundleReleaseAppVersionRepository bundleReleaseAppVersionRepository) {
        this.bundleReleaseRepository = bundleReleaseRepository;
        this.bundleReleaseAppVersionRepository = bundleReleaseAppVersionRepository;
    }

    /**
     * This method is responsible for generating bundle release if all the data provided is valid.
     *
     * @param bundledReleaseRequest target BundledReleaseRequest
     * @return generated deployment ids
     */

    @Transactional
    public BundleReleaseGenericResponse saveBundleRelease(BundledReleaseRequest bundledReleaseRequest) {
        try {
            Map<Integer, ProductVersions> versionsMap = new HashMap<>();
            List<BundledAppVersionRequest> bundledApps = bundledReleaseRequest.getBundledAppVersionRequestList() != null ? bundledReleaseRequest.getBundledAppVersionRequestList() : new ArrayList<>();
            BundleRelease bundleRelease = bundleReleaseRepository.save(buildBundleRelease(bundledReleaseRequest));
            createBundleAppProductCollections(bundledReleaseRequest.getProductVersionIdList(), versionsMap, bundledApps, bundleRelease);
            BundledReleaseResponse bundledReleaseResponse = new BundledReleaseResponse();
            bundledReleaseResponse.setReleaseId(bundleRelease.getId());
            bundledReleaseResponse.setReleaseName(bundleRelease.getReleaseName());
            List<SQSApplicationResponse> uniqueApplications = getUniqueApplications(bundledApps);
            if (uniqueApplications.size() != bundledApps.size()) {
                throw new BundleReleaseInvalidRequestException("Applications not found in database");
            }
            bundleReleaseUtility.sendSqsMessage(uniqueApplications, bundleRelease.getId(), bundledReleaseRequest.getReleaseName(), "POST");
            return bundleReleaseUtility.generateResponse(bundledApps.size(), bundledReleaseResponse);
        } catch (BundleReleaseInvalidRequestException ex) {
            throw ex;
        } catch (Exception e) {
            throw new BundleReleaseGenericException(StringUtils.isBlank(e.getMessage()) ? SOME_EXCEPTION_OCCURRED : e.getMessage());
        }
    }

    private List<SQSApplicationResponse> getUniqueApplications(List<BundledAppVersionRequest> bundledApps) {
        List<SQSApplicationResponse> uniqueApplications = new ArrayList<>();
        List<String> applicationName = new ArrayList<>();
        List<ProductApplications> applications = productApplicationRepository.getAllApplicationByNameAndVersion(bundledApps);
        applications.forEach(app -> {
            if (!applicationName.contains(app.getApplicationName())) {
                uniqueApplications.add(SQSApplicationResponse.builder()
                        .applicationName(app.getApplicationName())
                        .applicationVersion(app.getApplicationVersion())
                        .applicationUrl(app.getApplicationUrl()).build());
                applicationName.add(app.getApplicationName());
            }
        });

        return uniqueApplications;

    }

    private List<BundledReleaseAppVersionResponse> createBundleAppProductCollections(List<Integer> productVersionIdList, Map<Integer, ProductVersions> versionsMap, List<BundledAppVersionRequest> bundledApps, BundleRelease bundleRelease) {
        if (!CollectionUtils.isEmpty(productVersionIdList)) {
            handleProductRelatedApplications(productVersionIdList, bundledApps, versionsMap);
        }
        List<String> applicationName = bundledApps.stream().map(BundledAppVersionRequest::getAppName).collect(Collectors.toList());
        applicationName.forEach(i -> {
            if (Collections.frequency(applicationName, i) > 1) {
                throw new BundleReleaseInvalidRequestException("Duplicate Applications cannot be inserted in a Bundle");
            }
        });
        return bundledApps.stream().map(it -> {
            ProductVersions versions = null;
            if (it.getProductVersionId() != null) {
                versions = versionsMap.get(it.getProductVersionId());
            }
            BundleReleaseAppVersion bundleReleaseAppVersion = bundleReleaseAppVersionRepository
                    .save(buildBundleReleaseAppVersion(it, bundleRelease, versions));
            return BundledReleaseAppVersionResponse.build(it, bundleReleaseAppVersion.getId());
        }).collect(Collectors.toList());
    }

    private void handleProductRelatedApplications(List<Integer> productVersionIdList, List<BundledAppVersionRequest> bundledApps, Map<Integer, ProductVersions> versionsMap) {
        productVersionIdList.forEach(i -> {
            ProductVersions versions = productVersionRepository.findOne(i, new QuerySpec(ProductVersions.class));
            versionsMap.put(versions.getId(), versions);
            bundledApps.addAll(versions.getProductApplications().stream().map(application ->
                    BundledAppVersionRequest.builder()
                            .appName(application.getApplicationName()).appVersion(application.getApplicationVersion()).productVersionId(versions.getId()).build()
            ).collect(Collectors.toList()));
        });
    }

    /**
     * This method is responsible for validating bundle release status and editing the bundle release app version and path
     *
     * @param updateBundleReleaseRequest target UpdateBundleReleaseRequest
     * @return generated success status along with number of apps updated
     */
    @Transactional
    public BundleReleaseGenericResponse editBundleRelease(UpdateBundleReleaseRequest updateBundleReleaseRequest) {
        try {
            Map<Integer, ProductVersions> versionsMap = new HashMap<>();
            List<BundledAppVersionRequest> bundledApps = updateBundleReleaseRequest.getBundledAppVersionRequestList() != null ? updateBundleReleaseRequest.getBundledAppVersionRequestList() : new ArrayList<>();
            BundleRelease bundleRelease = bundleReleaseUtility.validateBundleReleaseStatus(updateBundleReleaseRequest.getReleaseId());
            bundleRelease.setUpdatedBy(updateBundleReleaseRequest.getUpdatedBy());
            bundleRelease.setStatus(BundledReleaseStatus.STAGED.value());
            bundleReleaseRepository.save(bundleRelease);
            bundleReleaseAppVersionRepository.deleteWithNativeQuery(bundleRelease.getId());
            List<BundledReleaseAppVersionResponse> appResponse = createBundleAppProductCollections(updateBundleReleaseRequest.getProductVersionIdList(), versionsMap, bundledApps, bundleRelease);
            List<SQSApplicationResponse> uniqueApplications = getUniqueApplications(bundledApps);
            if (uniqueApplications.size() != bundledApps.size()) {
                throw new BundleReleaseInvalidRequestException("Applications not found in database");
            }
            bundleReleaseUtility.sendSqsMessage(uniqueApplications, bundleRelease.getId(), bundleRelease.getReleaseName(), "PUT");
            return bundleReleaseUtility.generateResponse(appResponse.size(), new BundleReleaseGenericResponse());
        } catch (BundleReleaseInvalidRequestException ex) {
            throw ex;
        } catch (Exception e) {
            throw new BundleReleaseGenericException(StringUtils.isBlank(e.getMessage()) ? SOME_EXCEPTION_OCCURRED : e.getMessage());
        }
    }


    /**
     * This method is responsible for validating bundle release status and deleting record from BundleRelease and BundleReleaseAppVersion table     *
     *
     * @param releaseId target BundleRelease.id
     * @return generated success status along with number of apps data deleted from BundleReleaseAppVersion
     */
    @Transactional
    public BundleReleaseGenericResponse deleteBundleRelease(Integer releaseId) {
        try {
            BundleRelease release = bundleReleaseUtility.validateDeleteBundleReleaseStatus(releaseId);
            List<BundleReleaseAppVersion> bundleReleaseAppVersionList = bundleReleaseAppVersionRepository.findAll(bundleReleaseUtility.prepareQuerySpecFilter(releaseId));
            bundleReleaseAppVersionList.stream().forEach(bundleReleaseAppVersion -> bundleReleaseAppVersionRepository.delete(bundleReleaseAppVersion.getId()));
            bundleReleaseRepository.delete(releaseId);
            bundleReleaseUtility.sendSqsMessage(null, releaseId, release.getReleaseName(), "DELETE");
            return bundleReleaseUtility.generateResponse(bundleReleaseAppVersionList.size(), new BundleReleaseGenericResponse());
        } catch (ResourceNotFoundException e) {
            throw new BundleNotFoundException("No Bundle exist for the given id");
        } catch (Exception e) {
            throw new BundleReleaseGenericException(StringUtils.isBlank(e.getMessage()) ? SOME_EXCEPTION_OCCURRED : e.getMessage());
        }
    }

    @Transactional
    public BundleReleaseGenericResponse editDeploymentDetailForBundle(UpdateBundleDeploymentDetailsRequest deploymentDetailsRequest) {
        try {
            BundleRelease bundleRelease = bundleReleaseRepository.findOne(deploymentDetailsRequest.getReleaseId(), new QuerySpec(BundleRelease.class));
            bundleRelease.setGitTag(deploymentDetailsRequest.getGitTag());
            bundleRelease.setCommitId(deploymentDetailsRequest.getCommitId());
            bundleRelease.setRemarks(StringUtils.isBlank(deploymentDetailsRequest.getRemarks()) ? "Successful" : deploymentDetailsRequest.getRemarks());
            bundleRelease.setStatus(deploymentDetailsRequest.getStatus() != null ? deploymentDetailsRequest.getStatus().value() : BundledReleaseStatus.FAILED.value());
            bundleReleaseRepository.save(bundleRelease);
            return bundleReleaseUtility.generateResponse(bundleRelease.getBundleReleaseAppVersions().size(), new BundleReleaseGenericResponse());
        } catch (Exception e) {
            throw new BundleReleaseGenericException(StringUtils.isBlank(e.getMessage()) ? SOME_EXCEPTION_OCCURRED : e.getMessage());
        }
    }
}