package com.mcd.restaurant.bundledrelease.controller;

import com.mcd.restaurant.bundledrelease.error.BundleNotFoundException;
import com.mcd.restaurant.bundledrelease.error.BundleReleaseGenericException;
import com.mcd.restaurant.bundledrelease.error.BundleReleaseInvalidRequestException;
import com.mcd.restaurant.common.ErrorCode;
import com.mcd.restaurant.common.ResponseView;
import com.mcd.restaurant.service.ApiLoggingService;
import io.crnk.core.engine.document.ErrorData;
import io.crnk.core.engine.document.ErrorDataBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.servlet.mvc.method.annotation.ResponseEntityExceptionHandler;

/**
 * ControllerAdvice responsible for handling error for bundleRelease module
 */
@ControllerAdvice
public class BundledReleaseControllerAdvice extends ResponseEntityExceptionHandler {

    @Autowired
    private ApiLoggingService apiLoggingService;

    @ExceptionHandler(BundleReleaseGenericException.class)
    @ResponseBody
    public ResponseEntity<ResponseView<Object>> handleBundleReleaseException(BundleReleaseGenericException ex) {
        String errorId = apiLoggingService.logWarn(ex);
        ErrorDataBuilder builder = ErrorData.builder()
                .setId(errorId)
                .setCode(ErrorCode.BUNDLE_RELEASE_EXCEPTION)
                .setStatus(Integer.toString(HttpStatus.INTERNAL_SERVER_ERROR.value()))
                .setDetail(ex.getMessage());
        ResponseView<Object> responseView = new ResponseView<>();
        responseView.addError(builder.build());
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(responseView);
    }

    @ExceptionHandler(BundleReleaseInvalidRequestException.class)
    @ResponseBody
    public ResponseEntity<ResponseView<Object>> handleInvalidRequest(BundleReleaseInvalidRequestException ex) {
        String errorId = apiLoggingService.logWarn(ex);
        ErrorDataBuilder builder = ErrorData.builder()
                .setId(errorId)
                .setCode(ErrorCode.INVALID_OPERATION)
                .setStatus(Integer.toString(HttpStatus.BAD_REQUEST.value()))
                .setDetail(ex.getMessage());
        ResponseView<Object> responseView = new ResponseView<>();
        responseView.addError(builder.build());
        return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(responseView);
    }

    @ExceptionHandler(BundleNotFoundException.class)
    @ResponseBody
    public ResponseEntity<ResponseView<Object>> handleBundleNotFound(BundleNotFoundException ex) {
        String errorId = apiLoggingService.logWarn(ex);
        ErrorDataBuilder builder = ErrorData.builder()
                .setId(errorId)
                .setCode(ErrorCode.BUNDLE_NOT_FOUND)
                .setStatus(Integer.toString(HttpStatus.BAD_REQUEST.value()))
                .setDetail(ex.getMessage());
        ResponseView<Object> responseView = new ResponseView<>();
        responseView.addError(builder.build());
        return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(responseView);
    }

}
