package com.mcd.restaurant.deployment;

import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.S3Object;
import com.mcd.restaurant.common.MapperUtils;
import com.mcd.restaurant.deployment.controller.view.beans.DownloadReportHelper;
import com.mcd.restaurant.deployment.controller.view.request.*;
import com.mcd.restaurant.deployment.controller.view.response.*;
import com.mcd.restaurant.deployment.error.ComponentNotApplicableError;
import com.mcd.restaurant.deployment.error.DeploymentBadRequestException;
import com.mcd.restaurant.deployment.error.DeploymentError;
import com.mcd.restaurant.deployment.service.*;
import com.mcd.restaurant.deployment.testUtils.DeploymentTestUtility;
import com.mcd.restaurant.model.*;
import com.mcd.restaurant.repository.*;
import com.mcd.restaurant.repository.bulk.ComponentBulkCRUDRepository;
import com.opencsv.CSVWriter;
import io.crnk.core.queryspec.QuerySpec;
import io.crnk.core.resource.list.DefaultResourceList;
import io.crnk.core.resource.list.ResourceList;
import io.restassured.internal.util.IOUtils;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.CsvSource;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;
import org.springframework.mock.web.MockMultipartFile;
import org.springframework.test.util.ReflectionTestUtils;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.*;

;

class DeployServiceTest {

    @Mock
    private DeploymentGroupRepository deploymentGroupRepositoryMock;
    @Mock
    private DeploymentHistoryRepository deploymentHistoryRepositoryMock;
    @Mock
    private ComponentBulkCRUDRepository componentBulkCRUDRepository;
    @Mock
    private PrePostDeploymentRepository prePostDeploymentRepositoryMock;
    @Mock
    private RestaurantRepository restaurantRepository;
    @Mock
    private ComponentRepository componentRepository;
    @Mock
    private DeploymentTagRepository deploymentTagRepository;
    @InjectMocks
    private DeployService deployService;
    @InjectMocks
    private DeploymentUtils deploymentUtils;
    @Mock
    private DeploymentUtils deploymentUtilsMock;
    @Mock
    private MapperUtils mapperUtilsMock;
    @InjectMocks
    private MapperUtils mapperUtils;
    @InjectMocks
    private DeploymentTestUtility deploymentUtilityTest;
    @Mock
    private MarketRepository marketRepository;
    @Mock
    private ComponentTypeRepository componentTypeRepositoryMock;
    @Mock
    private AmazonS3 s3Client;


    @BeforeEach
    void setUp() {
        MockitoAnnotations.initMocks(this);
    }

    /**
     * Test:    If any component is not applicable for all the target restaurant.
     * Result:  Should throw error {@link ComponentNotApplicableError}
     */
//    @Test
//    void assertComponentNotApplicableError() {
//        when(componentRepository.getFilteredDockerComponents(any(DeployComponentRequest.class), anySet()))
//                .thenReturn(mockComponentsAtRestaurant(List.of(1)));
//        when(deploymentGroupRepositoryMock.save(any(DeploymentGroup.class)))
//                .thenAnswer(i -> i.getArguments()[0]);
//
//
//        DeployRequest deployRequest = deployRequest();
//        assertThrows(ComponentNotApplicableError.class,
//                () -> deployService.saveAllDeployments(deployRequest));
//    }

//    @Test
//    void assertTestComponentNotApplicableError() {
//        DeployRequest deployRequest = deployRequest();
//        assertThrows(ComponentNotApplicableError.class,
//                () -> deployService.saveAllDeployments(deployRequest));
//    }


    /**
     * Test:    If creation of deployment group across Restaurant 2 components.
     * Result:  Should return 2 deployment group that is created for the target restaurant
     */
    @Test
    void saveAllDeploymentsSuccess() {
        ReflectionTestUtils.setField(deployService, "deploymentUtils", deploymentUtilsMock);
        when(componentRepository.getFilteredDockerComponents(any(DeployComponentRequest.class), anySet()))
                .thenReturn(mockComponentsAtRestaurant(List.of(1, 2)));
        when(deploymentGroupRepositoryMock.save(any(DeploymentGroup.class))).thenReturn(mockDeployGroup());
        when(componentBulkCRUDRepository.saveAll(Mockito.anyList())).thenAnswer(i -> i.getArguments()[0]);
        when(deploymentUtilsMock.fetchComponentType(Mockito.anyString())).thenReturn(mockComponentType());
        when(restaurantRepository.findAll(Mockito.any()))
                .thenReturn(mockRestaurantList(List.of(1, 2)));
        when(deploymentTagRepository.save(any(DeploymentTag.class))).thenReturn(mockRelease());
        DeployResponse deployResponse = deployService.saveAllDeployments(deployRequest());
        assertEquals(2, deployResponse.getDeploymentIds().size());
        verify(deploymentGroupRepositoryMock, times(1)).save(any(DeploymentGroup.class));
    }

    /**
     * Test:    If creation of deployment group across Restaurant 2 components.
     * Result:  Should return 2 deployment group that is created for the target restaurant
     */
    @Test
    void saveAllDeploymentsFailure() {
        ReflectionTestUtils.setField(deployService, "deploymentUtils", deploymentUtilsMock);
        when(componentRepository.getFilteredDockerComponents(any(DeployComponentRequest.class), anySet()))
                .thenReturn(mockComponentsAtRestaurant(List.of(1, 2)));
        when(deploymentGroupRepositoryMock.save(any(DeploymentGroup.class))).thenReturn(mockDeployGroup());
        when(componentBulkCRUDRepository.saveAll(Mockito.anyList())).thenAnswer(i -> i.getArguments()[0]);
        when(deploymentUtilsMock.fetchComponentType(Mockito.anyString())).thenReturn(mockComponentType());
        when(restaurantRepository.findAll(Mockito.any()))
                .thenReturn(mockRestaurantList(List.of(1)));
        when(deploymentTagRepository.save(any(DeploymentTag.class))).thenReturn(mockRelease());
        DeployRequest request = deployRequest();
        assertThrows(DeploymentBadRequestException.class, () -> deployService.saveAllDeployments(request));
    }


    @Test
    void getGroupForArchivalSuccess() {
        Long time = System.currentTimeMillis();
        QuerySpec querySpec = new QuerySpec(DeploymentGroup.class);
        ReflectionTestUtils.setField(deployService, "deploymentUtils", deploymentUtils);
        when(deploymentUtilsMock.prepareQuerySpecFilterForArchival(time, 100l, 0))
                .thenReturn(querySpec);
        when(deploymentGroupRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(getDeploymentGroupList());
        when(mapperUtilsMock.map(anyList(), any())).thenCallRealMethod();
        ReflectionTestUtils.setField(deployService, "mapperUtils", mapperUtils);
        ArchivalDataResponseDTO archivalDataResponseDTO = deployService.getDeploymentGroupForArchival(time, 100l, 0);
        assertEquals(1, archivalDataResponseDTO.getDeploymentGroupList().size());
    }

    @Test
    void deleteGroupAfterArchivalSuccess() {
        List<Integer> groupList = new ArrayList<>();
        groupList.add(3);
        List<Integer> tagList = new ArrayList<>();
        tagList.add(1);
        ReflectionTestUtils.setField(deployService, "deploymentUtils", deploymentUtilsMock);
        PostArchivalDeleteRequestDTO archivalDeleteRequestDTO = new PostArchivalDeleteRequestDTO();
        archivalDeleteRequestDTO.setDeploymentGroupId(groupList);
        archivalDeleteRequestDTO.setDeploymentTagId(tagList);
        Mockito.doNothing().when(deploymentUtilsMock).asyncDeletePOstArchival(Mockito.any());
        ArchivalDeleteResponse archivalDataResponseDTO = deployService.deleteDeploymentRecordsAfterArchival(archivalDeleteRequestDTO);
        assertEquals(true, archivalDataResponseDTO.getIsDeleted());
    }


    @Test
    void deleteGroupAfterArchivalNullListException() {
        PostArchivalDeleteRequestDTO archivalDeleteRequestDTO = new PostArchivalDeleteRequestDTO();
        assertThrows(DeploymentError.class, () -> deployService.deleteDeploymentRecordsAfterArchival(archivalDeleteRequestDTO));
    }

    @Test
    void deleteGroupAfterArchivalEmptyListException() {
        List<Integer> groupList = new ArrayList<>();
        List<Integer> tagList = new ArrayList<>();
        PostArchivalDeleteRequestDTO archivalDeleteRequestDTO = new PostArchivalDeleteRequestDTO();
        archivalDeleteRequestDTO.setDeploymentGroupId(groupList);
        archivalDeleteRequestDTO.setDeploymentTagId(tagList);
        assertThrows(DeploymentError.class, () -> deployService.deleteDeploymentRecordsAfterArchival(archivalDeleteRequestDTO));
    }

    private DeployRequest deployRequest() {
        DeployComponentRequest componentRequest1 = new DeployComponentRequest();
        componentRequest1.setComponentName("abc");

        componentRequest1.setUpdateFromVersion("1");
        componentRequest1.setUpdateToVersion("2");

        DeployComponentRequest componentRequest2 = new DeployComponentRequest();
        componentRequest2.setComponentName("abc");
        componentRequest2.setUpdateFromVersion("1");
        componentRequest2.setUpdateToVersion("2");

        DeployRequest deployRequest = new DeployRequest();
        deployRequest.setDeploymentType(DeploymentType.docker);
        deployRequest.setDeployComponents(List.of(componentRequest1, componentRequest2));
        deployRequest.setRestaurantIds(Set.of(1, 2));
        return deployRequest;
    }

    private ResourceList<Component> mockComponentsAtRestaurant(List<Integer> restaurants) {
        ComponentType type = new ComponentType();
        type.setName("docker");
        List<Component> components = restaurants.stream().map(it -> {
            Restaurant restaurant = new Restaurant();
            restaurant.setId(it);
            Component component = new Component();
            component.setRestaurant(restaurant);
            component.setType(type);
            return component;
        }).collect(Collectors.toList());
        return new DefaultResourceList<>(components, null, null);
    }

    private ResourceList<Restaurant> mockRestaurantList(List<Integer> restaurants) {
        List<Restaurant> restaurantsList = restaurants.stream().map(it -> {
            Restaurant restaurant = new Restaurant();
            restaurant.setId(it);
            return restaurant;
        }).collect(Collectors.toList());
        return new DefaultResourceList<>(restaurantsList, null, null);
    }

    @Test
    void getsuggestionListIDSuccess() {
        DeploymentGroup deploymentGroup = new DeploymentGroup();
        deploymentGroup.setId(12);
        DefaultResourceList<DeploymentGroup> resourceList = new DefaultResourceList<>();
        resourceList.add(deploymentGroup);
        ReflectionTestUtils.setField(deployService, "deploymentUtils", deploymentUtils);
        Mockito.when(deploymentGroupRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(resourceList);
        Mockito.when(deploymentUtilsMock.prepareQuerySpecFilterForSuggestion(Mockito.any(), Mockito.anyString(), Mockito.anyString())).thenReturn(new QuerySpec(DeploymentGroup.class));
        SuggestionResponseDTO suggestionResponseDTO = deployService.getSuggestionList(SuggestionType.ID, "12");
        assertEquals(1, suggestionResponseDTO.getSuggestionList().size());
    }

    @Test
    void getDeploymentDetailsGenericException() {
        assertThrows(DeploymentError.class, () -> deployService.getDeploymentDetails(2));
    }

    @Test
    void modifyComponentRecordsForRestaurantGenericException() {
        assertThrows(DeploymentError.class, () -> deployService.modifyComponentRecordsForRestaurant(null));
    }

    @Test
    void getDeploymentTypeTest() {
        assertEquals(null, DeploymentType.getDeploymentType("ABC"));
    }


    @Test
    void GetsuggestionListStoreNameSuccess() {
        Restaurant restaurant = new Restaurant();
        restaurant.setName("abc");
        DefaultResourceList<Restaurant> resourceList = new DefaultResourceList<>();
        resourceList.add(restaurant);
        ReflectionTestUtils.setField(deployService, "deploymentUtils", deploymentUtils);
        Mockito.when(restaurantRepository.findAll(Mockito.any(QuerySpec.class))).thenReturn(resourceList);
        Mockito.when(deploymentUtilsMock.prepareQuerySpecFilterForSuggestion(Mockito.any(), Mockito.anyString(), Mockito.anyString())).thenReturn(new QuerySpec(Restaurant.class));
        SuggestionResponseDTO suggestionResponseDTO = deployService.getSuggestionList(SuggestionType.STORE_NAME, "abc");
        assertEquals(1, suggestionResponseDTO.getSuggestionList().size());
    }

    @Test
    void GetsuggestionListDeploymentNameSuccess() {
        DeploymentTag deploymentTag = new DeploymentTag();
        deploymentTag.setName("abc");
        DefaultResourceList<DeploymentTag> resourceList = new DefaultResourceList<>();
        resourceList.add(deploymentTag);
        ReflectionTestUtils.setField(deployService, "deploymentUtils", deploymentUtils);
        Mockito.when(deploymentTagRepository.findAll(Mockito.any(QuerySpec.class))).thenReturn(resourceList);
        Mockito.when(deploymentUtilsMock.prepareQuerySpecFilterForSuggestion(Mockito.any(), Mockito.anyString(), Mockito.anyString())).thenReturn(new QuerySpec(DeploymentTag.class));
        SuggestionResponseDTO suggestionResponseDTO = deployService.getSuggestionList(SuggestionType.DEPLOYMENT_NAME, "abc");
        assertEquals(1, suggestionResponseDTO.getSuggestionList().size());
    }

    @Test
    void GetsuggestionListrestaurantNumberSuccess() {
        Restaurant restaurant = new Restaurant();
        restaurant.setName("123");
        DefaultResourceList<Restaurant> resourceList = new DefaultResourceList<>();
        resourceList.add(restaurant);
        ReflectionTestUtils.setField(deployService, "deploymentUtils", deploymentUtils);
        Mockito.when(restaurantRepository.findAll(Mockito.any(QuerySpec.class))).thenReturn(resourceList);
        Mockito.when(deploymentUtilsMock.prepareQuerySpecFilterForSuggestion(Mockito.any(), Mockito.anyString(), Mockito.anyString())).thenReturn(new QuerySpec(Restaurant.class));
        SuggestionResponseDTO suggestionResponseDTO = deployService.getSuggestionList(SuggestionType.RESTAURANT_NO, "abc");
        assertEquals(1, suggestionResponseDTO.getSuggestionList().size());
    }


    @ParameterizedTest
    @CsvSource({"DAY", "WEEK", "MONTH"})
    void prepareStatisticsForDeployGraphSuccess(DeployGraphSegment segment) {
        Long startTime = 1658707200000l;
        Long endTime = 1658790000000l;
        QuerySpec inProgressQuerySpec = deploymentUtilityTest.prepareQuerySpecFilterForGraph(DeploymentStatus.InProgress.value(), startTime, endTime);
        QuerySpec failedQuerySpec = deploymentUtilityTest.prepareQuerySpecFilterForGraph(DeploymentStatus.Failed.value(), startTime, endTime);
        QuerySpec cancelledQuerySpec = deploymentUtilityTest.prepareQuerySpecFilterForGraph(DeploymentStatus.Cancelled.value(), startTime, endTime);
        QuerySpec completedQuerySpec = deploymentUtilityTest.prepareQuerySpecFilterForGraph(DeploymentStatus.Completed.value(), startTime, endTime);
        Mockito.when(deploymentGroupRepositoryMock.prepareQuerySpecFilterForGraph(DeploymentStatus.InProgress.value(), startTime, endTime)).thenReturn(inProgressQuerySpec);
        Mockito.when(deploymentGroupRepositoryMock.prepareQuerySpecFilterForGraph(DeploymentStatus.Failed.value(), startTime, endTime)).thenReturn(failedQuerySpec);
        Mockito.when(deploymentGroupRepositoryMock.prepareQuerySpecFilterForGraph(DeploymentStatus.Cancelled.value(), startTime, endTime)).thenReturn(cancelledQuerySpec);
        Mockito.when(deploymentGroupRepositoryMock.prepareQuerySpecFilterForGraph(DeploymentStatus.Completed.value(), startTime, endTime)).thenReturn(completedQuerySpec);
        Mockito.when(deploymentGroupRepositoryMock.findAll(inProgressQuerySpec)).thenReturn(deploymentUtilityTest.getDeploymentGroupList(startTime));
        Mockito.when(deploymentGroupRepositoryMock.findAll(cancelledQuerySpec)).thenReturn(deploymentUtilityTest.getDeploymentGroupList(startTime));
        Mockito.when(deploymentGroupRepositoryMock.findAll(completedQuerySpec)).thenReturn(deploymentUtilityTest.getDeploymentGroupList(startTime));
        Mockito.when(deploymentGroupRepositoryMock.findAll(failedQuerySpec)).thenReturn(deploymentUtilityTest.getDeploymentGroupList(startTime));
        ReflectionTestUtils.setField(deployService, "deploymentUtils", deploymentUtils);
        Mockito.when(deploymentUtilsMock.analyseDeploymentResultsDay(Mockito.anyLong(), Mockito.anyLong())).thenCallRealMethod();
        DeployGraphResponseDTO deployGraphResponseDTO = deployService.prepareStatisticsForDeployGraph(startTime, endTime, segment);
        assertNotNull(deployGraphResponseDTO);
    }


    @Test
    void prepareStatisticsForDeployGraphFailure() {
        Long startTime = 1658707200000l;
        Long endTime = 1658790000000l;
        ReflectionTestUtils.setField(deployService, "deploymentUtils", deploymentUtils);
        Mockito.when(deploymentUtilsMock.analyseDeploymentResultsDay(Mockito.anyLong(), Mockito.anyLong())).thenCallRealMethod();
        assertThrows(DeploymentError.class, () -> deployService.prepareStatisticsForDeployGraph(startTime, endTime, DeployGraphSegment.DAY));

    }

    @Test
    void updateDeploymentHistoryStatusMissingRequest() {
        UpdateDeploymentStatusRequestDTO requestDTO1 = buildUpdateDeploymentStatusRequestDTO(null, "abc", "1", "NONE", 1, DeploymentStatus.InProgress);
        UpdateDeploymentStatusRequestDTO requestDTO2 = buildUpdateDeploymentStatusRequestDTO("abc", null, "1", "NONE", 1, DeploymentStatus.InProgress);
        UpdateDeploymentStatusRequestDTO requestDTO3 = buildUpdateDeploymentStatusRequestDTO("abc", "abc", null, "NONE", 1, DeploymentStatus.InProgress);
        UpdateDeploymentStatusRequestDTO requestDTO4 = buildUpdateDeploymentStatusRequestDTO("abc", "abc", "1", null, 1, DeploymentStatus.InProgress);
        UpdateDeploymentStatusRequestDTO requestDTO5 = buildUpdateDeploymentStatusRequestDTO("abc", "abc", "1", "NONE", null, DeploymentStatus.InProgress);
        UpdateDeploymentStatusRequestDTO requestDTO6 = buildUpdateDeploymentStatusRequestDTO("abc", "abc", "1", "NONE", 1, null);
        ReflectionTestUtils.setField(deployService, "deploymentUtils", deploymentUtils);
        for (UpdateDeploymentStatusRequestDTO requestDTO : Stream.of(requestDTO1, requestDTO2, requestDTO3, requestDTO4, requestDTO5, requestDTO6).collect(Collectors.toList())) {
            assertThrows(DeploymentBadRequestException.class, () -> deployService.updateDeploymentHistoryStatus(requestDTO));
        }
    }

    @Test
    void updateDeploymentHistoryStatusNoMarket() {
        UpdateDeploymentStatusRequestDTO requestDTO = buildUpdateDeploymentStatusRequestDTO("Abc", "abc", "1", "NONE", 1, DeploymentStatus.InProgress);
        ReflectionTestUtils.setField(deployService, "deploymentUtils", deploymentUtils);
        ReflectionTestUtils.setField(deployService, "marketRepository", marketRepository);
        Mockito.when(marketRepository.findAll(Mockito.any(QuerySpec.class))).thenReturn(new DefaultResourceList<>());
        Mockito.when(marketRepository.prepareQuerySpecFilterForFilteringMarketByName(Mockito.anyString())).thenCallRealMethod();
        assertThrows(DeploymentBadRequestException.class, () -> deployService.updateDeploymentHistoryStatus(requestDTO));

    }

    @Test
    void updateDeploymentHistoryStatusGenericException() {
        UpdateDeploymentStatusRequestDTO requestDTO = buildUpdateDeploymentStatusRequestDTO("Abc", "abc", "1", "NONE", 1, DeploymentStatus.InProgress);
        assertThrows(DeploymentError.class, () -> deployService.updateDeploymentHistoryStatus(requestDTO));

    }

    @Test
    void modifyComponentRecordsForRestaurantMissingrequest() {
        UpdateComponentRequestDTO requestDTO1 = buildUpdateComponentRequestDTO(null, "abc", "1", "NONE");
        UpdateComponentRequestDTO requestDTO2 = buildUpdateComponentRequestDTO("Abc", null, "1", "NONE");
        UpdateComponentRequestDTO requestDTO3 = buildUpdateComponentRequestDTO("Abc", "abc", null, "NONE");
        UpdateComponentRequestDTO requestDTO4 = buildUpdateComponentRequestDTO("Abc", "abc", "1", null);
        for (UpdateComponentRequestDTO requestDTO : Stream.of(requestDTO1, requestDTO2, requestDTO3, requestDTO4).collect(Collectors.toList())) {
            assertThrows(DeploymentBadRequestException.class, () -> deployService.modifyComponentRecordsForRestaurant(requestDTO));
        }

    }

    @Test
    void modifyComponentRecordsForRestaurantNoMarket() {
        UpdateComponentRequestDTO requestDTO = buildUpdateComponentRequestDTO("Abc", "abc", "1", "Abc");
        ReflectionTestUtils.setField(deployService, "marketRepository", marketRepository);
        ReflectionTestUtils.setField(deployService, "deploymentUtils", deploymentUtils);
        Mockito.when(marketRepository.findAll(Mockito.any(QuerySpec.class))).thenReturn(new DefaultResourceList<>());
        Mockito.when(marketRepository.prepareQuerySpecFilterForFilteringMarketByName(Mockito.anyString())).thenCallRealMethod();
        assertThrows(DeploymentBadRequestException.class, () -> deployService.modifyComponentRecordsForRestaurant(requestDTO));

    }

    @Test
    void modifyComponentRecordsForRestaurantNoRestaurant() {
        UpdateComponentRequestDTO requestDTO = buildUpdateComponentRequestDTO("Abc", "abc", "1", "Abc");
        ReflectionTestUtils.setField(deployService, "marketRepository", marketRepository);
        ReflectionTestUtils.setField(deployService, "deploymentUtils", deploymentUtils);
        Mockito.when(marketRepository.findAll(Mockito.any(QuerySpec.class))).thenReturn(getMockMarket());
        Mockito.when(marketRepository.prepareQuerySpecFilterForFilteringMarketByName(Mockito.anyString())).thenCallRealMethod();
        Mockito.when(restaurantRepository.findAll(Mockito.any(QuerySpec.class))).thenReturn(new DefaultResourceList<>());
        Mockito.when(restaurantRepository.prepareQuerySpecFilterForRestaurantByRestaurantNumberAndMarketId(Mockito.anyString(), Mockito.anyInt())).thenCallRealMethod();
        assertThrows(DeploymentBadRequestException.class, () -> deployService.modifyComponentRecordsForRestaurant(requestDTO));

    }

    @Test
    void updateDeploymentHistoryStatusNoRestaurant() {
        UpdateDeploymentStatusRequestDTO requestDTO = buildUpdateDeploymentStatusRequestDTO("Abc", "abc", "1", "NONE", 1, DeploymentStatus.InProgress);
        ReflectionTestUtils.setField(deployService, "deploymentUtils", deploymentUtils);
        ReflectionTestUtils.setField(deployService, "marketRepository", marketRepository);
        ReflectionTestUtils.setField(deployService, "restaurantRepository", restaurantRepository);
        Mockito.when(marketRepository.findAll(Mockito.any(QuerySpec.class))).thenReturn(getMockMarket());
        Mockito.when(marketRepository.prepareQuerySpecFilterForFilteringMarketByName(Mockito.anyString())).thenCallRealMethod();
        Mockito.when(restaurantRepository.findAll(Mockito.any(QuerySpec.class))).thenReturn(new DefaultResourceList<>());
        Mockito.when(restaurantRepository.prepareQuerySpecFilterForRestaurantByRestaurantNumberAndMarketId(Mockito.anyString(), Mockito.anyInt())).thenCallRealMethod();
        assertThrows(DeploymentBadRequestException.class, () -> deployService.updateDeploymentHistoryStatus(requestDTO));

    }

    @Test
    void updateDeploymentHistoryStatusNoDeploymentHistory() {
        UpdateDeploymentStatusRequestDTO requestDTO = buildUpdateDeploymentStatusRequestDTO("Abc", "abc", "1", "NONE", 1, DeploymentStatus.InProgress);
        ReflectionTestUtils.setField(deployService, "deploymentUtils", deploymentUtils);
        ReflectionTestUtils.setField(deployService, "marketRepository", marketRepository);
        ReflectionTestUtils.setField(deployService, "restaurantRepository", restaurantRepository);
        ReflectionTestUtils.setField(deployService, "deploymentHistoryRepository", deploymentHistoryRepositoryMock);
        Mockito.when(marketRepository.findAll(Mockito.any(QuerySpec.class))).thenReturn(getMockMarket());
        Mockito.when(marketRepository.prepareQuerySpecFilterForFilteringMarketByName(Mockito.anyString())).thenCallRealMethod();
        Mockito.when(restaurantRepository.findAll(Mockito.any(QuerySpec.class))).thenReturn(getMockRestaurant());
        Mockito.when(restaurantRepository.prepareQuerySpecFilterForRestaurantByRestaurantNumberAndMarketId(Mockito.anyString(), Mockito.anyInt())).thenCallRealMethod();
        Mockito.when(deploymentHistoryRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(new DefaultResourceList<>());
        Mockito.when(deploymentHistoryRepositoryMock.getHistoryByGroupIdAndRestaurantIdAndApplicationNameAndApplicationVersion(Mockito.anyInt(), Mockito.anyInt(), Mockito.anyString(), Mockito.anyString())).thenCallRealMethod();
        assertThrows(DeploymentBadRequestException.class, () -> deployService.updateDeploymentHistoryStatus(requestDTO));

    }

    @Test
    void updateDeploymentHistoryStatusSuccess() {
        UpdateDeploymentStatusRequestDTO requestDTO = buildUpdateDeploymentStatusRequestDTO("Abc", "abc", "1", "NONE", 1, DeploymentStatus.InProgress);
        ReflectionTestUtils.setField(deployService, "deploymentUtils", deploymentUtils);
        ReflectionTestUtils.setField(deployService, "marketRepository", marketRepository);
        ReflectionTestUtils.setField(deployService, "restaurantRepository", restaurantRepository);
        ReflectionTestUtils.setField(deployService, "deploymentHistoryRepository", deploymentHistoryRepositoryMock);
        Mockito.when(marketRepository.findAll(Mockito.any(QuerySpec.class))).thenReturn(getMockMarket());
        Mockito.when(marketRepository.prepareQuerySpecFilterForFilteringMarketByName(Mockito.anyString())).thenCallRealMethod();
        Mockito.when(restaurantRepository.findAll(Mockito.any(QuerySpec.class))).thenReturn(getMockRestaurant());
        Mockito.when(restaurantRepository.prepareQuerySpecFilterForRestaurantByRestaurantNumberAndMarketId(Mockito.anyString(), Mockito.anyInt())).thenCallRealMethod();
        Mockito.when(deploymentHistoryRepositoryMock.findAll(Mockito.any(QuerySpec.class))).thenReturn(getMockDeploymentHistory());
        Mockito.when(deploymentHistoryRepositoryMock.save(Mockito.any(DeploymentHistory.class))).then(i -> i.getArguments()[0]);
        Mockito.when(deploymentHistoryRepositoryMock.getHistoryByGroupIdAndRestaurantIdAndApplicationNameAndApplicationVersion(Mockito.anyInt(), Mockito.anyInt(), Mockito.anyString(), Mockito.anyString())).thenCallRealMethod();
        StatusChangeResponse response = deployService.updateDeploymentHistoryStatus(requestDTO);
        assertNotNull(response);
        assertEquals(requestDTO.getStatus(), response.getStatus());

    }

    private DefaultResourceList<Market> getMockMarket() {
        Market market = new Market();
        market.setName("US");
        DefaultResourceList<Market> markets = new DefaultResourceList<>();
        markets.add(market);
        return markets;
    }

    private DefaultResourceList<Restaurant> getMockRestaurant() {
        Restaurant restaurant = new Restaurant();
        restaurant.setName("1");
        DefaultResourceList<Restaurant> restaurants = new DefaultResourceList<>();
        restaurants.add(restaurant);
        return restaurants;
    }

    private DefaultResourceList<DeploymentHistory> getMockDeploymentHistory() {
        DeploymentHistory history = new DeploymentHistory();
        DefaultResourceList<DeploymentHistory> histories = new DefaultResourceList<>();
        histories.add(history);
        return histories;
    }

    private UpdateDeploymentStatusRequestDTO buildUpdateDeploymentStatusRequestDTO(String applicationName, String applicationVersion, String restaurantNo, String marketName, Integer deploymentGroupId, DeploymentStatus status) {
        return UpdateDeploymentStatusRequestDTO.builder().applicationName(applicationName).applicationVersion(applicationVersion).deploymentGroupId(deploymentGroupId).marketName(marketName).restaurantNo(restaurantNo).status(status).build();
    }

    private UpdateComponentRequestDTO buildUpdateComponentRequestDTO(String applicationName, String applicationVersion, String restaurantNo, String marketName) {
        return UpdateComponentRequestDTO.builder().applicationName(applicationName).applicationVersion(applicationVersion).restaurantNo(restaurantNo).marketName(marketName).build();
    }

    private DeploymentGroup mockDeployGroup() {
        DeploymentGroup deploymentGroup = new DeploymentGroup();
        deploymentGroup.setIsPreCheckFlow("false");
        deploymentGroup.setStatus("Test");
        deploymentGroup.setDeploymentTag(new DeploymentTag());
        return deploymentGroup;
    }

    private List<DeploymentHistory> mockDeploymentHistory() {
        return Stream.of(new DeploymentHistory()).collect(Collectors.toList());
    }

    private ComponentType mockComponentType() {
        ComponentType type = new ComponentType();
        type.setId(5);
        return type;

    }

    private DefaultResourceList<DeploymentGroup> getDeploymentGroupList() {
        DeploymentGroup group = new DeploymentGroup();
        group.setId(1);
        Component component = new Component();
        component.setId(1);
        DeploymentHistory history = new DeploymentHistory();
        history.setId(1);
        history.setComponent(component);
        history.setRestaurant(getMockRestaurant().get(0));
        history.setDeployment_group(group);
        history.setProductVersions(ProductVersions.builder().id(1).build());
        group.setDeploymentHistory(Stream.of(history).collect(Collectors.toList()));
        DefaultResourceList<DeploymentGroup> groups = new DefaultResourceList<>();
        groups.add(group);
        return groups;
    }

    private DeploymentTag mockRelease() {
        return new DeploymentTag();
    }

    private BulkInsertComponentRequestDTO buildBulkInsertComponentRequestDTO(String marketName, String storeName, List<ComponentDetailRequestDTO> component) {
        return BulkInsertComponentRequestDTO.builder()
                .marketName(marketName)
                .storeName(storeName)
                .components(component).build();

    }

    @Test
    void insertComponentDataMissingRequest() {

        List<ComponentDetailRequestDTO> comp = new ArrayList<>();
        comp.add(getComponentDetailRequestDTO("svcs-restaurant-assets"));
        List<ComponentDetailRequestDTO> comp1 = new ArrayList<>();
        comp1.add(getComponentDetailRequestDTO(" "));
        BulkInsertComponentRequestDTO requestDTO1 = buildBulkInsertComponentRequestDTO(null, "abc", new ArrayList<>());
        BulkInsertComponentRequestDTO requestDTO2 = buildBulkInsertComponentRequestDTO("abc", null, comp);
        BulkInsertComponentRequestDTO requestDTO3 = buildBulkInsertComponentRequestDTO("abc", "abc", null);
        BulkInsertComponentRequestDTO requestDTO4 = buildBulkInsertComponentRequestDTO(null, null, null);
        BulkInsertComponentRequestDTO requestDTO5 = buildBulkInsertComponentRequestDTO("abc", "abc", comp1);
        ReflectionTestUtils.setField(deployService, "deploymentUtils", deploymentUtils);
        for (BulkInsertComponentRequestDTO requestDTO : Stream.of(requestDTO1, requestDTO2, requestDTO3, requestDTO4, requestDTO5).collect(Collectors.toList())) {
            assertThrows(DeploymentBadRequestException.class, () -> deployService.insertComponentData(requestDTO));
        }
    }

    @Test
    void insertComponentDataNoMarket() {
        List<ComponentDetailRequestDTO> comp = new ArrayList<>();
        comp.add(getComponentDetailRequestDTO("svcs-restaurant-assets"));
        BulkInsertComponentRequestDTO requestDTO = buildBulkInsertComponentRequestDTO("abc", "abc", comp);
        ReflectionTestUtils.setField(deployService, "deploymentUtils", deploymentUtils);
        ReflectionTestUtils.setField(deployService, "marketRepository", marketRepository);
        Mockito.when(marketRepository.findAll(Mockito.any(QuerySpec.class))).thenReturn(new DefaultResourceList<>());
        Mockito.when(marketRepository.prepareQuerySpecFilterForFilteringMarketByName(Mockito.anyString())).thenCallRealMethod();
        assertThrows(DeploymentBadRequestException.class, () -> deployService.insertComponentData(requestDTO));
    }

    @Test
    void insertComponentDataNoRestaurant() {
        List<ComponentDetailRequestDTO> comp = new ArrayList<>();
        comp.add(getComponentDetailRequestDTO("abc"));
        BulkInsertComponentRequestDTO requestDTO = buildBulkInsertComponentRequestDTO("abc", "abc", comp);
        ReflectionTestUtils.setField(deployService, "deploymentUtils", deploymentUtils);
        ReflectionTestUtils.setField(deployService, "marketRepository", marketRepository);
        ReflectionTestUtils.setField(deployService, "restaurantRepository", restaurantRepository);
        Mockito.when(marketRepository.findAll(Mockito.any(QuerySpec.class))).thenReturn(getMockMarket());
        Mockito.when(marketRepository.prepareQuerySpecFilterForFilteringMarketByName(Mockito.anyString())).thenCallRealMethod();
        Mockito.when(restaurantRepository.findAll(Mockito.any(QuerySpec.class))).thenReturn(new DefaultResourceList<>());
        Mockito.when(restaurantRepository.prepareQuerySpecFilterForRestaurantByRestaurantNumberAndMarketId(Mockito.anyString(), Mockito.anyInt())).thenCallRealMethod();
        assertThrows(DeploymentBadRequestException.class, () -> deployService.insertComponentData(requestDTO));
    }

    @Test
    void insertComponentDataDuplicateComponentFailure() {
        List<ComponentDetailRequestDTO> comp = new ArrayList<>();
        comp.add(getComponentDetailRequestDTO("abc"));
        comp.add(getComponentDetailRequestDTO("def"));
        BulkInsertComponentRequestDTO requestDTO = buildBulkInsertComponentRequestDTO("abc", "abc", comp);
        ResourceList<Component> list = new DefaultResourceList<>();
        list.add(Component.builder().build());
        ReflectionTestUtils.setField(deployService, "deploymentUtils", deploymentUtils);
        ReflectionTestUtils.setField(deployService, "marketRepository", marketRepository);
        ReflectionTestUtils.setField(deployService, "restaurantRepository", restaurantRepository);
        ReflectionTestUtils.setField(deployService, "componentRepository", componentRepository);
        Mockito.when(marketRepository.findAll(Mockito.any(QuerySpec.class))).thenReturn(getMockMarket());
        Mockito.when(marketRepository.prepareQuerySpecFilterForFilteringMarketByName(Mockito.anyString())).thenCallRealMethod();
        Mockito.when(restaurantRepository.findAll(Mockito.any(QuerySpec.class))).thenReturn(new DefaultResourceList<>());
        Mockito.when(restaurantRepository.prepareQuerySpecFilterForRestaurantByRestaurantNumberAndMarketId(Mockito.anyString(), Mockito.anyInt())).thenCallRealMethod();
        Mockito.when(componentRepository.getAllComponentByRestaurant(Mockito.anyInt(), Mockito.anyList())).thenReturn(list);
        assertThrows(DeploymentBadRequestException.class, () -> deployService.insertComponentData(requestDTO));
    }

    @Test
    void insertComponentDataSuccess() {
        BulkInsertComponentRequestDTO componentDTO = new BulkInsertComponentRequestDTO();
        ComponentDetailRequestDTO requestDTO = new ComponentDetailRequestDTO();
        requestDTO.setApplicationName("AIDT-app");
        requestDTO.setApplicationVersion("APX");
        componentDTO.setMarketName("Us");
        componentDTO.setStoreName("9918");
        componentDTO.setComponents(Stream.of(requestDTO).collect(Collectors.toList()));
        ReflectionTestUtils.setField(deployService, "marketRepository", marketRepository);
        ReflectionTestUtils.setField(deployService, "restaurantRepository", restaurantRepository);
        ReflectionTestUtils.setField(deploymentUtils, "mapperUtils", mapperUtils);
        ReflectionTestUtils.setField(deployService, "deploymentUtils", deploymentUtils);
        Mockito.when(marketRepository.findAll(Mockito.any(QuerySpec.class))).thenReturn(getMockMarket());
        Mockito.when(marketRepository.prepareQuerySpecFilterForFilteringMarketByName(Mockito.anyString())).thenCallRealMethod();
        Mockito.when(restaurantRepository.findAll(Mockito.any(QuerySpec.class))).thenReturn(getMockRestaurant());
        Mockito.when(restaurantRepository.prepareQuerySpecFilterForRestaurantByRestaurantNumberAndMarketId(Mockito.anyString(), Mockito.anyInt())).thenCallRealMethod();
        when(componentTypeRepositoryMock.getComponentTypeByName(Mockito.anyString())).thenReturn(mockComponentTypeList());
        Mockito.when(componentBulkCRUDRepository.saveAll(Mockito.anyList())).thenAnswer(i -> i.getArguments()[0]);
        BulkInsertComponentResponseDTO responseDDTO = deployService.insertComponentData(componentDTO);
        Assertions.assertEquals(componentDTO.getComponents().size(), responseDDTO.getComponents().size());
    }

    @Test
    void downloadFileSuccess() throws IOException {
        ReflectionTestUtils.setField(deployService, "deploymentUtils", deploymentUtils);
        ReflectionTestUtils.setField(deployService, "s3Client", s3Client);
        Mockito.when(prePostDeploymentRepositoryMock.findOne(Mockito.anyInt(),Mockito.any())).thenReturn(PrePostDeploymentModel.builder().reportPath("report.zip").status(DeploymentStatus.Completed.value()).build());
        MultipartFile file=getMultipartValidFileFors3DownloadTest();
        S3Object s3ObjectInputStream=new S3Object();
        s3ObjectInputStream.setObjectContent(file.getInputStream());
        Mockito.when(s3Client.getObject(Mockito.anyString(),Mockito.anyString())).thenReturn(s3ObjectInputStream);
        DownloadReportHelper helper=deployService.downloadFile(7);
        Assertions.assertNotNull(helper);
    }


    public MultipartFile getMultipartValidFileFors3DownloadTest() throws IOException {
        File storeNetwork = new File("store-network.csv");
        FileWriter outputfile = new FileWriter(storeNetwork);
        CSVWriter writer = new CSVWriter(outputfile);
        List<String[]> data = new ArrayList<>();
        data.add(new String[]{"National Store #", "Service-Network Segment"});
        data.add(new String[]{"1", "1:12:13:14"});
        writer.writeAll(data);
        writer.close();
        storeNetwork.deleteOnExit();
        return new MockMultipartFile("storenetwork",
                storeNetwork.getName(), "text/csv", IOUtils.toByteArray(new FileInputStream(storeNetwork)));

    }

    private ResourceList<ComponentType> mockComponentTypeList() {
        DefaultResourceList<ComponentType> types = new DefaultResourceList<>();
        types.add(mockComponentType());
        return types;
    }


    @Test
    void checkDuplicateComponentTestException() {
        List<ComponentDetailRequestDTO> comp = new ArrayList<>();
        comp.add(getComponentDetailRequestDTO("abc"));
        comp.add(getComponentDetailRequestDTO("abc"));
        BulkInsertComponentRequestDTO test = buildBulkInsertComponentRequestDTO("abc", "abc", comp);
        Restaurant restaurant = restaurantMock(5, "abc");
        List<String> components = Stream.of("abc", "abc").collect(Collectors.toList());
        assertThrows(DeploymentBadRequestException.class, () -> deploymentUtils.fetchExistingComponents(components, restaurant));

    }
    @Test
    void checkGetEnv(){
        Map<String,String> envMap=new HashMap<>();
        envMap.put("524430043955","prod");
        envMap.put("593265675765","int");
        envMap.put("688810906228","stg");
        envMap.put("xyz","dev");
        for(Map.Entry<String, String> entry : envMap.entrySet()){
            String env=deploymentUtils.getEnvironment(entry.getKey());
            Assertions.assertEquals(entry.getValue(),env);
        }
    }
    @Test
    void checkGetEnvNullAccountId(){
            String env=deploymentUtils.getEnvironment(null);
            Assertions.assertEquals("dev",env);
    }
    public ComponentDetailRequestDTO getComponentDetailRequestDTO(String applicationName) {
        return ComponentDetailRequestDTO.builder().applicationName(applicationName)
                .applicationVersion("1")
                .installDate(null)
                .build();
    }

    public Restaurant restaurantMock(int id, String Name) {
        Restaurant restaurant = new Restaurant();
        restaurant.setId(id);
        restaurant.setName(Name);
        return restaurant;
    }
}
