package com.mcd.restaurant.dashboard;

import com.mcd.restaurant.dashboard.controller.StoreDashboardController;
import com.mcd.restaurant.dashboard.controller.view.response.BREHierarchyNodeDataUploadResponseDTO;
import com.mcd.restaurant.dashboard.controller.view.response.BREStoreDemographicUploadResponseDTO;
import com.mcd.restaurant.dashboard.controller.view.response.BREStoreIPAddressResponseDTO;
import com.mcd.restaurant.dashboard.controller.view.response.DashboardDetailResponseDTO;
import com.mcd.restaurant.dashboard.service.StoreDashboardService;
import com.mcd.restaurant.dashboard.testUtils.StoreDashboardTestUtils;
import io.restassured.RestAssured;
import io.restassured.builder.RequestSpecBuilder;
import io.restassured.specification.RequestSpecification;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.web.server.LocalServerPort;
import org.springframework.restdocs.RestDocumentationContextProvider;
import org.springframework.restdocs.RestDocumentationExtension;
import org.springframework.restdocs.operation.preprocess.Preprocessors;
import org.springframework.restdocs.restassured3.RestAssuredRestDocumentation;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit.jupiter.SpringExtension;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static org.hamcrest.CoreMatchers.*;

@ExtendWith({SpringExtension.class, RestDocumentationExtension.class})
@SpringBootTest(webEnvironment = SpringBootTest.WebEnvironment.RANDOM_PORT)
@ActiveProfiles("test")
class StoreDashboardControllerTest {
    @LocalServerPort
    int port;

    private RequestSpecification spec;
    @InjectMocks
    private StoreDashboardTestUtils storeDashboardTestUtils;
    @Mock
    private StoreDashboardService storeDashboardServiceMock;
    @InjectMocks
    private StoreDashboardController storeDashboardController;

    @BeforeEach
    public void setUp(RestDocumentationContextProvider restDocumentation) {
        RestAssured.port = port;

        this.spec = new RequestSpecBuilder()
                .addFilter(RestAssuredRestDocumentation.documentationConfiguration(restDocumentation)
                        .operationPreprocessors().withResponseDefaults(Preprocessors.prettyPrint()))
                .build();
    }

    @Test
    void getStoreDetailsError() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .get("/store-dashboard/details/2")
                .then().assertThat().statusCode(is(500));
    }

    @Test
    void addIpaddressToStoreSuccess() throws IOException {
        MultipartFile file = storeDashboardTestUtils.getMultipartValidFileForIpAddress();
        String marketName = "US Country Office";
        Mockito.when(storeDashboardServiceMock.updateStoreHierarchyNodeWithIpAddress(marketName, file)).thenReturn(new BREStoreIPAddressResponseDTO());
        Assertions.assertNotNull(storeDashboardController.addIpaddressToStore(marketName, file));
    }

    @Test
    void getStoreDetailsSuccess() {
        Mockito.when(storeDashboardServiceMock.getStoreDetails(Mockito.anyInt())).thenReturn(new DashboardDetailResponseDTO());
        Assertions.assertNotNull(storeDashboardController.getStoreDetails(1));
    }

    @Test
    void onBoardDataToBREHierarchyNode() throws IOException {
        MultipartFile file = storeDashboardTestUtils.getMultipartValidFileForHierarchyStructure();
        Mockito.when(storeDashboardServiceMock.uploadDataToDataBase(file)).thenReturn(new BREHierarchyNodeDataUploadResponseDTO());
        Assertions.assertNotNull(storeDashboardController.onBoardDataToBREHierarchyNode(file));
    }

    @Test
    void onBoardDemographicDataToBREHierarchyNode() throws IOException {
        MultipartFile file = storeDashboardTestUtils.getMultipartValidFileForDemoGraphicStructure();
        Mockito.when(storeDashboardServiceMock.uploadDemographicDataToDataBase(file)).thenReturn(new BREStoreDemographicUploadResponseDTO());
        Assertions.assertNotNull(storeDashboardController.onBoardDemographicDataToBREHierarchyNode(file));
    }

    @Test
    void getHierarchyFilterDataSuccess() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .get("/store-dashboard/hierarchy-filter")
                .then().assertThat().statusCode(is(200));
    }

    @Test
    void mapRestaurantwithHierarchyNodeSuccess() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .patch("/store-dashboard/restaurant-data/mapping?actualMarketName=US&dbMarketId=1")
                .then().assertThat().statusCode(is(200));
    }

    @Test
    void patchUpdateAOTStatusSucccess() {
        String jsonBody = "{\n" +
                "    \"restaurantNo\":\"101\",\n" +
                "    \"market\":\"US\", \n" +
                "    \"aotStatus\":\"8002\"\n" +
                "}";
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .patch("/store-dashboard/status/update")
                .then().assertThat().statusCode(is(200));
    }

    @Test
    void patchUpdateAOTStatusMissingParameter() {
        String jsonBody = "{\n" +
                "   \n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .patch("/store-dashboard/status/update")
                .then().assertThat().statusCode(is(400));
    }

    @Test
    void patchUpdateAOTStatusInvalidStatus() {
        String jsonBody = "{\n" +
                "    \"restaurantNo\":\"101\",\n" +
                "    \"market\":\"US\"\n" +
                "    \"aotStatus\":\"12\"\n" +
                "}";

        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .patch("/store-dashboard/status/update")
                .then().assertThat().statusCode(is(400));
    }

    @Test
    void fetchDemographicDetailsSuccess() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .get("/store-dashboard/demographic/data?hierarchyLevel=REGION&marketName=US")
                .then().assertThat().statusCode(is(200));
    }

    @Test
    void postAddHarwareDetailForStoreSuccess() {
        String jsonBodyUniqueRecord = "{\n" +
                "    \"restaurantNo\": \"101\",\n" +
                "    \"marketName\": \"US\",\n" +
                "    \"hardwareDetails\": [\n" +
                "        {\n" +
                "            \"serialNumber\": \"serialNumber\",\n" +
                "            \"macAddress\": \"macAddress\",\n" +
                "            \"modelNumber\": \"modelNumber\",\n" +
                "            \"nodeId\": \"nodeId\",\n" +
                "            \"manufacturer\":\"manufacturer\",\n" +
                "            \"modelName\":\"modelName\",\n" +
                "            \"version\":\"version\",\n" +
                "            \"family\":\"family\",\n" +
                "            \"osType\":\"osType\",\n" +
                "            \"osVersion\":\"osVersion\"\n" +
                "        }\n" +
                "    ]\n" +
                "}";
        String jsonBodyAlreadyExistRecord = "{\n" +
                "    \"restaurantNo\": \"101\",\n" +
                "    \"marketName\": \"US\",\n" +
                "    \"hardwareDetails\": [\n" +
                "        {\n" +
                "            \"serialNumber\": \"serialNumber\",\n" +
                "            \"macAddress\": \"macAddress\",\n" +
                "            \"modelNumber\": \"modelNumber\",\n" +
                "            \"nodeId\": \"3\",\n" +
                "            \"manufacturer\":\"manufacturer\",\n" +
                "            \"modelName\":\"modelName\",\n" +
                "            \"version\":\"version\",\n" +
                "            \"family\":\"family\",\n" +
                "            \"osType\":\"osType\",\n" +
                "            \"osVersion\":\"osVersion\"\n" +
                "        }\n" +
                "    ]\n" +
                "}";
        String jsonBodyForUpdateRecords = "{\n" +
                "    \"restaurantNo\": \"101\",\n" +
                "    \"marketName\": \"US\",\n" +
                "    \"hardwareDetails\": [\n" +
                "        {\n" +
                "            \"serialNumber\": \"serialNumber\",\n" +
                "            \"macAddress\": \"mac_Address\",\n" +
                "            \"modelNumber\": \"modelNumber\",\n" +
                "            \"nodeId\": \"3\",\n" +
                "            \"manufacturer\":\"manufacturer\",\n" +
                "            \"modelName\":\"modelName\",\n" +
                "            \"version\":\"version\",\n" +
                "            \"family\":\"family\",\n" +
                "            \"osType\":\"osType\",\n" +
                "            \"osVersion\":\"osVersion\"\n" +
                "        }\n" +
                "    ]\n" +
                "}";
        for (String jsonBody : Stream.of(jsonBodyUniqueRecord, jsonBodyAlreadyExistRecord, jsonBodyForUpdateRecords).collect(Collectors.toList())) {
            RestAssured.given(this.spec)
                    .accept("application/json")
                    .contentType("application/json")
                    .body(jsonBody)
                    .when()
                    .post("/store-dashboard/hardwaredetail")
                    .then().assertThat().statusCode(is(201));
        }
    }

    @Test
    void postAddHarwareDetailForStoreInvalidStoreMarket() {
        String jsonBodyInvalidStoreMarket = "{\n" +
                "    \"restaurantNo\": \"101\",\n" +
                "    \"marketName\": \"USP\",\n" +
                "    \"hardwareDetails\": [\n" +
                "        {\n" +
                "            \"serialNumber\": \"serialNumber\",\n" +
                "            \"macAddress\": \"macAddress\",\n" +
                "            \"modelNumber\": \"modelNumber\",\n" +
                "            \"nodeId\": \"nodeId\",\n" +
                "            \"manufacturer\":\"manufacturer\",\n" +
                "            \"modelName\":\"modelName\",\n" +
                "            \"version\":\"version\",\n" +
                "            \"family\":\"family\",\n" +
                "            \"osType\":\"osType\",\n" +
                "            \"osVersion\":\"osVersion\"\n" +
                "        }\n" +
                "    ]\n" +
                "}";
        String jsonBodyInvalidMarket = "{\n" +
                "    \"restaurantNo\": \"101\",\n" +
                "    \"marketName\": \"USPP\",\n" +
                "    \"hardwareDetails\": [\n" +
                "        {\n" +
                "            \"serialNumber\": \"serialNumber\",\n" +
                "            \"macAddress\": \"macAddress\",\n" +
                "            \"modelNumber\": \"modelNumber\",\n" +
                "            \"nodeId\": \"nodeId\",\n" +
                "            \"manufacturer\":\"manufacturer\",\n" +
                "            \"modelName\":\"modelName\",\n" +
                "            \"version\":\"version\",\n" +
                "            \"family\":\"family\",\n" +
                "            \"osType\":\"osType\",\n" +
                "            \"osVersion\":\"osVersion\"\n" +
                "        }\n" +
                "    ]\n" +
                "}";
        String jsonBodyInvalidRequest1 = "{\n" +
                "    \"restaurantNo\": \"\",\n" +
                "    \"marketName\": \"USPP\",\n" +
                "    \"hardwareDetails\": [\n" +
                "        {\n" +
                "            \"serialNumber\": \"serialNumber\",\n" +
                "            \"macAddress\": \"macAddress\",\n" +
                "            \"modelNumber\": \"modelNumber\",\n" +
                "            \"nodeId\": \"nodeId\",\n" +
                "            \"manufacturer\":\"manufacturer\",\n" +
                "            \"modelName\":\"modelName\",\n" +
                "            \"version\":\"version\",\n" +
                "            \"family\":\"family\",\n" +
                "            \"osType\":\"osType\",\n" +
                "            \"osVersion\":\"osVersion\"\n" +
                "        }\n" +
                "    ]\n" +
                "}";
        String jsonBodyInvalidRequest2 = "{\n" +
                "    \"restaurantNo\": \"101\",\n" +
                "    \"marketName\": \"US\",\n" +
                "    \"hardwareDetails\": [\n" +
                "        {\n" +
                "            \"serialNumber\": \"\",\n" +
                "            \"macAddress\": \"\",\n" +
                "            \"modelNumber\": \"\",\n" +
                "            \"nodeId\": \"\",\n" +
                "            \"manufacturer\":\"\",\n" +
                "            \"modelName\":\"\",\n" +
                "            \"version\":\"\",\n" +
                "            \"family\":\"\",\n" +
                "            \"osType\":\"\",\n" +
                "            \"osVersion\":\"\"\n" +
                "        }\n" +
                "    ]\n" +
                "}";
        for (String jsonBody : Stream.of(jsonBodyInvalidStoreMarket, jsonBodyInvalidMarket, jsonBodyInvalidRequest1, jsonBodyInvalidRequest2).collect(Collectors.toList())) {
            RestAssured.given(this.spec)
                    .accept("application/json")
                    .contentType("application/json")
                    .body(jsonBody)
                    .when()
                    .post("/store-dashboard/hardwaredetail")
                    .then().assertThat().statusCode(is(400));
        }
    }

    @Test
    void fetchHardwareDetails() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .get("/store-dashboard/hardwaredetail")
                .then().assertThat().statusCode(is(200));
    }

    @Test
    void fetchSingleHardwareDetails() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .get("/store-dashboard/hardwaredetail?restaurant_no=101")
                .then().assertThat().statusCode(is(200));
    }

    @Test
    void fetchInstalledHardwareDetailsWithoutRequestParam() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .get("/store-dashboard/hardwaredata")
                .then().assertThat().statusCode(is(200));
    }

    @Test
    void fetchInstalledHardwareDetailsWithRequestParam() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .get("/store-dashboard/hardwaredata?dataPrefix=Axi&offset=0&limit=1")
                .then().assertThat().statusCode(is(200));
    }


    @Test
    void updateRestaurantMetaData() {
        String jsonBody = "{\n" +
                "    \"marketName\": \"US\",\n" +
                "    \"storeInfo\": [\n" +
                "        {\n" +
                "            \"restaurantNumber\": \"101\",\n" +
                "            \"goLiveDate\": \"06/13/2023\",\n" +
                "            \"storeStatus\": \"Live\"\n" +
                "        },\n" +
                "        {\n" +
                "            \"restaurantNumber\": \"Test4545\",\n" +
                "            \"goLiveDate\": \"06/13/2023\",\n" +
                "            \"storeStatus\": \"Live\"\n" +
                "        }\n" +
                "    ]\n" +
                "}";
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .patch("/store-dashboard/restaurant/metadata")
                .then().assertThat().statusCode(is(200));
    }

    @Test
    void updateRestaurantMetaDataBadRequest() {
        String jsonBody1 = "{\n" +
                "    \"storeInfo\": [\n" +
                "        {\n" +
                "            \"restaurantNumber\": \"101\",\n" +
                "            \"goLiveDate\": \"06/13/2023\",\n" +
                "            \"storeStatus\": \"Live\"\n" +
                "        }\n" +
                "    ]\n" +
                "}";
        String jsonBody2 = "{\n" +
                "    \"marketName\": \"US\"\n" +
                "}";
        String jsonBody3 = "{\n" +
                "    \"marketName\": \"US\",\n" +
                "    \"storeInfo\": [\n" +
                "        {\n" +
                "            \"goLiveDate\": \"06/13/2023\",\n" +
                "            \"storeStatus\": \"Live\"\n" +
                "        },\n" +
                "        {\n" +
                "            \"restaurantNumber\": \"Test4545\",\n" +
                "            \"goLiveDate\": \"06/13/2023\",\n" +
                "            \"storeStatus\": \"Live\"\n" +
                "        }\n" +
                "    ]\n" +
                "}";

        for (String json : Stream.of(jsonBody1, jsonBody2, jsonBody3).collect(Collectors.toList())) {
            RestAssured.given(this.spec)
                    .accept("application/json")
                    .contentType("application/json")
                    .body(json)
                    .when()
                    .patch("/store-dashboard/restaurant/metadata")
                    .then().assertThat().statusCode(is(400));
        }
    }

    @Test
    void updateRestaurantMetaDataBadRequestMarketNotFound() {

        String jsonBody = "{\n" +
                "    \"marketName\": \"USA\",\n" +
                "    \"storeInfo\": [\n" +
                "        {\n" +
                "            \"restaurantNumber\": \"101\",\n" +
                "            \"goLiveDate\": \"06/13/2023\",\n" +
                "            \"storeStatus\": \"Live\"\n" +
                "        }\n" +
                "    ]\n" +
                "}";


        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .patch("/store-dashboard/restaurant/metadata")
                .then().assertThat().statusCode(is(400));
    }

    @Test
    void updateUSITTrackerDataSuccess() {

        String jsonBody = "{\n" +
                "    \"marketName\": \"US\",\n" +
                "    \"storeTrackerRequestDTOList\": {\n" +
                "        \"101\": {\n" +
                "            \"ipAddress\": \"6.73.76.6\",\n" +
                "            \"prepVisitDate\": \"20/07/22\",\n" +
                "            \"otpName\": \"test\",\n" +
                "            \"otpEmail\": null,\n" +
                "            \"otpPhone\": null,\n" +
                "            \"hardwareInstallationDate\": null,\n" +
                "            \"smokeTestDate\": null,\n" +
                "            \"ssxVersion\": null,\n" +
                "            \"driveThruType\": \"SIDE BY SIDE 2 BOOTH\"\n" +
                "        },\n" +
                "        \"XYZ\": {\n" +
                "            \"ipAddress\": \"6.76.75.6\",\n" +
                "            \"prepVisitDate\": \"12/03/2023\",\n" +
                "            \"otpName\": \"test\",\n" +
                "            \"otpEmail\": null,\n" +
                "            \"otpPhone\": null,\n" +
                "            \"hardwareInstallationDate\": null,\n" +
                "            \"smokeTestDate\": null,\n" +
                "            \"ssxVersion\": \"1.1.1\",\n" +
                "            \"driveThruType\": \"SIDE BY SIDE 2 BOOTH\"\n" +
                "        }\n" +
                "    }\n" +
                "}";


        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .patch("/store-dashboard/usit")
                .then().assertThat().statusCode(is(200));
    }

    @Test
    void updateUSITTrackerDataMarketNotFOund() {

        String jsonBody = "{\n" +
                "    \"marketName\": \"USAPAS\",\n" +
                "    \"storeTrackerRequestDTOList\": {\n" +
                "        \"97\": {\n" +
                "            \"ipAddress\": \"6.73.76.6\",\n" +
                "            \"prepVisitDate\": \"20/07/22\",\n" +
                "            \"otpName\": \"test\",\n" +
                "            \"otpEmail\": null,\n" +
                "            \"otpPhone\": null,\n" +
                "            \"hardwareInstallationDate\": null,\n" +
                "            \"smokeTestDate\": null,\n" +
                "            \"ssxVersion\": null,\n" +
                "            \"driveThruType\": \"SIDE BY SIDE 2 BOOTH\"\n" +
                "        }\n" +
                "    }\n" +
                "}";


        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .body(jsonBody)
                .when()
                .patch("/store-dashboard/usit")
                .then().assertThat().statusCode(is(400));
    }

    @Test
    void getRestaurantforOnboarding() {
        RestAssured.given(this.spec)
                .accept("application/json")
                .contentType("application/json")
                .when()
                .get("/store-dashboard/restaurant/onboard?fromDate=1688256000000&toDate=1688515200000")
                .then().assertThat().statusCode(is(200));
    }

}
