package com.mcd.restaurant;

import io.restassured.RestAssured;
import io.restassured.builder.RequestSpecBuilder;
import io.restassured.specification.RequestSpecification;
import net.minidev.json.JSONObject;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.web.server.LocalServerPort;
import org.springframework.restdocs.RestDocumentationContextProvider;
import org.springframework.restdocs.RestDocumentationExtension;
import org.springframework.restdocs.operation.preprocess.Preprocessors;
import org.springframework.restdocs.restassured3.RestAssuredRestDocumentation;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit.jupiter.SpringExtension;

import static org.hamcrest.CoreMatchers.is;

@ExtendWith({SpringExtension.class, RestDocumentationExtension.class})
@SpringBootTest(webEnvironment = SpringBootTest.WebEnvironment.RANDOM_PORT)
@ActiveProfiles("test")
class DevicePropTests {

    private RequestSpecification spec;

    @LocalServerPort
    int port;

    @BeforeEach
    public void setUp(RestDocumentationContextProvider restDocumentation) {
        RestAssured.port = port;

        this.spec = new RequestSpecBuilder()
                .addFilter(RestAssuredRestDocumentation.documentationConfiguration(restDocumentation).operationPreprocessors()
                        .withResponseDefaults(Preprocessors.prettyPrint()))
                .build();
    }

    @Test
    void getDevicePropWhenIDIsInvalid() {
        RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/device_props/123")
                .then().assertThat().statusCode(is(404));
    }

    @Test
    void getDevicePropWhenIDIsNull() {
        RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/device_props/null")
                .then().assertThat().statusCode(is(500));
    }

    @Test
    void getDevicePropWhenIDIsMissing() {
        RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/device_props/").then()
                .statusCode(is(200));
    }

    @Test
    void postDevicePropWhenInputJSONIsEmpty() {
        String jsonBody = "";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(jsonBody).when()
                .post("/restaurant_assets/device_props").then().assertThat().statusCode(is(400));
    }

    @Test
    void postDevicePropWhenInputJSONIsInvalid() {
        String jsonBody = "\n" +
                "{\n" +
                "    \"data\": {\n" +
                "        \"type\": \"device_props\",,\n" +
                "        \"attributes\": {\n" +
                "                \"propertyName\": \"ip_address\",\n" +
                "                \"dataType\": \"string\"\n" +
                "            },\n" +
                "        \"relationships\": {\n" +
                "            \"device\": {\n" +
                "                \"data\": {\n" +
                "                    \"type\": \"device\",\n" +
                "                    \"id\": 1\n" +
                "                }\n" +
                "            }\n" +
                "        }\n" +
                "    }\n" +
                "}";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(jsonBody).when()
                .post("/restaurant_assets/device_props").then().assertThat().statusCode(is(400));
    }

    @Test
    void postDevicePropWhenInputJSONIsValid() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        JSONObject subDataJSON = new JSONObject();
        JSONObject outerDataJSON = new JSONObject();
        JSONObject deviceJSON = new JSONObject();
        attributeJSON.put("dataType", "string");
        attributeJSON.put("propertyName", "ip_address");
        subDataJSON.put("type", "device");
        subDataJSON.put("id", 1);
        outerDataJSON.put("data", subDataJSON);
        deviceJSON.put("device", outerDataJSON);
        // fill data JSON
        dataJSON.put("type", "device_props");
        dataJSON.put("attributes", attributeJSON);
        dataJSON.put("relationships", deviceJSON);
        body.put("data", dataJSON);
        System.out.println(body.toString());
        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/device_props").then().assertThat().statusCode(is(201));
    }
}