package com.mcd.restaurant;

import io.restassured.RestAssured;
import io.restassured.builder.RequestSpecBuilder;
import io.restassured.response.Response;
import io.restassured.specification.RequestSpecification;
import net.minidev.json.JSONObject;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.CsvSource;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.web.server.LocalServerPort;
import org.springframework.restdocs.RestDocumentationContextProvider;
import org.springframework.restdocs.RestDocumentationExtension;
import org.springframework.restdocs.operation.preprocess.Preprocessors;
import org.springframework.restdocs.restassured3.RestAssuredRestDocumentation;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit.jupiter.SpringExtension;

import static org.hamcrest.CoreMatchers.is;

@ExtendWith({ SpringExtension.class, RestDocumentationExtension.class })
@SpringBootTest(webEnvironment = SpringBootTest.WebEnvironment.RANDOM_PORT)
@ActiveProfiles("test")
 class ComponentTypeTests {

	private RequestSpecification spec;

	@LocalServerPort
	int port;

	@BeforeEach
	public void setUp(RestDocumentationContextProvider restDocumentation) {
		RestAssured.port = port;

		this.spec = new RequestSpecBuilder()
				.addFilter(RestAssuredRestDocumentation.documentationConfiguration(restDocumentation)
						.operationPreprocessors().withResponseDefaults(Preprocessors.prettyPrint()))
				.build();
	}

	@Test
	 void getComponentTypeWhenIDIsInvalid() {
		RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/component_type/123").then()
				.assertThat().statusCode(is(404));
	}

	@Test
	 void getComponentTypeWhenIDIsNull() {
		RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/component_type/null").then()
				.assertThat().statusCode(is(500));
	}

	@Test
	 void getComponentTypeWhenIDIsMissing() {
		RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/component_type/").then()
				.statusCode(is(200));
	}

	@Test
	 void postComponentTypeWhenInputJSONIsEmpty() {
		String jsonBody = "";

		RestAssured.given(this.spec).contentType("application/vnd.api+json").body(jsonBody).when()
				.post("/restaurant_assets/component_type").then().assertThat().statusCode(is(400));
	}

	@Test
	 void postComponentTypeWhenInputJSONIsInvalid() {
		String jsonBody = "{}";

		RestAssured.given(this.spec).contentType("application/vnd.api+json").body(jsonBody).when()
				.post("/restaurant_assets/component_type").then().assertThat().statusCode(is(400));
	}

	@Test
	 void postComponentTypeWhenInputJSONIsInvalidDataKeyMissing() {
		JSONObject body = new JSONObject();
		JSONObject attributeJSON = new JSONObject();
		attributeJSON.put("name", "ComponentType_2");
		attributeJSON.put("description", "description2");
		// fill data JSON
		body.put("type", "component_type");
		body.put("attributes", attributeJSON);

		RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
				.post("/restaurant_assets/component_type").then().assertThat().statusCode(is(400));
	}

	@Test
	 void postComponentTypeNameAndTypeMissing() {
		JSONObject body = new JSONObject();
		JSONObject dataJSON = new JSONObject();
		body.put("data", dataJSON);

		RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
				.post("/restaurant_assets/component_type").then().assertThat().statusCode(is(422));
	}

	@Test
	 void postComponentTypeWhenTypeNameBothMissing() {
		JSONObject body = new JSONObject();
		JSONObject dataJSON = new JSONObject();
		JSONObject attributeJSON = new JSONObject();
		attributeJSON.put("description", "description 2");
		// fill data JSON
		dataJSON.put("attributes", attributeJSON);
		body.put("data", dataJSON);

		RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
				.post("/restaurant_assets/component_type").then().assertThat().statusCode(is(422));
	}

	@Test
	 void postComponentTypeWhenDescriptionIsMissing() {
		JSONObject body = new JSONObject();
		JSONObject dataJSON = new JSONObject();
		JSONObject attributeJSON = new JSONObject();
		attributeJSON.put("name", "component_type1");
		// fill data JSON
		dataJSON.put("type", "component_type");
		dataJSON.put("attributes", attributeJSON);
		body.put("data", dataJSON);

		RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
				.post("/restaurant_assets/component_type").then().assertThat().statusCode(is(201));
	}

	@Test
	 void postComponentTypeWhenDescriptionIsEmpty() {
		JSONObject body = new JSONObject();
		JSONObject dataJSON = new JSONObject();
		JSONObject attributeJSON = new JSONObject();
		attributeJSON.put("name", "component_type2");
		attributeJSON.put("description", "");
		// fill data JSON
		dataJSON.put("type", "component_type");
		dataJSON.put("attributes", attributeJSON);
		body.put("data", dataJSON);

		RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
				.post("/restaurant_assets/component_type").then().assertThat().statusCode(is(201));
	}

	@ParameterizedTest
	@CsvSource({"abc,500","null,500",",500"})
	 void patchComponentTypeWhenComponentTypeTyeIdIsInvalid(String id, int code) {
		JSONObject body = new JSONObject();
		JSONObject dataJSON = new JSONObject();
		JSONObject attributeJSON = new JSONObject();
		attributeJSON.put("name", "component_type2");
		attributeJSON.put("description", "hh");
		// fill data JSON
		dataJSON.put("type", "component_type");
		dataJSON.put("attributes", attributeJSON);
		body.put("data", dataJSON);

		RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
				.patch("/restaurant_assets/component_type/"+id).then().assertThat().statusCode(is(code));
	}

	@Test
	 void patchComponentTypeWhenComponentTypeIdIsMising() {
		JSONObject body = new JSONObject();
		JSONObject dataJSON = new JSONObject();
		JSONObject attributeJSON = new JSONObject();
		attributeJSON.put("name", "component_type2");
		attributeJSON.put("description", "desc");
		// fill data JSON
		dataJSON.put("type", "component_type");
		dataJSON.put("attributes", attributeJSON);
		body.put("data", dataJSON);

		RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
				.patch("/restaurant_assets/component_type/").then().assertThat().statusCode(is(400));
	}

	@Test
	 void patchComponentTypeWhenInputJSONIsEmptyString() {
		String body = "";

		RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
				.patch("/restaurant_assets/component_type/1").then().assertThat().statusCode(is(400));
	}

	@Test
	 void patchComponentTypeWhenInputJSONIsEmptyAndComponentTypeIdIsInvalid() {
		String body = "";

		RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
				.patch("/restaurant_assets/component_type/abc").then().assertThat().statusCode(is(500));
	}
}
