package com.mcd.restaurant;

import com.mcd.restaurant.bundledrelease.enums.BundledReleaseStatus;
import io.restassured.RestAssured;
import io.restassured.builder.RequestSpecBuilder;
import io.restassured.specification.RequestSpecification;
import net.minidev.json.JSONObject;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.CsvSource;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.web.server.LocalServerPort;
import org.springframework.restdocs.RestDocumentationContextProvider;
import org.springframework.restdocs.RestDocumentationExtension;
import org.springframework.restdocs.operation.preprocess.Preprocessors;
import org.springframework.restdocs.restassured3.RestAssuredRestDocumentation;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit.jupiter.SpringExtension;

import static org.hamcrest.CoreMatchers.is;

@ExtendWith({SpringExtension.class, RestDocumentationExtension.class})
@SpringBootTest(webEnvironment = SpringBootTest.WebEnvironment.RANDOM_PORT)
@ActiveProfiles("test")
 class BundleReleaseTests {

    private RequestSpecification spec;

    @LocalServerPort
    int port;

    @BeforeEach
    public void setUp(RestDocumentationContextProvider restDocumentation) {
        RestAssured.port = port;

        this.spec = new RequestSpecBuilder()
                .addFilter(RestAssuredRestDocumentation.documentationConfiguration(restDocumentation)
                        .operationPreprocessors().withResponseDefaults(Preprocessors.prettyPrint()))
                .build();
    }

    @Test
     void getBundleWhenIDIsInvalid() {
        RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/bundle_release/12").then()
                .assertThat().statusCode(is(404));
    }

    @Test
     void getBundleWhenIDIsNull() {
        RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/bundle_release/null").then()
                .assertThat().statusCode(is(500));
    }

    @Test
     void getBundleWhenIDIsMissing() {
        RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/bundle_release/").then()
                .statusCode(is(200));
    }

    @Test
     void postBundleWhenInputJSONIsEmpty() {
        String jsonBody = "";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(jsonBody).when()
                .post("/restaurant_assets/bundle_release").then().assertThat().statusCode(is(400));
    }

    @Test
     void postBundleWhenInputJSONIsInvalid() {
        String jsonBody = "{}";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(jsonBody).when()
                .post("/restaurant_assets/bundle_release").then().assertThat().statusCode(is(400));
    }

    @Test
     void postBundleWhenInputJSONIsInvalidDataKeyMissing() {
        JSONObject body = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "R-15");
        // fill data JSON
        body.put("type", "bundle_release");
        body.put("attributes", attributeJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/bundle_release").then().assertThat().statusCode(is(400));
    }

    @Test
     void postBundleTypeAndNameMissing() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/bundle_release").then().assertThat().statusCode(is(422));
    }

    @Test
     void postBundleWhenNameIsMissing() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        // fill data JSON
        dataJSON.put("attributes", attributeJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/bundle_release").then().assertThat().statusCode(is(422));
    }

   @ParameterizedTest
   @CsvSource({"abc,500","null,500",",500"})
     void patchBundleWhenBundleIdIsInvalid(String appId,int code) {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "R-15");
        // fill data JSON
        dataJSON.put("type", "bundle_release");
        dataJSON.put("attributes", attributeJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .patch("/restaurant_assets/bundle_release/"+appId).then().assertThat().statusCode(is(code));
    }

    @Test
     void patchBundleWhenAppIdIsMising() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "R-15");
        // fill data JSON
        dataJSON.put("type", "bundle_release");
        dataJSON.put("attributes", attributeJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .patch("/restaurant_assets/bundle_release/").then().assertThat().statusCode(is(400));
    }

    @Test
     void patchBundleWhenInputJSONIsEmptyString() {
        String body = "";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .patch("/restaurant_assets/bundle_release/1").then().assertThat().statusCode(is(400));
    }

    @Test
     void patchBundleWhenInputJSONIsEmptyAndAppIdIsInvalid() {
        String body = "";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .patch("/restaurant_assets/bundle_release/abc").then().assertThat().statusCode(is(500));
    }

    @Test
     void postBundleSuccess() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        JSONObject relationShipJSON = new JSONObject();
        attributeJSON.put("releaseName", "R-15");
        attributeJSON.put("status", BundledReleaseStatus.CREATED.value());
        attributeJSON.put("createdBy", "John Doe");
        attributeJSON.put("updatedBy", "John Doe");
        dataJSON.put("attributes", attributeJSON);
        dataJSON.put("type", "bundle_release");
        dataJSON.put("relationships", relationShipJSON);
        body.put("data", dataJSON);
        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/bundle_release").then().assertThat().statusCode(is(201));
    }


}
