package com.mcd.restaurant;

import com.mcd.restaurant.common.AssetExceptionMapper;
import com.mcd.restaurant.common.AssetPersistenceExceptionMapper;
import com.mcd.restaurant.service.ApiLoggingService;
import io.crnk.core.engine.document.ErrorData;
import io.crnk.core.engine.error.ErrorResponse;
import io.crnk.core.exception.*;
import io.restassured.RestAssured;
import io.restassured.builder.RequestSpecBuilder;
import io.restassured.specification.RequestSpecification;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.CsvSource;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.web.server.LocalServerPort;
import org.springframework.restdocs.RestDocumentationContextProvider;
import org.springframework.restdocs.RestDocumentationExtension;
import org.springframework.restdocs.operation.preprocess.Preprocessors;
import org.springframework.restdocs.restassured3.RestAssuredRestDocumentation;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit.jupiter.SpringExtension;

@ExtendWith({SpringExtension.class, RestDocumentationExtension.class})
@SpringBootTest(webEnvironment = SpringBootTest.WebEnvironment.RANDOM_PORT)
@ActiveProfiles("test")
class AssetExceptionMapperTest {
    private RequestSpecification spec;

    @LocalServerPort
    int port;

    @InjectMocks
    private AssetExceptionMapper assetExceptionMapper;

    @InjectMocks
    private AssetPersistenceExceptionMapper assetPersistenceExceptionMapper;

    @Mock
    private ApiLoggingService apiLoggingService;

    @BeforeEach
    public void setUp(RestDocumentationContextProvider restDocumentation) {
        RestAssured.port = port;

        this.spec = new RequestSpecBuilder()
                .addFilter(RestAssuredRestDocumentation.documentationConfiguration(restDocumentation)
                        .operationPreprocessors().withResponseDefaults(Preprocessors.prettyPrint()))
                .build();
    }

    @Test
    void toErrorResponseTest() {
        CrnkMappableException exception = new BadRequestException("Exception");
        Mockito.when(apiLoggingService.logWarn(exception)).thenReturn("ERRORID");
        ErrorResponse errorResponse = assetExceptionMapper.toErrorResponse(exception);
        Assertions.assertEquals(errorResponse.getHttpStatus(), errorResponse.getHttpStatus());
        errorResponse.getErrors().stream().forEach(i -> Assertions.assertEquals("Exception", i.getDetail()));
    }

    @ParameterizedTest
    @CsvSource({"520,Exception", "500,Exception"})
    void toErrorResponseBoundaryTest(int errorcode, String message) {
        ErrorData errorData = ErrorData.builder().setId("123").setDetail(message).build();
        CrnkMappableException exception = new CrnkMappableException(errorcode, errorData) {
            @Override
            public ErrorData getErrorData() {
                return super.getErrorData();
            }

            @Override
            public int getHttpStatus() {
                return super.getHttpStatus();
            }
        };
        Mockito.when(apiLoggingService.logError(exception)).thenReturn("ERRORID");
        ErrorResponse errorResponse = assetExceptionMapper.toErrorResponse(exception);
        Assertions.assertEquals(errorResponse.getHttpStatus(), errorResponse.getHttpStatus());
        errorResponse.getErrors().stream().forEach(i -> Assertions.assertEquals("There was an issue with processing your request. Please try after sometime.", i.getDetail()));
    }

    @Test
    void toErrorResponseOutsideBoundaryTest() {
        ErrorData errorData = ErrorData.builder().setId("123").setDetail("Exception").build();
        CrnkMappableException exception = new CrnkMappableException(600, errorData) {
            @Override
            public ErrorData getErrorData() {
                return super.getErrorData();
            }

            @Override
            public int getHttpStatus() {
                return super.getHttpStatus();
            }
        };
        Mockito.when(apiLoggingService.logWarn(exception)).thenReturn("ERRORID");
        ErrorResponse errorResponse = assetExceptionMapper.toErrorResponse(exception);
        Assertions.assertEquals(errorResponse.getHttpStatus(), errorResponse.getHttpStatus());
        errorResponse.getErrors().stream().forEach(i -> Assertions.assertEquals("There was an issue with processing your request. Please try after sometime.", i.getDetail()));
    }

    @ParameterizedTest
    @CsvSource({"403,ForbiddenException", "405,MethodNotAllowedException", "401,UnauthorizedException", "404,ResourceNotFoundException", "400,BadRequestException", "500,InternalServerErrorException"})
    void fromErrorResponseTest(int errorCode, String classType) {
        ErrorResponse errorResponse = ErrorResponse.builder().setStatus(errorCode).setSingleErrorData(ErrorData.builder().setTitle("ABC").setDetail("Exception").build()).build();
        Assertions.assertEquals(classType, assetExceptionMapper.fromErrorResponse(errorResponse).getClass().getSimpleName());

    }

    @ParameterizedTest
    @CsvSource({"403,ForbiddenException", "401,UnauthorizedException", "404,ResourceNotFoundException", "400,BadRequestException", "500,InternalServerErrorException"})
    void fromErrorResponseTestNoMessage(int errorCode, String classType) {
        ErrorResponse errorResponse = ErrorResponse.builder().setStatus(errorCode).setSingleErrorData(ErrorData.builder().setTitle("ABC").build()).build();
        Assertions.assertEquals(classType, assetExceptionMapper.fromErrorResponse(errorResponse).getClass().getSimpleName());
        Assertions.assertEquals("ABC", assetExceptionMapper.fromErrorResponse(errorResponse).getErrorData().getDetail());

    }

    @ParameterizedTest
    @CsvSource({"403,ForbiddenException", "401,UnauthorizedException", "404,ResourceNotFoundException", "400,BadRequestException", "500,InternalServerErrorException"})
    void fromErrorResponseTestNoMessageAndTitle(int errorCode, String classType) {
        ErrorResponse errorResponse = ErrorResponse.builder().setStatus(errorCode).setSingleErrorData(ErrorData.builder().setCode("ABC").build()).build();
        Assertions.assertEquals(classType, assetExceptionMapper.fromErrorResponse(errorResponse).getClass().getSimpleName());
        Assertions.assertEquals("ABC", assetExceptionMapper.fromErrorResponse(errorResponse).getErrorData().getDetail());

    }

    @ParameterizedTest
    @CsvSource({"402,false", "401,true", "404,true", "400,true", "405,true", "403,true", "500,true"})
    void acceptsTest(int errorCode, boolean boolVal) {
        ErrorResponse errorResponse = ErrorResponse.builder().setStatus(errorCode).setSingleErrorData(ErrorData.builder().setCode("ABC").build()).build();
        Assertions.assertEquals(boolVal, assetExceptionMapper.accepts(errorResponse));

    }

    @Test
    void fromErrorResponseTest() {
        ErrorResponse errorResponse = ErrorResponse.builder().setStatus(400).setSingleErrorData(ErrorData.builder().setCode("ABC").build()).build();
        Assertions.assertThrows(UnsupportedOperationException.class, () -> assetPersistenceExceptionMapper.fromErrorResponse(errorResponse));

    }

    @Test
    void acceptTest() {
        ErrorResponse errorResponse = ErrorResponse.builder().setStatus(400).setSingleErrorData(ErrorData.builder().setCode("ABC").build()).build();
        Assertions.assertFalse(assetPersistenceExceptionMapper.accepts(errorResponse));

    }


}
