package com.mcd.restaurant.scheduler;


import com.mcd.restaurant.deployment.controller.view.request.DeploymentCheckRequestDTO;
import com.mcd.restaurant.deployment.service.DeployService;
import com.mcd.restaurant.deployment.service.DeploymentCheckType;
import com.mcd.restaurant.deployment.service.DeploymentStatus;
import com.mcd.restaurant.model.DeploymentGroup;
import com.mcd.restaurant.model.DeploymentHistory;
import com.mcd.restaurant.model.PrePostDeploymentModel;
import com.mcd.restaurant.repository.DeploymentGroupRepository;
import com.mcd.restaurant.repository.DeploymentHistoryRepository;
import com.mcd.restaurant.repository.PrePostDeploymentRepository;
import io.crnk.core.queryspec.FilterOperator;
import io.crnk.core.queryspec.FilterSpec;
import io.crnk.core.queryspec.PathSpec;
import io.crnk.core.queryspec.QuerySpec;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import javax.transaction.Transactional;
import java.sql.Timestamp;
import java.time.LocalDateTime;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Component("deploymentStatusUpdateScheduler")
@Slf4j
public class DeploymentStatusUpdateScheduler {

    public static final String STATUS = "status";
    @Value("${scheduler.fail-after}")
    private Long failAfterTime;
    @Autowired
    private DeploymentHistoryRepository deploymentHistoryRepository;
    @Autowired
    private DeploymentGroupRepository deploymentGroupRepository;
    @Autowired
    private DeployService deployService;
    @Autowired
    private PrePostDeploymentRepository prePostDeploymentRepository;


    @Transactional
    @Scheduled(fixedDelayString = "${scheduler.fixed-delay}", initialDelayString = "${scheduler.initial-delay}")
    public void cronJobToUpdateDeploymentStatus() {
        log.info("DeploymentStatusUpdateScheduler initiated");
        List<DeploymentHistory> deploymentHistoryList = deploymentHistoryRepository.findAll(prepareQuerySpecFilterForSingleVersionInBundleAppVersion());
        deploymentHistoryList.forEach(i -> {
            i.setStatus(DeploymentStatus.Failed.value());
            i.setDescription("Deployment is marked as failed by the system");
            deploymentHistoryRepository.save(i);
        });
        List<PrePostDeploymentModel> model = prePostDeploymentRepository.findAll(prepareQuerySpecFilterForPrePostModel());
        model.forEach(records -> prePostDeploymentRepository.delete(records.getId()));
    }

    @Transactional
    @Scheduled(fixedDelayString = "60000", initialDelayString = "12000")
    public void cronJobToTriggerPostDeployCheck() {
        log.info("DeploymentPostDeployment initiated now");
        List<DeploymentGroup> deploymentGroupList = deploymentGroupRepository.findAll(prepareQuerySpecFilterForDeploymentGroupPostCheck());
        deploymentGroupList.forEach(i -> {
            deployService.saveDeploymentCheckDetails(DeploymentCheckRequestDTO.builder()
                    .deploymentGroupId(i.getId())
                    .initiatedBy("System")
                    .status(DeploymentStatus.InProgress)
                    .type(DeploymentCheckType.POST_DEPLOYMENT_CHECK)
                    .build());
            i.setIsPostDeploymentTriggered(Boolean.TRUE.toString());
            deploymentGroupRepository.save(i);
        });
        deploymentGroupList = deploymentGroupRepository.findAll(prepareQuerySpecFilterForDeploymentGroupPreCheck());
        deploymentGroupList.forEach(i -> {
            deployService.saveDeploymentCheckDetails(DeploymentCheckRequestDTO.builder()
                    .deploymentGroupId(i.getId())
                    .initiatedBy("System")
                    .status(DeploymentStatus.InProgress)
                    .type(DeploymentCheckType.PRE_DEPLOYMENT_CHECK)
                    .build());
            i.setIsPreDeploymentTriggered(Boolean.TRUE.toString());
            deploymentGroupRepository.save(i);
        });
    }

    public QuerySpec prepareQuerySpecFilterForSingleVersionInBundleAppVersion() {
        FilterSpec filterStatus = new FilterSpec(PathSpec.of(STATUS), FilterOperator.EQ, DeploymentStatus.InProgress.value());
        FilterSpec filterUpdatedAT = new FilterSpec(PathSpec.of("updated"), FilterOperator.LT, Timestamp.valueOf(LocalDateTime.now().minusMinutes(failAfterTime)));
        FilterSpec filterSpecFinal = FilterSpec.and(Stream.of(filterStatus, filterUpdatedAT).collect(Collectors.toList()));
        QuerySpec querySpec = new QuerySpec(DeploymentHistory.class);
        querySpec.setFilters(Stream.of(filterSpecFinal).collect(Collectors.toList()));
        return querySpec;

    }

    public QuerySpec prepareQuerySpecFilterForDeploymentGroupPostCheck() {
        FilterSpec filterStatus1 = new FilterSpec(PathSpec.of(STATUS), FilterOperator.EQ, DeploymentStatus.Completed.value());
        FilterSpec filterStatus2 = new FilterSpec(PathSpec.of(STATUS), FilterOperator.EQ, DeploymentStatus.Failed.value());
        FilterSpec filterPostCheck = new FilterSpec(PathSpec.of("isPostDeploymentTriggered"), FilterOperator.EQ, Boolean.FALSE.toString());
        FilterSpec filterPostNullCheck = new FilterSpec(PathSpec.of("isPostDeploymentTriggered"), FilterOperator.EQ, null);
        FilterSpec filterSpecOrFinal = FilterSpec.or(Stream.of(filterPostCheck, filterPostNullCheck).collect(Collectors.toList()));
        FilterSpec filterSpecStatusFinal = FilterSpec.or(Stream.of(filterStatus1, filterStatus2).collect(Collectors.toList()));
        FilterSpec filterSpecFinal = FilterSpec.and(Stream.of(filterSpecStatusFinal, filterSpecOrFinal).collect(Collectors.toList()));
        QuerySpec querySpec = new QuerySpec(DeploymentGroup.class);
        querySpec.setFilters(Stream.of(filterSpecFinal).collect(Collectors.toList()));
        return querySpec;

    }

    public QuerySpec prepareQuerySpecFilterForDeploymentGroupPreCheck() {
        FilterSpec filterStatus1 = new FilterSpec(PathSpec.of(STATUS), FilterOperator.EQ, DeploymentStatus.STAGED.value());
        FilterSpec filterPreCheck = new FilterSpec(PathSpec.of("isPreDeploymentTriggered"), FilterOperator.EQ, Boolean.FALSE.toString());
        FilterSpec filterPreNullCheck = new FilterSpec(PathSpec.of("isPreDeploymentTriggered"), FilterOperator.EQ, null);
        FilterSpec filterPreCheckFlow = new FilterSpec(PathSpec.of("isPreCheckFlow"), FilterOperator.EQ, Boolean.TRUE.toString());
        FilterSpec filterProcessing= new FilterSpec(PathSpec.of("processing"), FilterOperator.EQ, Boolean.FALSE);
        FilterSpec filterSpecOrFinal = FilterSpec.or(Stream.of(filterPreCheck, filterPreNullCheck).collect(Collectors.toList()));
        FilterSpec filterSpecFinal = FilterSpec.and(Stream.of(filterStatus1, filterSpecOrFinal,filterPreCheckFlow,filterProcessing).collect(Collectors.toList()));
        QuerySpec querySpec = new QuerySpec(DeploymentGroup.class);
        querySpec.addFilter(filterSpecFinal);
        return querySpec;
    }

    public QuerySpec prepareQuerySpecFilterForPrePostModel() {
        FilterSpec filetrUpdatedAT = new FilterSpec(PathSpec.of("updated"), FilterOperator.LT, Timestamp.valueOf(LocalDateTime.now().minusDays(7)));
        QuerySpec querySpec = new QuerySpec(PrePostDeploymentModel.class);
        querySpec.setFilters(Stream.of(filetrUpdatedAT).collect(Collectors.toList()));
        return querySpec;

    }
}
