package com.mcd.restaurant.repository;

import com.mcd.restaurant.dashboard.enums.HierarchyLevel;
import com.mcd.restaurant.model.BreHierarchyNode;
import com.mcd.restaurant.model.Restaurant;
import com.mcd.restaurant.model.StoreDemographic;
import io.crnk.core.exception.BadRequestException;
import io.crnk.core.queryspec.FilterOperator;
import io.crnk.core.queryspec.FilterSpec;
import io.crnk.core.queryspec.PathSpec;
import io.crnk.core.queryspec.QuerySpec;
import io.crnk.data.jpa.JpaEntityRepositoryBase;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Component
public class RestaurantRepository extends JpaEntityRepositoryBase<Restaurant, Integer> {
    public static final String MARKET_ID = "market.id";
    @Autowired
    private StoreDemographicRepository storeDemographicRepository;
    @Autowired
    private BreHierarchyNodeRepository breHierarchyNodeRepository;

    public RestaurantRepository() {
        super(Restaurant.class);
    }

    @Override
    public Restaurant create(Restaurant resource) {
        if (!super.findAll(prepareQuerySpecFilterForRestaurantByRestaurantNumberAndMarketId(resource.getName(), resource.getMarket().getId())).isEmpty()) {
            throw new BadRequestException("Restaurant number is already present for the given market");
        }
        if (resource.getBreHierarchyNode() != null) {
            validateAndInsertStoreDemographicPresenceForBreHierarchyNode(resource);
        } else {
            List<BreHierarchyNode> market = breHierarchyNodeRepository.findAll(breHierarchyNodeRepository.prepareQuerySpecFilterForBreHierarchyNodeMarketName(HierarchyLevel.MARKET, resource.getMarket().getName()));
            if (market.isEmpty()) {
                throw new BadRequestException("Market information not present in BreHierarchyNode table");
            }
            List<BreHierarchyNode> hierarchyNodes = breHierarchyNodeRepository.findAll(breHierarchyNodeRepository.prepareQuerySpecFilterForBreHierarchyNodeRestaurantNumber(resource.getName(), market.get(0).getId()));
            if (hierarchyNodes.isEmpty()) {
                BreHierarchyNode hierarchyNode = BreHierarchyNode.builder().restaurantNo(resource.getName()).marketNodeId(market.get(0).getId()).hierarchyLevel(HierarchyLevel.RESTAURANT.value()).build();
                BreHierarchyNode savedBreHierarchyNode = breHierarchyNodeRepository.save(hierarchyNode);
                resource.setBreHierarchyNode(savedBreHierarchyNode);
                insertDummyDemographicDetails(savedBreHierarchyNode);
            } else {
                resource.setBreHierarchyNode(hierarchyNodes.get(0));
                validateAndInsertStoreDemographicPresenceForBreHierarchyNode(resource);
            }
        }
        return super.create(resource);
    }

    public void validateAndInsertStoreDemographicPresenceForBreHierarchyNode(Restaurant resource) {
        List<StoreDemographic> storeDemographics = storeDemographicRepository.findAll(storeDemographicRepository.prepareQuerySpecFilterForStoreDemographicHierarchyNodeId(resource.getBreHierarchyNode().getId()));
        if (storeDemographics.isEmpty()) {
            insertDummyDemographicDetails(resource.getBreHierarchyNode());
        }
    }

    public void insertDummyDemographicDetails(BreHierarchyNode hierarchyNode) {
        StoreDemographic storeDemographic = StoreDemographic.builder().hierarchyNode(hierarchyNode).build();
        storeDemographicRepository.save(storeDemographic);
    }

    public QuerySpec prepareQuerySpecFilterForRestaurantByMarketName(Integer marketId) {
        QuerySpec querySpec = new QuerySpec(Restaurant.class);
        FilterSpec filterNameSpec = new FilterSpec(PathSpec.of(MARKET_ID), FilterOperator.EQ, marketId);
        querySpec.setFilters(Stream.of(filterNameSpec).collect(Collectors.toList()));
        return querySpec;
    }

    public QuerySpec prepareQuerySpecFilterForRestaurantWithBreHierarchyNodePresent() {
        QuerySpec querySpec = new QuerySpec(Restaurant.class);
        FilterSpec filterSpec = new FilterSpec(PathSpec.of("breHierarchyNode"), FilterOperator.NEQ, null);
        querySpec.setFilters(Stream.of(filterSpec).collect(Collectors.toList()));
        return querySpec;
    }

    public QuerySpec prepareQuerySpecFilterForRestaurantByRestaurantNumberAndMarketId(String restaurantNo, Integer marketId) {
        QuerySpec querySpec = new QuerySpec(Restaurant.class);
        FilterSpec filterMarketNameSpec = new FilterSpec(PathSpec.of(MARKET_ID), FilterOperator.EQ, marketId);
        FilterSpec filterNameSpec = new FilterSpec(PathSpec.of("name"), FilterOperator.EQ, restaurantNo);
        FilterSpec filterFinalType = FilterSpec.and(Stream.of(filterMarketNameSpec, filterNameSpec).collect(Collectors.toList()));
        querySpec.setFilters(Stream.of(filterFinalType).collect(Collectors.toList()));
        return querySpec;
    }

    public QuerySpec prepareQuerySpecFilterForRestaurantByRegionAndMarket(Integer regionId, Integer marketId) {
        QuerySpec querySpec = new QuerySpec(Restaurant.class);
        FilterSpec filterRegionNameSpec = new FilterSpec(PathSpec.of("breHierarchyNode.regionNodeId"), FilterOperator.EQ, regionId);
        FilterSpec filterNameSpec = new FilterSpec(PathSpec.of("breHierarchyNode.marketNodeId"), FilterOperator.EQ, marketId);
        FilterSpec filterFinalType = FilterSpec.and(Stream.of(filterRegionNameSpec, filterNameSpec).collect(Collectors.toList()));
        querySpec.setFilters(Stream.of(filterFinalType).collect(Collectors.toList()));
        return querySpec;
    }

    public QuerySpec prepareQuerySpecFilterForRestaurantByIds(Set<Integer> ids) {
        QuerySpec querySpec = new QuerySpec(Restaurant.class);
        FilterSpec filterIdSpec = new FilterSpec(PathSpec.of("id"), FilterOperator.EQ, ids);
        querySpec.setFilters(Stream.of(filterIdSpec).collect(Collectors.toList()));
        return querySpec;
    }

    public QuerySpec prepareQuerySpecFilterForRestaurantByRestaurantNumbersAndMarketId(List<String> restaurantNames, Integer marketId) {
        QuerySpec querySpec = new QuerySpec(Restaurant.class);
        FilterSpec filterMarketNameSpec = new FilterSpec(PathSpec.of(MARKET_ID), FilterOperator.EQ, marketId);
        FilterSpec filterNameSpec = new FilterSpec(PathSpec.of("name"), FilterOperator.EQ, restaurantNames);
        FilterSpec filterFinalType = FilterSpec.and(Stream.of(filterMarketNameSpec, filterNameSpec).collect(Collectors.toList()));
        querySpec.setFilters(Stream.of(filterFinalType).collect(Collectors.toList()));
        return querySpec;
    }

    public QuerySpec prepareQuerySpecFilterForRestaurantByName(String restaurantName) {
        QuerySpec querySpec = new QuerySpec(Restaurant.class);
        FilterSpec filterIdSpec = new FilterSpec(PathSpec.of("name"), FilterOperator.EQ, restaurantName);
        querySpec.setFilters(Stream.of(filterIdSpec).collect(Collectors.toList()));
        return querySpec;
    }

}
