package com.mcd.restaurant.productversions.utils;


import com.mcd.restaurant.productversions.controller.view.request.*;
import com.mcd.restaurant.productversions.error.ProductDeploymentBadRequestException;
import com.mcd.restaurant.repository.ProductRepository;
import com.mcd.restaurant.repository.ProductVersionRepository;
import io.crnk.core.engine.internal.utils.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

@Component
public class ProductDeploymentValidator {
    public static final String DUPLICATE_EXCEPTION_MESSAGE = "Duplicate Applications cannot be inserted in a product version";
    @Autowired
    private ProductRepository productRepository;
    @Autowired
    private ProductVersionRepository productVersionRepository;

    public void createProductRequestDTOValidator(CreateProductRequestDTO createProductRequestDTO) {
        if (StringUtils.isBlank(createProductRequestDTO.getProductName())) {
            throw new ProductDeploymentBadRequestException("Product applicationName is required");
        }
        if (CollectionUtils.isEmpty(createProductRequestDTO.getApplications())) {
            throw new ProductDeploymentBadRequestException("Atleast one aplication is needed to create a product");
        } else {
            List<ProductApplicationRequestDTO> applcations = createProductRequestDTO.getApplications();
            applcations.forEach(this::productApplicationRequestDTOValidator);
            List<String> applicationNameList = applcations.stream().map(ProductApplicationRequestDTO::getApplicationName).collect(Collectors.toList());
            applicationNameList.forEach(i -> {
                if (Collections.frequency(applicationNameList, i) > 1) {
                    throw new ProductDeploymentBadRequestException(DUPLICATE_EXCEPTION_MESSAGE);
                }
            });
        }
    }

    public void productApplicationRequestDTOValidator(ProductApplicationRequestDTO productApplicationRequestDTO) {
        if (StringUtils.isBlank(productApplicationRequestDTO.getApplicationName())) {
            throw new ProductDeploymentBadRequestException("Application: applicationName is required");
        }
        if (StringUtils.isBlank(productApplicationRequestDTO.getApplicationVersion())) {
            throw new ProductDeploymentBadRequestException("Application: applicationVersion is required");
        }
        if (StringUtils.isBlank(productApplicationRequestDTO.getApplicationURL())) {
            throw new ProductDeploymentBadRequestException("Application: applicationUrl is required");
        }
    }


    public void updateProductRequestDTOValidator(UpdateProductRequestDTO updateProductRequestDTO) {
        if (updateProductRequestDTO.getProductId() == null && StringUtils.isBlank(updateProductRequestDTO.getProductName())) {
            throw new ProductDeploymentBadRequestException("Product id or name is required");
        }
        if (StringUtils.isBlank(updateProductRequestDTO.getVersionName())) {
            throw new ProductDeploymentBadRequestException("Version Name is required");
        }
        if (CollectionUtils.isEmpty(updateProductRequestDTO.getApplications())) {
            throw new ProductDeploymentBadRequestException("Atleast one aplication is needed to create a version");
        } else {
            List<ProductApplicationRequestDTO> applcations = updateProductRequestDTO.getApplications();
            applcations.forEach(this::productApplicationRequestDTOValidator);
            List<String> applicationNameList = applcations.stream().map(ProductApplicationRequestDTO::getApplicationName).collect(Collectors.toList());
            applicationNameList.forEach(i -> {
                if (Collections.frequency(applicationNameList, i) > 1) {
                    throw new ProductDeploymentBadRequestException(DUPLICATE_EXCEPTION_MESSAGE);
                }
            });
        }
    }

    public void updateProductVersion(UpdateProductVersionRequestDTO updateProductVersionRequestDTO) {
        if (updateProductVersionRequestDTO.getProductId() == null) {
            throw new ProductDeploymentBadRequestException("Product id is required");
        }
        if (updateProductVersionRequestDTO.getProductVersionId() == null) {
            throw new ProductDeploymentBadRequestException("ProductVersion id is required");
        }
        if (CollectionUtils.isEmpty(updateProductVersionRequestDTO.getApplications())) {
            throw new ProductDeploymentBadRequestException("Atleast one aplication is needed in a version");
        } else {
            List<ProductApplicationRequestDTO> applcations = updateProductVersionRequestDTO.getApplications();
            applcations.forEach(this::productApplicationRequestDTOValidator);
            List<String> applicationNameList = applcations.stream().map(ProductApplicationRequestDTO::getApplicationName).collect(Collectors.toList());
            applicationNameList.forEach(i -> {
                if (Collections.frequency(applicationNameList, i) > 1) {
                    throw new ProductDeploymentBadRequestException(DUPLICATE_EXCEPTION_MESSAGE);
                }
            });
        }
    }

    public void validateBulkProductDetailFetchRequestDTO(BulkProductDetailFetchRequestDTO requestDTO) {
        if (CollectionUtils.isEmpty(requestDTO.getProductVersion())) {
            throw new ProductDeploymentBadRequestException("Product-Version details are required");
        }
        requestDTO.getProductVersion().forEach(this::productRequestDTOValidator);
    }

    public void productRequestDTOValidator(ProductRequestDTO requestDTO) {
        if (StringUtils.isBlank(requestDTO.getProductName())) {
            throw new ProductDeploymentBadRequestException("Application: productName is required");
        }
        if (StringUtils.isBlank(requestDTO.getVersionName())) {
            throw new ProductDeploymentBadRequestException("Application: productVersion is required");
        }
    }
}
