package com.mcd.restaurant.productversions.service;


import com.mcd.restaurant.common.MapperUtils;
import com.mcd.restaurant.model.ProductApplications;
import com.mcd.restaurant.model.ProductVersions;
import com.mcd.restaurant.model.Products;
import com.mcd.restaurant.productversions.controller.view.request.BulkProductDetailFetchRequestDTO;
import com.mcd.restaurant.productversions.controller.view.request.CreateProductRequestDTO;
import com.mcd.restaurant.productversions.controller.view.request.UpdateProductRequestDTO;
import com.mcd.restaurant.productversions.controller.view.request.UpdateProductVersionRequestDTO;
import com.mcd.restaurant.productversions.controller.view.response.BulkProductDetailResponseDTO;
import com.mcd.restaurant.productversions.controller.view.response.DeleteVersionResponseDTO;
import com.mcd.restaurant.productversions.controller.view.response.ProductResponseDTO;
import com.mcd.restaurant.productversions.controller.view.response.ProductVersionResponseDTO;
import com.mcd.restaurant.productversions.error.ProductDeploymentBadRequestException;
import com.mcd.restaurant.productversions.error.ProductDeploymentException;
import com.mcd.restaurant.productversions.utils.ProductDeploymentUtility;
import com.mcd.restaurant.productversions.utils.ProductDeploymentValidator;
import com.mcd.restaurant.repository.ProductApplicationRepository;
import com.mcd.restaurant.repository.ProductRepository;
import com.mcd.restaurant.repository.ProductVersionRepository;
import io.crnk.core.exception.ResourceNotFoundException;
import io.crnk.core.queryspec.QuerySpec;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.List;
import java.util.stream.Collectors;


@Service
public class ProductDeploymentService {
    @Autowired
    private ProductDeploymentValidator productDeploymentValidator;
    @Autowired
    private ProductVersionRepository productVersionRepository;
    @Autowired
    private ProductDeploymentUtility productDeploymentUtility;
    @Autowired
    private ProductApplicationRepository productApplicationRepository;
    @Autowired
    private ProductRepository productRepository;
    @Autowired
    private MapperUtils mapperUtils;

    @Transactional
    public ProductResponseDTO saveAllProductApplication(CreateProductRequestDTO createProductRequestDTO) {
        productDeploymentValidator.createProductRequestDTOValidator(createProductRequestDTO);
        try {
            Products product;
            List<Products> products = productRepository.getProductByName(createProductRequestDTO.getProductName());
            if (CollectionUtils.isEmpty(products)) {
                product = productRepository.save(productDeploymentUtility.buildProducts(createProductRequestDTO));
            } else {
                throw new ProductDeploymentBadRequestException(String.format("Product with product name %s already exist", createProductRequestDTO.getProductName()));
            }
            ProductVersions productVersion = productVersionRepository.save(productDeploymentUtility.buildProductVersions(product, StringUtils.isBlank(createProductRequestDTO.getVersionName()) ? "1.0.0" : createProductRequestDTO.getVersionName()));
            List<ProductApplications> productApplications = createProductRequestDTO.getApplications().stream().map(application ->
                    productApplicationRepository.save(productDeploymentUtility.buildProductApplication(application, productVersion))
            ).collect(Collectors.toList());

            return productDeploymentUtility.buildProductVersionResponse(product, productVersion, productApplications);
        } catch (ProductDeploymentBadRequestException ex) {
            throw ex;
        } catch (Exception ex) {
            throw new ProductDeploymentException(ex.getMessage());
        }
    }

    @Transactional
    public ProductResponseDTO updateProducts(UpdateProductRequestDTO updateProductRequestDTO) {
        productDeploymentValidator.updateProductRequestDTOValidator(updateProductRequestDTO);
        try {
            Products product = fetchProductModel(updateProductRequestDTO);
            if (!CollectionUtils.isEmpty(productVersionRepository.getProductVersionByProductIdAndVersionName(product.getId(), updateProductRequestDTO.getVersionName()))) {
                throw new ProductDeploymentBadRequestException(String.format("Product-version %s already exist for the product %s", updateProductRequestDTO.getVersionName(), product.getName()));
            }
            ProductVersions productVersion = productVersionRepository.save(productDeploymentUtility.buildProductVersions(product, updateProductRequestDTO.getVersionName()));
            List<ProductApplications> productApplications = updateProductRequestDTO.getApplications().stream().map(application ->
                    productApplicationRepository.save(productDeploymentUtility.buildProductApplication(application, productVersion))
            ).collect(Collectors.toList());

            return productDeploymentUtility.buildProductVersionResponse(product, productVersion, productApplications);
        } catch (ResourceNotFoundException rex) {
            throw new ProductDeploymentBadRequestException(rex.getMessage());
        } catch (ProductDeploymentBadRequestException ex) {
            throw ex;
        } catch (Exception ex) {
            throw new ProductDeploymentException(ex.getMessage());
        }
    }

    private Products fetchProductModel(UpdateProductRequestDTO updateProductRequestDTO) {

        if (updateProductRequestDTO.getProductId() != null) {
            return productRepository.findOne(updateProductRequestDTO.getProductId(), new QuerySpec(Products.class));
        } else {
            List<Products> products = productRepository.getProductByName(updateProductRequestDTO.getProductName());
            if (products.isEmpty()) {
                throw new ProductDeploymentBadRequestException(String.format("No product exist by the name %s", updateProductRequestDTO.getProductName()));
            }
            return products.get(0);
        }

    }

    @Transactional
    public ProductVersionResponseDTO updateProductVersions(UpdateProductVersionRequestDTO updateProductVersionRequestDTO) {
        productDeploymentValidator.updateProductVersion(updateProductVersionRequestDTO);
        try {
            Products product = productRepository.findOne(updateProductVersionRequestDTO.getProductId(), new QuerySpec(Products.class));
            ProductVersions versions = productVersionRepository.findOne(updateProductVersionRequestDTO.getProductVersionId(), new QuerySpec(ProductVersions.class));
            if (versions.getProducts().getId() != product.getId()) {
                throw new ProductDeploymentBadRequestException(String.format("productVersionId %s does not belong to productId %s", updateProductVersionRequestDTO.getProductVersionId(), updateProductVersionRequestDTO.getProductId()));
            }
            if (!StringUtils.isBlank(versions.getIsDeployed()) && versions.getIsDeployed().equalsIgnoreCase(Boolean.TRUE.toString())) {
                throw new ProductDeploymentBadRequestException("Cannot execute this update request as this version is already deployed");
            }
            if (!StringUtils.isBlank(updateProductVersionRequestDTO.getVersionName())) {
                List<ProductVersions> versionsList = productVersionRepository.getProductVersionByProductIdAndVersionName(updateProductVersionRequestDTO.getProductId(), updateProductVersionRequestDTO.getVersionName());
                if (!CollectionUtils.isEmpty(versionsList) && versionsList.get(0).getId() != versions.getId()) {
                    throw new ProductDeploymentBadRequestException(String.format("Product-version %s already exist for the product %s", updateProductVersionRequestDTO.getVersionName(), product.getName()));
                }
            }
            if (!StringUtils.isBlank(updateProductVersionRequestDTO.getVersionName())) {
                versions.setVersionName(updateProductVersionRequestDTO.getVersionName());
            }
            productApplicationRepository.deleteWithNativeQuery(versions.getId());
            List<ProductApplications> productApplications = updateProductVersionRequestDTO.getApplications().stream().map(application ->
                    productDeploymentUtility.buildProductApplication(application, versions)
            ).collect(Collectors.toList());
            versions.setProductApplications(productApplications);

            ProductVersions updatedVersion = productVersionRepository.save(versions);
            return mapperUtils.map(updatedVersion, ProductVersionResponseDTO.class);
        } catch (ResourceNotFoundException rex) {
            throw new ProductDeploymentBadRequestException(rex.getMessage());
        } catch (ProductDeploymentBadRequestException ex) {
            throw ex;
        } catch (Exception ex) {
            throw new ProductDeploymentException(ex.getMessage());
        }
    }

    @Transactional
    public DeleteVersionResponseDTO deleteProductVersions(Integer productVersionId, Integer productId) {

        try {
            Products product = productRepository.findOne(productId, new QuerySpec(Products.class));
            ProductVersions versions = productVersionRepository.findOne(productVersionId, new QuerySpec(ProductVersions.class));
            if (versions.getProducts().getId() != product.getId()) {
                throw new ProductDeploymentBadRequestException(String.format("productVersionId %s does not belong to productId %s", productVersionId, productId));
            }
            if (!StringUtils.isBlank(versions.getIsDeployed()) && versions.getIsDeployed().equalsIgnoreCase(Boolean.TRUE.toString())) {
                throw new ProductDeploymentBadRequestException("Cannot execute this delete request as this version is already deployed");
            }
            productVersionRepository.delete(productVersionId);
            return DeleteVersionResponseDTO.builder().isDeleted(true).productId(productId).productVersionId(productVersionId).build();
        } catch (ResourceNotFoundException rex) {
            throw new ProductDeploymentBadRequestException(rex.getMessage());
        } catch (ProductDeploymentBadRequestException ex) {
            throw ex;
        } catch (Exception ex) {
            throw new ProductDeploymentException(ex.getMessage());
        }
    }

    /*This method fetches the applications details for the provided version of the product*/
    public BulkProductDetailResponseDTO fetchBulkProductDetails(BulkProductDetailFetchRequestDTO requestDTO) {
        productDeploymentValidator.validateBulkProductDetailFetchRequestDTO(requestDTO);
        try {
            BulkProductDetailResponseDTO responseDTO = BulkProductDetailResponseDTO.builder().build();
            List<ProductVersions> versions =
                    productVersionRepository.getAllApplicationByProductNameAndVersion(requestDTO.getProductVersion());
            responseDTO.setProductDetails(versions
                    .stream()
                    .collect(Collectors.groupingBy(i -> i.getProducts().getName(), Collectors.toMap(ProductVersions::getVersionName, i -> i.getProductApplications().stream().map(app -> app.getApplicationName() + ":" + app.getApplicationVersion()).collect(Collectors.toList())
                    ))));
            return responseDTO;
        } catch (Exception ex) {
            throw new ProductDeploymentException(ex.getMessage());
        }
    }


}
