package com.mcd.restaurant.deployment.service;


import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectInputStream;
import com.amazonaws.util.IOUtils;
import com.mcd.restaurant.bundledrelease.controller.view.response.SQSApplicationResponse;
import com.mcd.restaurant.bundledrelease.enums.BundleType;
import com.mcd.restaurant.bundledrelease.enums.BundledReleaseStatus;
import com.mcd.restaurant.bundledrelease.error.BundleReleaseInvalidRequestException;
import com.mcd.restaurant.common.Constants;
import com.mcd.restaurant.common.MapperUtils;
import com.mcd.restaurant.deployment.controller.view.beans.DeploymentHelper;
import com.mcd.restaurant.deployment.controller.view.beans.DeploymentStatisticsSegment;
import com.mcd.restaurant.deployment.controller.view.beans.DownloadReportHelper;
import com.mcd.restaurant.deployment.controller.view.request.BulkInsertComponentRequestDTO;
import com.mcd.restaurant.deployment.controller.view.request.BundleDeployRequest;
import com.mcd.restaurant.deployment.controller.view.request.ComponentDetailRequestDTO;
import com.mcd.restaurant.deployment.controller.view.request.DeployComponentRequest;
import com.mcd.restaurant.deployment.controller.view.request.DeployRequest;
import com.mcd.restaurant.deployment.controller.view.request.DeploymentCheckRequestDTO;
import com.mcd.restaurant.deployment.controller.view.request.PostArchivalDeleteRequestDTO;
import com.mcd.restaurant.deployment.controller.view.request.UpdateComponentRequestDTO;
import com.mcd.restaurant.deployment.controller.view.request.UpdateDeploymentStatusRequestDTO;
import com.mcd.restaurant.deployment.controller.view.request.UpdateDeploymentStoreRequestDTO;
import com.mcd.restaurant.deployment.controller.view.request.UpdatePreDeploymentStatus;
import com.mcd.restaurant.deployment.controller.view.response.ArchivalDataResponseDTO;
import com.mcd.restaurant.deployment.controller.view.response.ArchivalDeleteResponse;
import com.mcd.restaurant.deployment.controller.view.response.BulkInsertComponentResponseDTO;
import com.mcd.restaurant.deployment.controller.view.response.DeployGraphResponseDTO;
import com.mcd.restaurant.deployment.controller.view.response.DeployResponse;
import com.mcd.restaurant.deployment.controller.view.response.DeploymentCheckResponseDTO;
import com.mcd.restaurant.deployment.controller.view.response.DeploymentGroupArchivalResponseDTO;
import com.mcd.restaurant.deployment.controller.view.response.DeploymentGroupDetailsResponseDTO;
import com.mcd.restaurant.deployment.controller.view.response.PreCheckStatusResponseDTO;
import com.mcd.restaurant.deployment.controller.view.response.StatusChangeResponse;
import com.mcd.restaurant.deployment.controller.view.response.SuggestionResponseDTO;
import com.mcd.restaurant.deployment.controller.view.response.UpdateComponentResponseDTO;
import com.mcd.restaurant.deployment.error.ComponentNotApplicableError;
import com.mcd.restaurant.deployment.error.DeploymentBadRequestException;
import com.mcd.restaurant.deployment.error.DeploymentError;
import com.mcd.restaurant.model.BundleRelease;
import com.mcd.restaurant.model.Component;
import com.mcd.restaurant.model.ComponentType;
import com.mcd.restaurant.model.DeploymentGroup;
import com.mcd.restaurant.model.DeploymentHistory;
import com.mcd.restaurant.model.DeploymentTag;
import com.mcd.restaurant.model.PrePostDeploymentModel;
import com.mcd.restaurant.model.ProductApplications;
import com.mcd.restaurant.model.ProductVersions;
import com.mcd.restaurant.model.Restaurant;
import com.mcd.restaurant.model.RestaurantView;
import com.mcd.restaurant.repository.BundleReleaseRepository;
import com.mcd.restaurant.repository.ComponentRepository;
import com.mcd.restaurant.repository.ComponentTypeRepository;
import com.mcd.restaurant.repository.DeploymentGroupRepository;
import com.mcd.restaurant.repository.DeploymentHistoryRepository;
import com.mcd.restaurant.repository.DeploymentTagRepository;
import com.mcd.restaurant.repository.MarketRepository;
import com.mcd.restaurant.repository.ProductVersionRepository;
import com.mcd.restaurant.repository.RestaurantRepository;
import com.mcd.restaurant.repository.RestaurantViewRepository;
import com.mcd.restaurant.repository.bulk.ComponentBulkCRUDRepository;
import io.crnk.core.exception.ResourceNotFoundException;
import io.crnk.core.queryspec.QuerySpec;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.sql.Timestamp;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.mcd.restaurant.deployment.service.DeploymentBuilder.buildDeploymentGroup;
import static com.mcd.restaurant.deployment.service.DeploymentBuilder.buildDeploymentHistory;
import static com.mcd.restaurant.deployment.service.DeploymentBuilder.buildDeploymentTag;

/**
 * Service to handle all deployment transaction scenarios
 */
@Slf4j
@Service
public class DeployService {

    @Autowired
    private DeploymentGroupRepository deploymentGroupRepository;
    @Autowired
    private ComponentRepository componentRepository;
    @Autowired
    private DeploymentTagRepository deploymentTagRepository;
    @Autowired
    private RestaurantRepository restaurantRepository;
    @Autowired
    private DeploymentUtils deploymentUtils;
    @Autowired
    private DeploymentHistoryRepository deploymentHistoryRepository;
    @Autowired
    private ComponentBulkCRUDRepository componentBulkCRUDRepository;
    @Autowired
    private ProductVersionRepository productVersionRepository;
    @Autowired
    private MapperUtils mapperUtils;
    @Autowired
    private ComponentTypeRepository componentTypeRepository;
    @Autowired
    private MarketRepository marketRepository;
    @Autowired
    private RestaurantViewRepository restaurantViewRepository;
    @Autowired
    private BundleReleaseRepository bundleReleaseRepository;
    @Autowired
    private AmazonS3 s3Client;
    @Value("${aws.sqs.queue.account.id}")
    String accountId;

    /**
     * This method is responsible for validating components and restaurants data and will save all
     * Deployment request if all data provided is valid.
     *
     * @param request target DeployRequest
     * @return generated deployment ids
     */
    @Transactional
    public DeployResponse saveAllDeployments(DeployRequest request) {
        try {
            BundleRelease bundleRelease = null;
            deploymentRequestValidator(request);
            ComponentType type = deploymentUtils.fetchComponentType(request.getDeploymentType().name());
            List<Restaurant> selectedRestaurants = restaurantRepository.findAll(restaurantRepository.prepareQuerySpecFilterForRestaurantByIds(request.getRestaurantIds()));
            if (selectedRestaurants.size() != request.getRestaurantIds().size()) {
                throw new DeploymentBadRequestException(Constants.INVALID_RESTAURANTS_PASSED_FOR_DEPLOYMENT);
            }
            Map<Integer, ProductVersions> versionsMap = new HashMap<>();
            if (Boolean.TRUE.equals(request.getIsProductDeployment())) {
                request.setDeployComponents(getApplicationsFromProducts(request, versionsMap));
            }
            if (Boolean.TRUE.equals(request.getIsBundleDeployment())) {
                bundleRelease = bundleReleaseRepository.findOne(request.getBundleId(), new QuerySpec(BundleRelease.class));
                if (bundleRelease.getStatus().equals(BundledReleaseStatus.STAGED.value()) || bundleRelease.getStatus().equals(BundledReleaseStatus.FAILED.value())) {
                    throw new BundleReleaseInvalidRequestException(String.format(Constants.BUNDLE_IS_HENCE_CAN_NOT_BE_DISPLAYED, bundleRelease.getStatus()));
                }
            }
            DeploymentTag deploymentTag = deploymentTagRepository.save(buildDeploymentTag(request.getTagName()));
            DeploymentGroup deploymentGroup = deploymentGroupRepository.save(buildDeploymentGroup(request, deploymentTag, versionsMap, bundleRelease));
            DeploymentHelper helper = DeploymentHelper.builder()
                    .deploymentType(request.getDeploymentType())
                    .isProductDeployment(request.getIsProductDeployment())
                    .versionsMap(versionsMap)
                    .selectedRestaurantMap(selectedRestaurants.stream().collect(Collectors.toMap(Restaurant::getId, restaurant -> restaurant)))
                    .type(type)
                    .deploymentGroup(deploymentGroup).build();
            List<DeploymentHistory> deploymentHistoryList = request.getDeployComponents().stream()
                    .map(application -> createDeploymentHistoryRecord(application, helper))
                    .flatMap(Collection::stream).collect(Collectors.toList());
            return DeployResponse.build(request, deploymentGroup, deploymentHistoryList);
        } catch (ComponentNotApplicableError | DeploymentBadRequestException e) {
            throw e;
        } catch (Exception ex) {
            throw new DeploymentError(ex.getMessage());
        }
    }

    private void deploymentRequestValidator(DeployRequest request) {
        if (Boolean.FALSE.equals(request.getIsProductDeployment()) && CollectionUtils.isEmpty(request.getDeployComponents())) {
            throw new ComponentNotApplicableError(Constants.INVALID_COMPONENT_REQUEST);
        }
        if (Boolean.TRUE.equals(request.getIsBundleDeployment()) && request.getBundleId() == null) {
            throw new DeploymentBadRequestException(Constants.BUNDLE_IS_REQUIRED_FOR_BUNDLE_DEPLOYMENT);
        }
        if (!CollectionUtils.isEmpty(request.getDeployComponents())) {
            request.getDeployComponents().forEach(i -> {
                if (i.getProductVersionId() != null) {
                    throw new DeploymentBadRequestException(Constants.PRODUCT_DEPLOYMENT_DETAILS_CAN_ONLY_BE_PASSED_IN_DEPLOYMENT_LIST);
                }
            });
        }
        if (Boolean.TRUE.equals(request.getIsProductDeployment()) && CollectionUtils.isEmpty(request.getDeployProducts())) {
            throw new ComponentNotApplicableError(Constants.INVALID_PRODUCT_REQUEST);
        }
    }

    private List<DeployComponentRequest> getApplicationsFromProducts(DeployRequest request, Map<Integer, ProductVersions> versionsMap) {
        List<DeployComponentRequest> deployComponents = !CollectionUtils.isEmpty(request.getDeployComponents()) ?
                request.getDeployComponents() : new ArrayList<>();
        request.getDeployProducts().forEach(versionId -> {
            if (request.getDeploymentType() != DeploymentType.docker) {
                throw new DeploymentBadRequestException(Constants.PRODUCT_DEPLOYMENT_IS_APPLICABLE_FOR_DOCKER_APPLICATION_ONLY);
            }
            ProductVersions versions = productVersionRepository.findOne(versionId, new QuerySpec(ProductVersions.class));
            versionsMap.put(versions.getId(), versions);
            deployComponents.addAll(versions.getProductApplications().stream().map(application -> DeployComponentRequest.builder()
                    .componentName(application.getApplicationName())
                    .updateToVersion(application.getApplicationVersion())
                    .productVersionId(versionId)
                    .build()
            ).collect(Collectors.toList()));
        });
        return deployComponents;
    }

    private List<DeploymentHistory> createDeploymentHistoryRecord(DeployComponentRequest deployComponent, DeploymentHelper helper) {
        log.info(String.format("INFO: Initiate deployment for %s, type: %s, from: %s, to: %s",
                deployComponent.getComponentName(), helper.getDeploymentType(),
                deployComponent.getUpdateFromVersion(), deployComponent.getUpdateToVersion()));
        List<Component> components = getEligibleComponents(deployComponent, helper);
        if (CollectionUtils.isEmpty(components) && helper.getDeploymentType().equals(DeploymentType.firmware)) {
            throw new ComponentNotApplicableError(String.format(
                    "The component [%s] with version [%s] is not applicable in requested restaurant",
                    deployComponent.getComponentName(), deployComponent.getUpdateFromVersion()));
        }
        List<Component> newComponents = filterNewComponents(deployComponent, helper, components);
        if (!CollectionUtils.isEmpty(newComponents)) {
            componentBulkCRUDRepository.saveAll(newComponents).forEach(components::add);
        }
        ProductVersions versions = getEligibleProductVersions(deployComponent, helper);
        if (versions != null && !Boolean.TRUE.toString().equalsIgnoreCase(versions.getIsDeployed())) {
            versions.setIsDeployed(Boolean.TRUE.toString());
            productVersionRepository.save(versions);
        }
        return getDeploymentHistories(components, deployComponent, versions, helper);
    }

    private ProductVersions getEligibleProductVersions(DeployComponentRequest deployComponent, DeploymentHelper helper) {
        if (helper.getIsProductDeployment() != null && helper.getIsProductDeployment() && deployComponent.getProductVersionId() != null) {
            return helper.getVersionsMap().get(deployComponent.getProductVersionId());
        }
        return null;
    }

    private List<Component> filterNewComponents(DeployComponentRequest deployComponent, DeploymentHelper helper, List<Component> components) {
        if (helper.getDeploymentType().equals(DeploymentType.firmware)) {
            return Collections.emptyList();
        }
        List<Integer> restaurantWithComponent = components.stream().map(i -> i.getRestaurant().getId()).collect(Collectors.toList());
        return helper.getSelectedRestaurantMap().keySet().stream()
                .filter(it -> !restaurantWithComponent.contains(it))
                .map(restaurant -> {
                    Restaurant restaurantModel = Optional.ofNullable(helper.getSelectedRestaurantMap().get(restaurant))
                            .orElseThrow(() -> new DeploymentBadRequestException("Restaurant with id" + restaurant + "does not exist"));
                    UpdateComponentRequestDTO requestDTO = UpdateComponentRequestDTO
                            .builder()
                            .applicationName(deployComponent.getComponentName())
                            .build();
                    return deploymentUtils.handleMissingComponentForDeployment(requestDTO, restaurantModel, helper.getType());
                }).collect(Collectors.toList());
    }

    private List<Component> getEligibleComponents(DeployComponentRequest deployComponent, DeploymentHelper helper) {
        switch (helper.getDeploymentType()) {
            case docker:
                return componentRepository.getFilteredDockerComponents(deployComponent, helper.getSelectedRestaurantMap().keySet());
            case firmware:
                return componentRepository.getFilteredFirmwareComponents(deployComponent, helper.getSelectedRestaurantMap().keySet());
            case crd:
                return componentRepository.getFilteredCrdComponents(deployComponent, helper.getSelectedRestaurantMap().keySet());
            default:
                return Collections.emptyList();
        }
    }

    private List<DeploymentHistory> getDeploymentHistories(List<Component> components,
                                                           DeployComponentRequest deployComponent,
                                                           ProductVersions finalVersions,
                                                           DeploymentHelper helper) {
        Set<Integer> restaurants = new HashSet<>();
        List<DeploymentHistory> deploymentHistories = components.stream()
                .map(it -> {
                    restaurants.add(it.getRestaurant().getId());
                    return buildDeploymentHistory(helper.getDeploymentGroup(), it, deployComponent, finalVersions, helper.getDeploymentType());
                })
                .collect(Collectors.toList());
        if (restaurants.size() < helper.getSelectedRestaurantMap().keySet().size()) {
            log.debug(String.format("INFO: Restaurants applicable %d", restaurants.size()));
            throw new ComponentNotApplicableError(String.format(
                    "The component [%s] with version [%s] is not applicable in some of the requested restaurants",
                    deployComponent.getComponentName(), deployComponent.getUpdateFromVersion()));
        }
        return deploymentHistories;

    }

    @Transactional
    public StatusChangeResponse changeDeploymentStatus(int id, DeploymentStatus deploymentStatus) {
        DeploymentGroup deploymentGroup = deploymentGroupRepository.findOne(id, new QuerySpec(DeploymentGroup.class));
        deploymentGroup.getDeploymentHistory().forEach(it -> {
            if (!it.getStatus().equalsIgnoreCase(DeploymentStatus.ALREADYPRESENT.value()))
                it.setStatus(deploymentStatus.value());
        });
        deploymentGroupRepository.save(deploymentGroup);
        return StatusChangeResponse.build(deploymentStatus);
    }

    public SuggestionResponseDTO getSuggestionList(SuggestionType suggestionType, String searchParam) {
        QuerySpec querySpec;
        List<String> suggestionList = new ArrayList<>();
        switch (suggestionType) {
            case ID:
                querySpec = deploymentUtils.prepareQuerySpecFilterForSuggestion(DeploymentGroup.class, SuggestionType.ID.value(), searchParam);
                suggestionList.addAll(deploymentGroupRepository.findAll(querySpec).stream().map(i ->
                        Integer.toString(i.getId())
                ).distinct().collect(Collectors.toList()));
                break;

            case STORE_NAME:
                querySpec = deploymentUtils.prepareQuerySpecFilterForSuggestion(Restaurant.class, SuggestionType.STORE_NAME.value(), searchParam);
                suggestionList.addAll(restaurantRepository.findAll(querySpec).stream().map(Restaurant::getName
                ).distinct().collect(Collectors.toList()));
                break;

            case DEPLOYMENT_NAME:
                querySpec = deploymentUtils.prepareQuerySpecFilterForSuggestion(DeploymentTag.class, SuggestionType.DEPLOYMENT_NAME.value(), searchParam);
                suggestionList.addAll(deploymentTagRepository.findAll(querySpec).stream().map(DeploymentTag::
                        getName
                ).distinct().collect(Collectors.toList()));
                break;
            case RESTAURANT_NO:
                querySpec = deploymentUtils.prepareQuerySpecFilterForSuggestion(Restaurant.class, SuggestionType.RESTAURANT_NO.value(), searchParam);
                suggestionList.addAll(restaurantRepository.findAll(querySpec).stream().map(Restaurant::
                        getName
                ).distinct().collect(Collectors.toList()));
                break;
            case DEPLOYED_BY:
                querySpec = deploymentUtils.prepareQuerySpecFilterForSuggestion(DeploymentGroup.class, SuggestionType.DEPLOYED_BY.value(), searchParam);
                suggestionList.addAll(deploymentGroupRepository.findAll(querySpec).stream().map(DeploymentGroup::
                        getDeployedBy
                ).distinct().collect(Collectors.toList()));
                break;
            default:
                throw new DeploymentError(Constants.UNKNOWN_SUGGESTION_TYPE);


        }
        return SuggestionResponseDTO.builder().suggestionList(suggestionList).build();
    }


    @Transactional
    public ArchivalDataResponseDTO getDeploymentGroupForArchival(Long archiveBeforeTime, Long limit, Integer offset) {
        List<DeploymentGroup> deploymentGroupList = deploymentGroupRepository.findAll(deploymentUtils.prepareQuerySpecFilterForArchival(archiveBeforeTime, limit, offset));
        List<DeploymentGroupArchivalResponseDTO> deploymentGroupArchivalResponseDTOList = mapperUtils.map(deploymentGroupList, DeploymentGroupArchivalResponseDTO.class);
        deploymentGroupArchivalResponseDTOList.parallelStream().forEach(group ->
                {
                    group.getDeploymentHistory().stream().forEach(history -> {
                        history.setComponentId(history.getComponent().getId());
                        history.setRestaurantId(history.getRestaurant().getId());
                        history.setDeploymentGroupId(history.getDeployment_group().getId());
                        history.setRestaurant(null);
                        history.setDeployment_group(null);
                        history.setComponent(null);
                        if (history.getProductVersions() != null) {
                            history.setProductVersionId(Integer.toString(history.getProductVersions().getId()));
                        }
                    });
                    group.setBundleId(group.getBundleRelease() != null ? group.getBundleRelease().getId() : null);
                    group.setBundleRelease(null);
                }
        );

        return ArchivalDataResponseDTO.builder().deploymentGroupList(deploymentGroupArchivalResponseDTOList).build();
    }

    @Transactional
    public ArchivalDeleteResponse deleteDeploymentRecordsAfterArchival(PostArchivalDeleteRequestDTO requestDTO) {
        if (CollectionUtils.isEmpty(requestDTO.getDeploymentGroupId())) {
            throw new DeploymentError(Constants.ID_LIST_CONNOT_BE_EMPTY);
        }
        try {
            //deployment history records will get deleted along with groups
            log.info(Constants.INITIATING_DELETION_OF_RECORDS_POST_ARCHIVAL + System.currentTimeMillis());
            deploymentUtils.asyncDeletePOstArchival(requestDTO);
            return ArchivalDeleteResponse.builder().isDeleted(true).build();
        } catch (Exception e) {
            log.error(Constants.EXCEPTION_OCCURED_WHILE_DELETING_THE_RECORDS_AFTER_ARCHIVAL);
            log.error(e.getMessage());
            return ArchivalDeleteResponse.builder().isDeleted(false).build();
        }
    }


    @Transactional
    public DeployGraphResponseDTO prepareStatisticsForDeployGraph(Long startDate, Long endDate, DeployGraphSegment
            deployGraphSegment) {
        try {

            boolean immediateStop = true;
            LocalDateTime segmentEndTime;
            Integer counter = 0;
            Integer totalCount = 0;
            Map<Integer, Integer> completedSegmentmap = new HashMap<>();
            Map<Integer, Integer> failedSegmentmap = new HashMap<>();

            LocalDateTime startTime = LocalDateTime.ofInstant(Instant.ofEpochMilli(startDate), ZoneId.of("UTC"));
            LocalDateTime endTime = LocalDateTime.ofInstant(Instant.ofEpochMilli(endDate), ZoneId.of("UTC"));

            while (counter < deployGraphSegment.value() && immediateStop) {
                switch (deployGraphSegment) {
                    case DAY:
                        return deploymentUtils.analyseDeploymentResultsDay(startTime.toEpochSecond(ZoneOffset.UTC) * 1000, endTime.toEpochSecond(ZoneOffset.UTC) * 1000);
                    case WEEK:
                        segmentEndTime = startTime.plusDays(deployGraphSegment.getSegment());
                        break;
                    case MONTH:
                        if (counter == 0)
                            segmentEndTime = startTime.plusDays(deployGraphSegment.getSegment());
                        else
                            segmentEndTime = startTime.plusDays(deployGraphSegment.getSegment() + 1l);
                        break;
                    default:
                        throw new DeploymentError(Constants.UNKNOWN_GRAPH_SEGMENT);
                }
                if (segmentEndTime.isAfter(endTime)) {
                    segmentEndTime = endTime;
                    immediateStop = false;
                }
                DeploymentStatisticsSegment statistics = deploymentUtils.analyseDeploymentResults(startTime.toEpochSecond(ZoneOffset.UTC) * 1000, segmentEndTime.toEpochSecond(ZoneOffset.UTC) * 1000);
                completedSegmentmap.put(counter, statistics.getCompleted());
                failedSegmentmap.put(counter, statistics.getFailed());
                totalCount = totalCount + statistics.getCompleted() + statistics.getFailed();
                startTime = segmentEndTime;
                counter++;
            }
            return DeployGraphResponseDTO.builder().failedDeployments(failedSegmentmap).completedDeployments(completedSegmentmap).totalDeployment(totalCount).build();
        } catch (Exception e) {
            throw new DeploymentError(e.getMessage());
        }
    }

    @Transactional
    public UpdateComponentResponseDTO modifyComponentRecordsForRestaurant(UpdateComponentRequestDTO
                                                                                  updateComponentRequestDTO) {
        try {
            if (StringUtils.isBlank(updateComponentRequestDTO.getApplicationName()) || StringUtils.isBlank(updateComponentRequestDTO.getApplicationVersion()) || StringUtils.isBlank(updateComponentRequestDTO.getRestaurantNo()) || StringUtils.isBlank(updateComponentRequestDTO.getMarketName())) {
                throw new DeploymentBadRequestException(Constants.APPLICATION_NAME_APPLICATION_VERSION_RESTAURANT_NAME_MARKET_NAME_IS_MANDATORY);
            }
            UpdateComponentResponseDTO updateComponentResponseDTO = UpdateComponentResponseDTO.builder().isNewComponentInserted(false).build();
            Component component;
            Restaurant restaurant = deploymentUtils.validateMarketAndRestaurant(updateComponentRequestDTO.getMarketName(), updateComponentRequestDTO.getRestaurantNo());
            List<Component> components = componentRepository.getFilteredDockerComponents(DeployComponentRequest
                    .builder()
                    .componentName(updateComponentRequestDTO.getApplicationName())
                    .build(), Stream.of(restaurant.getId()).collect(Collectors.toSet()));
            if (CollectionUtils.isEmpty(components)) {
                component = deploymentUtils.handleMissingComponent(updateComponentRequestDTO, updateComponentResponseDTO, restaurant);
            } else {
                component = components.get(0);
                component.setReportedVersion(updateComponentRequestDTO.getApplicationVersion());
            }

            component = componentRepository.save(component);
            updateComponentResponseDTO.setComponentId(component.getId());
            updateComponentResponseDTO.setIsVersionUpdated(true);
            return updateComponentResponseDTO;
        } catch (DeploymentBadRequestException ex) {
            throw ex;
        } catch (Exception e) {
            throw new DeploymentError(e.getMessage());
        }
    }


    @Transactional
    public DeploymentGroupDetailsResponseDTO getDeploymentDetails(Integer deploymentGroupId) {
        try {
            List<ProductApplications> applicationsList = new ArrayList<>();
            HashSet<String> status = getSelectedStatus();
            List<DeploymentHistory> deploymentHistoryListFull = null;
            DeploymentGroup preCheckFlow = deploymentGroupRepository.findOne(deploymentGroupId, new QuerySpec(DeploymentGroup.class));
            if (StringUtils.isBlank(preCheckFlow.getIsPreCheckFlow()) || preCheckFlow.getIsPreCheckFlow().equalsIgnoreCase(Boolean.FALSE.toString())) {
                deploymentHistoryListFull = preCheckFlow.getDeploymentHistory();
            } else {
                deploymentHistoryListFull = preCheckFlow.getDeploymentHistory().stream().filter(history ->
                        history.getSelectedStatus() == null || status.contains(history.getSelectedStatus())
                ).collect(Collectors.toList());
            }

            List<DeploymentHistory> deploymentHistoryList = deploymentHistoryListFull.stream()
                    .filter(history -> !history.getApplicationName().equalsIgnoreCase(Constants.IMAGE_CACHE))
                    .collect(Collectors.toList());
            if (CollectionUtils.isEmpty(deploymentHistoryListFull)) {
                throw new DeploymentBadRequestException(Constants.NO_DEPLOYMENT_RECORDS_PRESENT_FOR_THE_GIVEN_ID);
            }
            Set<Integer> restaurantSet = deploymentHistoryListFull
                    .stream()
                    .map(history ->
                            history.getRestaurant().getId()
                    ).collect(Collectors.toSet());
            if (!CollectionUtils.isEmpty(deploymentHistoryList)) {
                applicationsList = deploymentUtils.getUniqueProductApplicationsInDeployment(deploymentHistoryList);
            }
            List<RestaurantView> restaurantViewList = restaurantViewRepository.getRestaurantDetailsById(restaurantSet);
            Map<String, List<SQSApplicationResponse>> applicationNameSpace = deploymentUtils.updateApplicationDetailsGroupedByProduct(applicationsList, deploymentHistoryListFull);
            return deploymentUtils.buildDeploymentDetailResponseDTO(deploymentHistoryList, restaurantViewList, applicationNameSpace);
        } catch (DeploymentBadRequestException | ResourceNotFoundException ex) {
            throw new DeploymentBadRequestException(ex.getMessage());
        } catch (Exception e) {
            throw new DeploymentError(e.getMessage());
        }
    }

    private HashSet<String> getSelectedStatus() {
        HashSet<String> selectedStatus = new HashSet<>();
        selectedStatus.add(PreCheckStatus.PASSED.value());
        selectedStatus.add(PreCheckStatus.USER_APPROVED.value());
        return selectedStatus;
    }


    @Transactional
    public StatusChangeResponse updateDeploymentHistoryStatus(UpdateDeploymentStatusRequestDTO requestDTO) {
        try {
            deploymentUtils.deploymentStatusUpdateRequestValidator(requestDTO);
            Restaurant restaurant = deploymentUtils.validateMarketAndRestaurant(requestDTO.getMarketName(), requestDTO.getRestaurantNo());
            List<DeploymentHistory> deploymentHistoryList = deploymentHistoryRepository.getHistoryByGroupIdAndRestaurantIdAndApplicationNameAndApplicationVersion(requestDTO.getDeploymentGroupId(), restaurant.getId(), requestDTO.getApplicationName(), requestDTO.getApplicationVersion());
            if (CollectionUtils.isEmpty(deploymentHistoryList)) {
                throw new DeploymentBadRequestException(Constants.NO_DEPLOYMENT_RECORD_FOUND);
            }
            DeploymentHistory deploymentHistory = deploymentHistoryList.get(0);
            deploymentHistory.setStatus(requestDTO.getStatus().value());
            deploymentHistoryRepository.save(deploymentHistory);
            return StatusChangeResponse.build(requestDTO.getStatus());
        } catch (DeploymentBadRequestException ex) {
            throw ex;
        } catch (Exception e) {
            throw new DeploymentError(e.getMessage());
        }
    }

    @Transactional
    public DeployResponse saveAllBundleDeployments(BundleDeployRequest bundleDeployRequest) {
        try {
            List<Restaurant> selectedRestaurants;
            deploymentUtils.bundleDeploymentRequestValidator(bundleDeployRequest);
            Map<Integer, ProductVersions> versionsMap = new HashMap<>();
            List<DeployComponentRequest> componentRequests = new ArrayList<>();

            DeploymentGroup group = deploymentGroupRepository.findOne(bundleDeployRequest.getDeploymentGroupId(), new QuerySpec(DeploymentGroup.class));
            BundleRelease bundleRelease = group.getBundleRelease();
            DeployRequest request = mapperUtils.map(bundleDeployRequest, DeployRequest.class);
            request.setDeploymentType(DeploymentType.docker);
            if (bundleRelease.getType().equalsIgnoreCase(BundleType.PRODUCTS.value())) {
                request.setIsProductDeployment(Boolean.TRUE);
            } else {
                request.setIsProductDeployment(Boolean.FALSE);
            }
            bundleRelease.getBundleReleaseAppVersions().forEach(application -> {
                if (bundleRelease.getType().equalsIgnoreCase(BundleType.PRODUCTS.value())) {
                    versionsMap.put(application.getProductVersions().getId(), application.getProductVersions());
                    componentRequests.add(DeployComponentRequest.builder().productVersionId(application.getProductVersions().getId()).updateToVersion(application.getVersion()).componentName(application.getAppName()).build());
                } else {
                    componentRequests.add(DeployComponentRequest.builder().updateToVersion(application.getVersion()).componentName(application.getAppName()).build());
                }
            });
            request.setDeployComponents(componentRequests);
            if (group.getIsPreCheckFlow().equalsIgnoreCase("true"))
                selectedRestaurants = new ArrayList<>(group.getDeploymentHistory().stream().filter(i -> i.getSelectedStatus().equalsIgnoreCase(PreCheckStatus.PASSED.value()) || i.getSelectedStatus().equalsIgnoreCase(PreCheckStatus.USER_APPROVED.value())).map(DeploymentHistory::getRestaurant).collect(Collectors.toSet()));
            else
                selectedRestaurants = new ArrayList<>(group.getDeploymentHistory().stream().map(DeploymentHistory::getRestaurant).collect(Collectors.toSet()));
            ComponentType type = deploymentUtils.fetchComponentType(request.getDeploymentType().name());
            Map<Integer, Restaurant> selectedRestaurantMap = selectedRestaurants.stream().collect(Collectors.toMap(Restaurant::getId, restaurant -> restaurant));
            request.setRestaurantIds(selectedRestaurantMap.keySet());
            DeploymentHelper helper = DeploymentHelper.builder()
                    .deploymentType(request.getDeploymentType())
                    .isProductDeployment(request.getIsProductDeployment())
                    .versionsMap(versionsMap)
                    .selectedRestaurantMap(selectedRestaurantMap)
                    .type(type)
                    .deploymentGroup(group).build();
            List<DeploymentHistory> deploymentHistoryList = request.getDeployComponents().stream()
                    .map(application -> createDeploymentHistoryRecord(application, helper))
                    .flatMap(Collection::stream).collect(Collectors.toList());
            return DeployResponse.build(request, group, deploymentHistoryList);
        } catch (ComponentNotApplicableError | DeploymentBadRequestException e) {
            throw e;
        } catch (Exception ex) {
            throw new DeploymentError(ex.getMessage());
        }

    }

    @Transactional
    public BulkInsertComponentResponseDTO insertComponentData(BulkInsertComponentRequestDTO componentDTO) {
        try {
            deploymentUtils.validateRequest(componentDTO);
            Restaurant restaurant = deploymentUtils.validateMarketAndRestaurant(componentDTO.getMarketName(), componentDTO.getStoreName());
            List<String> components = componentDTO.getComponents().stream().map(ComponentDetailRequestDTO::getApplicationName).collect(Collectors.toList());
            List<Component> existingComponents = deploymentUtils.fetchExistingComponents(components, restaurant);
            Map<String, String> mapRequest = componentDTO.getComponents().stream().collect(Collectors.toMap(ComponentDetailRequestDTO::getApplicationName, ComponentDetailRequestDTO::getApplicationVersion));
            List<String> existingComponentName = new ArrayList<>();
            existingComponents.forEach(component -> {
                if (mapRequest.containsKey(component.getName())) {
                    component.setReportedVersion(mapRequest.get(component.getName()));
                    existingComponentName.add(component.getName());
                }
            });
            List<ComponentDetailRequestDTO> newComponents = componentDTO.getComponents().stream().filter(component -> !existingComponentName.contains(component.getApplicationName())).collect(Collectors.toList());
            List<com.mcd.restaurant.model.Component> prepareResponse = deploymentUtils
                    .bulkInsertComponentsData(newComponents, restaurant, existingComponents);
            return deploymentUtils.bulkInsertComponentResponse(prepareResponse, restaurant, componentDTO.getMarketName());

        } catch (DeploymentBadRequestException ex) {
            throw ex;
        } catch (Exception e) {
            throw new DeploymentError(e.getMessage());
        }
    }


    @Transactional
    public SuggestionResponseDTO getApplicationForDeployment(SuggestionType suggestionType, String applicationName) {
        SuggestionResponseDTO responseDTO = SuggestionResponseDTO.builder().build();
        deploymentUtils.validateInputForFetchingApplicationDetails(suggestionType, applicationName);
        try {
            if (suggestionType.equals(SuggestionType.APPLICATION_NAME)) {

                responseDTO.setSuggestionList(deploymentUtils.fetchAllProductApplications());
            } else if (suggestionType.equals(SuggestionType.APPLICATION_VERSION)) {
                responseDTO.setSuggestionList(deploymentUtils.fetchAllVersionForApplication(applicationName));
            }
            return responseDTO;
        } catch (Exception ex) {
            throw new DeploymentError(ex.getMessage());
        }
    }

    @Transactional
    public DeploymentCheckResponseDTO saveDeploymentCheckDetails(DeploymentCheckRequestDTO deploymentCheckRequestDTO) {
        try {
            DeploymentGroup group = deploymentUtils.deploymentCheckRequestValidator(deploymentCheckRequestDTO);
            PrePostDeploymentModel model = deploymentUtils.prepareAndSavePrePostDeploymentDetails(group, deploymentCheckRequestDTO);
            deploymentUtils.callPrePostLambda(model, group);
            return DeploymentCheckResponseDTO.build(model);
        } catch (DeploymentBadRequestException ex) {
            throw ex;
        } catch (Exception e) {
            throw new DeploymentError(e.getMessage());
        }
    }

    @Transactional
    public DownloadReportHelper downloadFile(Integer id) {
        try {
            PrePostDeploymentModel model = deploymentUtils.getPrePostModelById(id);
            if (model.getStatus().equalsIgnoreCase(DeploymentStatus.Completed.value())) {
                String bucketName = getBucketName();
                S3Object s3Object = s3Client.getObject(bucketName, model.getReportPath());
                S3ObjectInputStream inputStream = s3Object.getObjectContent();
                byte[] content = IOUtils.toByteArray(inputStream);
                return DownloadReportHelper.builder().content(content).fileName(model.getReportPath()).build();
            } else {
                throw new DeploymentBadRequestException("Report is not Generated. Current Status is: " + model.getStatus());
            }
        } catch (DeploymentBadRequestException e) {
            throw e;
        } catch (Exception e) {
            throw new DeploymentError(e.getMessage());
        }
    }

    public String getBucketName() {
        String env = deploymentUtils.getEnvironment(accountId);
        return "bre-" + env + "-pre-post-check";

    }

    @Transactional
    public PreCheckStatusResponseDTO updatePreCheckStatus(UpdatePreDeploymentStatus updatePreDeploymentStatus) {
        try {
            DeploymentGroup group = deploymentUtils.preCheckStatusValidator(updatePreDeploymentStatus);
            return PreCheckStatusResponseDTO.builder().isStatusUpdated(Boolean.TRUE).deploymentGroup(group).build();
        } catch (DeploymentBadRequestException e) {
            throw e;
        } catch (Exception e) {
            throw new DeploymentError(e.getMessage());
        }
    }

    @Transactional
    public DeployResponse modifyDeploymentStores(UpdateDeploymentStoreRequestDTO requestDTO) {
        try {
            DeploymentGroup deploymentGroup = deploymentUtils.modifyDeploymentStoresValidator(requestDTO);
            if (deploymentGroup.getStatus().equalsIgnoreCase(DeploymentStatus.Completed.value()) || deploymentGroup.getStatus().equalsIgnoreCase(DeploymentStatus.Failed.value()) ||
                    deploymentGroup.getStatus().equalsIgnoreCase(DeploymentStatus.Scheduled.value()) || (!deploymentGroup.getStatus().equalsIgnoreCase(DeploymentStatus.STAGED.value()) && !deploymentGroup.getType().equalsIgnoreCase("Bundle"))) {
                throw new DeploymentBadRequestException("Invalid operation");
            }
            Set<Integer> restaurantIds = new HashSet<>();
            DeploymentType type = DeploymentType.getDeploymentType(deploymentGroup.getDeploymentHistory().get(deploymentGroup.getDeploymentHistory().size() - 1).getType());
            List<DeployComponentRequest> request = deploymentGroup.getDeploymentHistory().stream()
                    .map(history -> {
                        restaurantIds.add(history.getRestaurant().getId());
                        return DeployComponentRequest.builder()
                                .componentName(history.getApplicationName())
                                .updateToVersion(history.getApplicationVersion())
                                .build();
                    }).collect(Collectors.toList());
            DeployRequest deployRequest = DeployRequest.builder().restaurantIds(restaurantIds).deploymentType(type).deployComponents(request).build();
            return DeployResponse.build(deployRequest, deploymentGroup, null);
        } catch (DeploymentBadRequestException e) {
            throw e;
        } catch (Exception e) {
            throw new DeploymentError(e.getMessage());
        }
    }

    @Transactional
    public void initiateDeploymentProcessing(DeploymentGroup group) {
        group.setProcessing(true);
        deploymentGroupRepository.save(group);
    }

    @Async
    @Transactional
    @org.springframework.transaction.annotation.Transactional(propagation = Propagation.REQUIRES_NEW)
    public void saveDeploymentHistoriesAsync(List<DeploymentHistory> deploymentHistoryList, int deploymentGroupId) {
        log.info("************* Start Saving Deployment History *************");
        DeploymentGroup group = deploymentGroupRepository.findOne(deploymentGroupId, new QuerySpec(DeploymentGroup.class));
        group.setDeploymentHistory(deploymentHistoryList);
        group.setProcessing(false);
        deploymentGroupRepository.save(group);
        log.info("**************** Saved {} Deployment History ****************", deploymentHistoryList.size());
    }

    @Async
    @Transactional
    @org.springframework.transaction.annotation.Transactional(propagation = Propagation.REQUIRES_NEW)
    public void updateDeploymentHistoriesAsync(UpdatePreDeploymentStatus updatePreDeploymentStatus, int deploymentGroupId) {
        DeploymentGroup group = deploymentGroupRepository.findOne(deploymentGroupId, new QuerySpec(DeploymentGroup.class));
        group.getDeploymentHistory().forEach(it -> {
            if (updatePreDeploymentStatus.getUnhealthyRestaurants().contains(it.getRestaurant().getName())) {
                it.setPreCheckStatus(PreCheckStatus.FAILED.value());
                it.setSelectedStatus(PreCheckStatus.FAILED.value());
            } else {
                it.setPreCheckStatus(PreCheckStatus.PASSED.value());
                it.setSelectedStatus(PreCheckStatus.PASSED.value());
            }
            it.setModifiedBy("System");
        });
        group.setProcessing(false);
        deploymentGroupRepository.save(group);
    }

    @Async
    @Transactional
    @org.springframework.transaction.annotation.Transactional(propagation = Propagation.REQUIRES_NEW)
    public void updateDeploymentHistoriesAsync(UpdateDeploymentStoreRequestDTO requestDTO, int deploymentGroupId) {
        DeploymentGroup group = deploymentGroupRepository.findOne(deploymentGroupId, new QuerySpec(DeploymentGroup.class));
        group.getDeploymentHistory().forEach(it -> {
            if (requestDTO.getUnSelectedRestaurants().contains(it.getRestaurant().getId())) {
                modifyHistoryStatus(it, PreCheckStatus.PASSED, PreCheckStatus.USER_REJECTED, PreCheckStatus.FAILED);
            } else {
                modifyHistoryStatus(it, PreCheckStatus.FAILED, PreCheckStatus.USER_APPROVED, PreCheckStatus.PASSED);
                if (Boolean.TRUE.equals(requestDTO.getIsStatusUpdate())) {
                    if ((!StringUtils.isBlank(it.getComponent().getReportedVersion()) &&
                            it.getComponent().getReportedVersion().equalsIgnoreCase(it.getApplicationVersion())) ||
                            it.getComponent().getType().getName().equalsIgnoreCase("CRD")) {
                        it.setStatus(DeploymentStatus.ALREADYPRESENT.value());
                    } else {
                        it.setStatus(DeploymentStatus.InProgress.value());
                    }
                    LocalDateTime currentTime = LocalDateTime.now(ZoneId.of("UTC"));
                    currentTime = currentTime.plusHours(requestDTO.getEventBridgeTime());
                    group.setDeploymentTime(new Timestamp(currentTime.toEpochSecond(ZoneOffset.UTC) * 1000));
                }
            }
            it.setModifiedBy(requestDTO.getModifiedBy());
        });
        group.setProcessing(false);
        deploymentGroupRepository.save(group);
    }

    private static void modifyHistoryStatus(DeploymentHistory it, PreCheckStatus previousStatus,
                                            PreCheckStatus userStatus, PreCheckStatus defaultStatus) {
        if (it.getPreCheckStatus().equalsIgnoreCase(previousStatus.value())) {
            it.setSelectedStatus(userStatus.value());
        } else {
            it.setSelectedStatus(defaultStatus.value());
        }
    }
}