package com.mcd.restaurant.deployment.controller;


import com.mcd.restaurant.bundledrelease.controller.view.response.SchedulerTimeResponse;
import com.mcd.restaurant.common.ResponseView;
import com.mcd.restaurant.deployment.controller.view.beans.DownloadReportHelper;
import com.mcd.restaurant.deployment.controller.view.request.BulkInsertComponentRequestDTO;
import com.mcd.restaurant.deployment.controller.view.request.BundleDeployRequest;
import com.mcd.restaurant.deployment.controller.view.request.DeployRequest;
import com.mcd.restaurant.deployment.controller.view.request.DeploymentCheckRequestDTO;
import com.mcd.restaurant.deployment.controller.view.request.PostArchivalDeleteRequestDTO;
import com.mcd.restaurant.deployment.controller.view.request.StatusChangeRequest;
import com.mcd.restaurant.deployment.controller.view.request.SuggestionRequestDTO;
import com.mcd.restaurant.deployment.controller.view.request.UpdateComponentRequestDTO;
import com.mcd.restaurant.deployment.controller.view.request.UpdateDeploymentStatusRequestDTO;
import com.mcd.restaurant.deployment.controller.view.request.UpdateDeploymentStoreRequestDTO;
import com.mcd.restaurant.deployment.controller.view.request.UpdatePreDeploymentStatus;
import com.mcd.restaurant.deployment.controller.view.response.ArchivalDataResponseDTO;
import com.mcd.restaurant.deployment.controller.view.response.ArchivalDeleteResponse;
import com.mcd.restaurant.deployment.controller.view.response.BulkInsertComponentResponseDTO;
import com.mcd.restaurant.deployment.controller.view.response.DeployGraphResponseDTO;
import com.mcd.restaurant.deployment.controller.view.response.DeployResponse;
import com.mcd.restaurant.deployment.controller.view.response.DeploymentCheckResponseDTO;
import com.mcd.restaurant.deployment.controller.view.response.DeploymentGroupDetailsResponseDTO;
import com.mcd.restaurant.deployment.controller.view.response.PreCheckStatusResponseDTO;
import com.mcd.restaurant.deployment.controller.view.response.StatusChangeResponse;
import com.mcd.restaurant.deployment.controller.view.response.SuggestionResponseDTO;
import com.mcd.restaurant.deployment.controller.view.response.UpdateComponentResponseDTO;
import com.mcd.restaurant.deployment.service.DeployGraphSegment;
import com.mcd.restaurant.deployment.service.DeployService;
import com.mcd.restaurant.deployment.service.SuggestionType;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;


/**
 * Controller responsible for bulk deployment and saving all deployment histories together.
 */
@RestController
@RequestMapping(path = "/deployment")
public class DeployController {

    @Autowired
    DeployService deployService;

    @Value("${scheduler.fail-after}")
    private Long failAfterTime;
    @Value("${scheduler.fixed-delay}")
    private Long fixedDelay;
    @Value("${scheduler.initial-delay}")
    private Long initialDelay;

    /**
     * This API will create the Deployment entries in DeploymentGroup and DeploymentHistory table
     */
    @PostMapping(path = "/records")
    public ResponseEntity<ResponseView<DeployResponse>> postDeploymentRecord(
            @RequestBody DeployRequest deployRequest) {
        DeployResponse data = deployService.saveAllDeployments(deployRequest);
        deployService.initiateDeploymentProcessing(data.getDeploymentGroup());
        deployService.saveDeploymentHistoriesAsync(data.getDeploymentHistories(), data.getDeploymentGroup().getId());
        ResponseView<DeployResponse> responseView = new ResponseView<>();
        responseView.setData(data);
        return ResponseEntity.status(HttpStatus.CREATED).body(responseView);
    }

    /**
     * This API will create the Deployment entries in DeploymentGroup and DeploymentHistory table
     */
    @PostMapping(path = "/bundle")
    public ResponseEntity<ResponseView<DeployResponse>> postDeploymentRecordForBundle(
            @RequestBody BundleDeployRequest bundleDeployRequest) {
        DeployResponse data = deployService.saveAllBundleDeployments(bundleDeployRequest);
        deployService.initiateDeploymentProcessing(data.getDeploymentGroup());
        deployService.saveDeploymentHistoriesAsync(data.getDeploymentHistories(), data.getDeploymentGroup().getId());
        ResponseView<DeployResponse> responseView = new ResponseView<>();
        responseView.setData(data);
        return ResponseEntity.status(HttpStatus.CREATED).body(responseView);
    }

    /**
     * This API updates the status of the deployment histories
     */
    @PatchMapping(path = "/group/{groupId}")
    public ResponseEntity<ResponseView<StatusChangeResponse>> postDeploymentRecord(@PathVariable int groupId, @RequestBody StatusChangeRequest request) {
        ResponseView<StatusChangeResponse> responseView = new ResponseView<>();
        responseView.setData(deployService.changeDeploymentStatus(groupId, request.getStatus()));
        return ResponseEntity.status(HttpStatus.OK).body(responseView);
    }

    /**
     * This API returns the scheduler time parameters for failing a job
     */
    @GetMapping("/scheduler/time")
    public ResponseEntity<ResponseView<SchedulerTimeResponse>> getSchedulerTime() {
        ResponseView<SchedulerTimeResponse> responseView = new ResponseView<>();
        responseView.setData(SchedulerTimeResponse.builder().failAfter(failAfterTime).fixedDelay(fixedDelay).intialDelay(initialDelay).build());
        return ResponseEntity.status(HttpStatus.OK).body(responseView);
    }

    /**
     * This API provides the suggestion list at various filtering levels in platform
     */
    @PostMapping("/suggestion")
    public ResponseEntity<ResponseView<SuggestionResponseDTO>> getSuggestion(@RequestBody SuggestionRequestDTO suggestionRequestDTO) {
        ResponseView<SuggestionResponseDTO> responseView = new ResponseView<>();
        responseView.setData(deployService.getSuggestionList(suggestionRequestDTO.getSuggestionType(), suggestionRequestDTO.getSearchParam()));
        return ResponseEntity.status(HttpStatus.OK).body(responseView);
    }

    /**
     * This API fetches the list of deployments that are eligible for archival
     */
    @GetMapping("/archival/group")
    public ResponseEntity<ResponseView<ArchivalDataResponseDTO>> getDeploymentGroupForArchival(@RequestParam(name = "archiveBefore") Long archiveBeforeTime, @RequestParam(name = "limit") Long limit, @RequestParam(name = "offset") Integer offset) {
        ResponseView<ArchivalDataResponseDTO> responseView = new ResponseView<>();
        responseView.setData(deployService.getDeploymentGroupForArchival(archiveBeforeTime, limit, offset));
        return ResponseEntity.status(HttpStatus.OK).body(responseView);
    }

    /**
     * This API deletes the deployment data post archival
     */
    @DeleteMapping("/archival/group")
    public ResponseEntity<ResponseView<ArchivalDeleteResponse>> deleteRecordPostArchival(@RequestBody PostArchivalDeleteRequestDTO postArchivalDeleteRequestDTO) {
        ResponseView<ArchivalDeleteResponse> responseView = new ResponseView<>();
        responseView.setData(deployService.deleteDeploymentRecordsAfterArchival(postArchivalDeleteRequestDTO));
        return ResponseEntity.status(HttpStatus.OK).body(responseView);
    }

    /**
     * This API provides the list of failed and completed deployment count for graph present in deploy-dashboard page
     */
    @GetMapping("/graph/details")
    public ResponseEntity<ResponseView<DeployGraphResponseDTO>> getDeploymentStatisticsForGraph(@RequestParam(name = "startDate") Long startDate, @RequestParam(name = "endDate") Long endDate, @RequestParam(name = "segment") DeployGraphSegment deployGraphSegment) {
        ResponseView<DeployGraphResponseDTO> responseView = new ResponseView<>();
        responseView.setData(deployService.prepareStatisticsForDeployGraph(startDate, endDate, deployGraphSegment));
        return ResponseEntity.status(HttpStatus.OK).body(responseView);
    }

    /**
     * This API updates the component record for a store
     */
    @PatchMapping(path = "/component")
    public ResponseEntity<ResponseView<UpdateComponentResponseDTO>> updateComponentForStore(@RequestBody UpdateComponentRequestDTO updateComponentRequestDTO) {
        ResponseView<UpdateComponentResponseDTO> responseView = new ResponseView<>();
        responseView.setData(deployService.modifyComponentRecordsForRestaurant(updateComponentRequestDTO));
        return ResponseEntity.status(HttpStatus.OK).body(responseView);
    }

    @GetMapping(path = "/details/{deploymentGroupId}")
    public ResponseEntity<ResponseView<DeploymentGroupDetailsResponseDTO>> fetchDeploymentDetails(@PathVariable(name = "deploymentGroupId") Integer deploymentGroupId) {
        ResponseView<DeploymentGroupDetailsResponseDTO> responseView = new ResponseView<>();
        responseView.setData(deployService.getDeploymentDetails(deploymentGroupId));
        return ResponseEntity.status(HttpStatus.OK).body(responseView);
    }

    @PatchMapping(path = "/status")
    public ResponseEntity<ResponseView<StatusChangeResponse>> updateDeploymentHistoryStatus(@RequestBody UpdateDeploymentStatusRequestDTO updateDeploymentStatusRequestDTO) {
        ResponseView<StatusChangeResponse> responseView = new ResponseView<>();
        responseView.setData(deployService.updateDeploymentHistoryStatus(updateDeploymentStatusRequestDTO));
        return ResponseEntity.status(HttpStatus.OK).body(responseView);
    }

    /**
     * This API insert bulk application data of docker component type
     */
    @PostMapping(path = "/component/data")
    public ResponseEntity<ResponseView<BulkInsertComponentResponseDTO>> insertComponentData(@RequestBody BulkInsertComponentRequestDTO bulkInsertComponentRequestDTO) {
        ResponseView<BulkInsertComponentResponseDTO> responseView = new ResponseView<>();
        responseView.setData(deployService.insertComponentData(bulkInsertComponentRequestDTO));
        return ResponseEntity.status(HttpStatus.CREATED).body(responseView);
    }

    /**
     * This API fetches the list of unique application and its corresponding versions
     */
    @GetMapping(path = "/application")
    public ResponseEntity<ResponseView<SuggestionResponseDTO>> fetchApplicationDetail(@RequestParam(name = "suggestionType") SuggestionType suggestionType, @RequestParam(name = "applicationName", required = false) String applicationName) {
        ResponseView<SuggestionResponseDTO> responseView = new ResponseView<>();
        responseView.setData(deployService.getApplicationForDeployment(suggestionType, applicationName));
        return ResponseEntity.status(HttpStatus.OK).body(responseView);
    }

    /**
     * This API will create the Deployment check data entries
     */
    @PostMapping(path = "/check")
    public ResponseEntity<ResponseView<DeploymentCheckResponseDTO>> postDeploymentCheckRecord(
            @RequestBody DeploymentCheckRequestDTO deploymentCheckRequestDTO) {
        ResponseView<DeploymentCheckResponseDTO> responseView = new ResponseView<>();
        responseView.setData(deployService.saveDeploymentCheckDetails(deploymentCheckRequestDTO));
        return ResponseEntity.status(HttpStatus.CREATED).body(responseView);
    }

    /**
     * This API downloads the pre/post deployment check report from s3 bucket
     */
    @GetMapping("/download/{id}")
    public ResponseEntity<ByteArrayResource> downloadFile(@PathVariable Integer id) {
        DownloadReportHelper helper = deployService.downloadFile(id);
        ByteArrayResource resource = new ByteArrayResource(helper.getContent());
        return ResponseEntity
                .ok()
                .contentLength(helper.getContent().length)
                .header("Content-type", "application/octet-stream")
                .header("Content-disposition", "attachment; filename=\"" + helper.getFileName() + "\"")
                .body(resource);
    }

    @PatchMapping(path = "/precheck/status")
    public ResponseEntity<ResponseView<PreCheckStatusResponseDTO>> updateDeploymentHistoryStatus(@RequestBody UpdatePreDeploymentStatus updatePreDeploymentStatus) {
        PreCheckStatusResponseDTO data = deployService.updatePreCheckStatus(updatePreDeploymentStatus);
        deployService.initiateDeploymentProcessing(data.getDeploymentGroup());
        deployService.updateDeploymentHistoriesAsync(updatePreDeploymentStatus, data.getDeploymentGroup().getId());
        ResponseView<PreCheckStatusResponseDTO> responseView = new ResponseView<>();
        responseView.setData(data);
        return ResponseEntity.status(HttpStatus.OK).body(responseView);
    }

    @PatchMapping(path = "/records")
    public ResponseEntity<ResponseView<DeployResponse>> updateDeploymentStore(@RequestBody UpdateDeploymentStoreRequestDTO updateDeploymentStoreRequestDTO) {
        DeployResponse data = deployService.modifyDeploymentStores(updateDeploymentStoreRequestDTO);
        deployService.initiateDeploymentProcessing(data.getDeploymentGroup());
        deployService.updateDeploymentHistoriesAsync(updateDeploymentStoreRequestDTO, data.getDeploymentGroup().getId());
        ResponseView<DeployResponse> responseView = new ResponseView<>();
        responseView.setData(data);
        return ResponseEntity.status(HttpStatus.OK).body(responseView);
    }

}
