package com.mcd.restaurant.dashboard.utils;


import com.mcd.restaurant.common.MapperUtils;
import com.mcd.restaurant.dashboard.controller.view.request.HardwareDetailRequestDTO;
import com.mcd.restaurant.dashboard.controller.view.request.HierarchyStructureBean;
import com.mcd.restaurant.dashboard.controller.view.request.HierarchyStructureCreationBean;
import com.mcd.restaurant.dashboard.controller.view.request.StoreDetailRequestDTO;
import com.mcd.restaurant.dashboard.controller.view.response.BREHierarchyNodeDataUploadResponseDTO;
import com.mcd.restaurant.dashboard.controller.view.response.DashboardDetailResponseDTO;
import com.mcd.restaurant.dashboard.controller.view.response.HardwareDetailResponseDTO;
import com.mcd.restaurant.dashboard.controller.view.response.StoreDetailResponseDTO;
import com.mcd.restaurant.dashboard.enums.*;
import com.mcd.restaurant.dashboard.enums.HierarchyLevel;
import com.mcd.restaurant.dashboard.error.StoreDashboardBadRequestException;
import com.mcd.restaurant.dashboard.error.StoreDashboardException;
import com.mcd.restaurant.dashboard.utils.csvhandler.CSVHierarchyHandler;
import com.mcd.restaurant.model.*;
import com.mcd.restaurant.repository.*;
import com.mcd.restaurant.repository.bulk.BREHierarchyNodeCRUDRepository;
import com.mcd.restaurant.repository.bulk.RestaurantBulkRepository;
import com.mcd.restaurant.repository.bulk.StoreDemographicCRUDRepository;
import io.crnk.core.queryspec.FilterOperator;
import io.crnk.core.queryspec.FilterSpec;
import io.crnk.core.queryspec.PathSpec;
import io.crnk.core.queryspec.QuerySpec;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.scheduling.annotation.EnableAsync;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.web.multipart.MultipartFile;

import javax.transaction.Transactional;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Component("storeDashboardUtility")
@EnableAsync
@Slf4j
public class StoreDashboardUtility {

    @Autowired
    private BreHierarchyNodeRepository breHierarchyNodeRepository;
    @Autowired
    private CSVHierarchyHandler csvHierarchyHandler;
    @Autowired
    private StoreDemographicRepository storeDemographicRepository;
    @Autowired
    private BREHierarchyNodeCRUDRepository breHierarchyNodeCRUDRepository;
    @Autowired
    private StoreDemographicCRUDRepository storeDemographicCRUDRepository;
    @Autowired
    private RestaurantBulkRepository restaurantBulkRepository;
    @Autowired
    private StoreDataJobRepository storeDataJobRepository;
    @Autowired
    private RestaurantRepository restaurantRepository;
    @Autowired
    private MarketRepository marketRepository;
    @Autowired
    private HardwareDetailRepository hardwareDetailRepository;
    @Autowired
    private MapperUtils mapperUtils;


    public DashboardDetailResponseDTO buildDashboardDetailResponseDTO(RestaurantView restaurantView, Integer serverCount, Integer platformCount, Integer hardwareCount) {
        return DashboardDetailResponseDTO.builder()
                .id(restaurantView.getId())
                .hierarchyNodeId(restaurantView.getHierarchyNodeId())
                .storeDemographicId(restaurantView.getStoreDashboardId())
                .address(restaurantView.getStoreAddress())
                .contactNumber(restaurantView.getContactNumber())
                .ipAddress(restaurantView.getIpAddress())
                .noOfLanes(restaurantView.getNumberOfLanes())
                .market(restaurantView.getMarketName())
                .region(restaurantView.getRegionName())
                .coop(restaurantView.getCoopName())
                .platformCount(platformCount)
                .hardwareCount(hardwareCount)
                .restaurantNo(restaurantView.getRestaurantNo())
                .storeName(restaurantView.getNodeName())
                .storeStatus(restaurantView.getStoreStatus())
                .serverCountInstaller(serverCount)
                .operationalHours(restaurantView.getOperationalHours())
                .ownerType(restaurantView.getOwnerType())
                .onboardedDate(restaurantView.getCreated())
                .build();
    }


    public BreHierarchyNode insertNewRecordToBREHierarchyNode(HierarchyStructureBean hierarchyStructureBean, HierarchyLevel hierarchyLevel, Integer regionNodeId, Integer coopNodeId, Integer marketNodeId, String restaurantNo, String ipAddress) {
        BreHierarchyNode breHierarchyNode = BreHierarchyNode.builder().nodeName(hierarchyStructureBean.getNodeName()).hierarchyLevel(hierarchyLevel.value()).regionNodeId(regionNodeId).coopNodeId(coopNodeId).marketNodeId(marketNodeId).restaurantNo(restaurantNo).ipAddress(ipAddress).build();
        return breHierarchyNodeRepository.save(breHierarchyNode);
    }

    public BreHierarchyNode buildBREHierarchyNodeForStore(HierarchyStructureBean hierarchyStructureBean, HierarchyLevel hierarchyLevel, Integer regionNodeId, Integer coopNodeId, Integer marketNodeId, String restaurantNo, String ipAddress) {
        return BreHierarchyNode.builder().nodeName(hierarchyStructureBean.getNodeName()).hierarchyLevel(hierarchyLevel.value()).regionNodeId(regionNodeId).coopNodeId(coopNodeId).marketNodeId(marketNodeId).restaurantNo(restaurantNo).ipAddress(ipAddress).build();
    }


    @Async
    @Transactional
    public void saveStoreHierarchyNode(List<BreHierarchyNode> newBreHierarchyNodes, List<BreHierarchyNode> existingBreHierarchyNodes, StoreDataJob storeDataJob) {
        try {

            if (!newBreHierarchyNodes.isEmpty()) {
                breHierarchyNodeCRUDRepository.saveAll(newBreHierarchyNodes);
            }
            if (!existingBreHierarchyNodes.isEmpty()) {
                breHierarchyNodeCRUDRepository.saveAll(existingBreHierarchyNodes);
            }
            storeDataJob.setStatus(StoreDataJobStatus.COMPLETED.name());
        } catch (Exception ex) {
            storeDataJob.setStatus(StoreDataJobStatus.FAILED.name());
        } finally {
            storeDataJobRepository.saveJobWithNativeQuery(storeDataJob);
        }
    }


    @Async
    @Transactional
    public void saveStoreIPAddressInHierarchyNode(List<BreHierarchyNode> existingBreHierarchyNodes, StoreDataJob storeDataJob) {
        try {
            if (!existingBreHierarchyNodes.isEmpty()) {
                breHierarchyNodeCRUDRepository.saveAll(existingBreHierarchyNodes);
            }
            storeDataJob.setStatus(StoreDataJobStatus.COMPLETED.name());
        } catch (Exception ex) {
            storeDataJob.setStatus(StoreDataJobStatus.FAILED.name());
        } finally {
            storeDataJobRepository.saveJobWithNativeQuery(storeDataJob);
        }
    }

    @Async
    @Transactional
    public void saveBulkRestaurants(List<Restaurant> restaurants) {
        restaurantBulkRepository.saveAll(restaurants);
    }


    public void marketHierarchyNodeHandler(MultipartFile hierarchyFile, BREHierarchyNodeDataUploadResponseDTO responseDTO, Map<String, Integer> existingMarketMap) {
        List<HierarchyStructureBean> marketHierarchyStructureBeans = csvHierarchyHandler.convertHierarchyCSVToBean(hierarchyFile, "Market", "1");
        List<HierarchyStructureBean> newMarketHierarchyStructureBeans = marketHierarchyStructureBeans.stream().filter(i -> !existingMarketMap.keySet().contains(i.getNodeName())).distinct().collect(Collectors.toList());
        List<BreHierarchyNode> addedMarket = new ArrayList<>();
        newMarketHierarchyStructureBeans.forEach(i -> {
            if (validateHierarchyDataForMarketRegionCoop(i)) {
                BreHierarchyNode hierarchyNode = insertNewRecordToBREHierarchyNode(i, HierarchyLevel.MARKET, null, null, null, null, null);
                existingMarketMap.put(hierarchyNode.getNodeName(), hierarchyNode.getId());
                addedMarket.add(hierarchyNode);
            }
        });
        responseDTO.setMarketInserted(addedMarket.size());
        addedMarket.clear();
        newMarketHierarchyStructureBeans.clear();
        marketHierarchyStructureBeans.clear();
    }

    public void regionHierarchyNodeHandler(MultipartFile hierarchyFile, BREHierarchyNodeDataUploadResponseDTO responseDTO, Map<String, Integer> existingRegionMap, Map<String, Integer> existingMarketMap, Map<Integer, Integer> existingRegionMarketMap) {
        List<HierarchyStructureBean> regionHierarchyStructureBeans = csvHierarchyHandler.convertHierarchyCSVToBean(hierarchyFile, "Region", "2");
        List<HierarchyStructureBean> newRegionHierarchyStructureBeans = regionHierarchyStructureBeans.stream().filter(i -> !existingRegionMap.keySet().contains(i.getNodeName())).collect(Collectors.toList());
        List<BreHierarchyNode> addedRegion = new ArrayList<>();
        newRegionHierarchyStructureBeans.forEach(i -> {
            if (validateHierarchyDataForMarketRegionCoop(i)) {
                Integer marketId = existingMarketMap.get(i.getParentNodeName());
                if (marketId != null) {
                    BreHierarchyNode hierarchyNode = insertNewRecordToBREHierarchyNode(i, HierarchyLevel.REGION, null, null, marketId, null, null);
                    existingRegionMap.put(hierarchyNode.getNodeName(), hierarchyNode.getId());
                    existingRegionMarketMap.put(hierarchyNode.getId(), marketId);
                    addedRegion.add(hierarchyNode);
                }
            }
        });
        responseDTO.setRegionInserted(addedRegion.size());
        addedRegion.clear();
        newRegionHierarchyStructureBeans.clear();
        regionHierarchyStructureBeans.clear();
    }

    public void coopHierarchyNodeHandler(MultipartFile hierarchyFile, BREHierarchyNodeDataUploadResponseDTO responseDTO, Map<String, Integer> existingRegionMap, Map<Integer, Integer> existingRegionMarketMap, Map<String, Integer> existingCoopMap, Map<Integer, Integer> existingCoopMarketMap, Map<Integer, Integer> existingCoopRegionMap) {
        List<HierarchyStructureBean> coopHierarchyStructureBeans = csvHierarchyHandler.convertHierarchyCSVToBean(hierarchyFile, "Coop", "2");
        List<HierarchyStructureBean> newCoopHierarchyStructureBeans = coopHierarchyStructureBeans.stream().filter(i -> !existingCoopMap.keySet().contains(i.getNodeName())).collect(Collectors.toList());
        List<BreHierarchyNode> addedCoop = new ArrayList<>();
        newCoopHierarchyStructureBeans.forEach(i -> {
            if (validateHierarchyDataForMarketRegionCoop(i)) {
                Integer regionId = existingRegionMap.get(i.getParentNodeName());
                Integer marketId = existingRegionMarketMap.get(regionId);
                if (marketId != null && regionId != null) {
                    BreHierarchyNode hierarchyNode = insertNewRecordToBREHierarchyNode(i, HierarchyLevel.COOP, regionId, null, marketId, null, null);
                    existingCoopMap.put(hierarchyNode.getNodeName(), hierarchyNode.getId());
                    existingCoopMarketMap.put(hierarchyNode.getId(), marketId);
                    existingCoopRegionMap.put(hierarchyNode.getId(), regionId);
                    addedCoop.add(hierarchyNode);
                }
            }
        });
        responseDTO.setCoopInserted(addedCoop.size());
        addedCoop.clear();
        newCoopHierarchyStructureBeans.clear();
        coopHierarchyStructureBeans.clear();
    }

    public void storeHierarchyNodeHandler(MultipartFile hierarchyFile, HierarchyStructureCreationBean hierarchyStructureCreationBean, List<BreHierarchyNode> storeBreHierarchyNodes, List<BreHierarchyNode> updatedStoreBreHierarchyNodes) {
        List<HierarchyStructureBean> storeHierarchyStructureBeans = csvHierarchyHandler.convertHierarchyCSVToBean(hierarchyFile, "Store", "2");
        storeHierarchyStructureBeans.forEach(i -> {
            if (Boolean.TRUE.equals(validateHierarchyDataForStore(i))) {
                Integer coopId = hierarchyStructureCreationBean.getExistingCoopMap().get(i.getParentNodeName());
                Integer regionId = hierarchyStructureCreationBean.getExistingCoopRegionMap().get(coopId);
                Integer marketId = hierarchyStructureCreationBean.getExistingCoopMarketMap().get(coopId);
                if (marketId != null && regionId != null && coopId != null) {
                    if (!(hierarchyStructureCreationBean.getExistingStoreMap().keySet().contains(i.getRestaurantNumber()) && hierarchyStructureCreationBean.getExistingStoreMarketMap().get(i.getRestaurantNumber()).equals(marketId))) {
                        storeBreHierarchyNodes.add(BreHierarchyNode.builder().nodeName(i.getNodeName()).hierarchyLevel(HierarchyLevel.RESTAURANT.value()).regionNodeId(regionId).coopNodeId(coopId).marketNodeId(marketId).restaurantNo(i.getRestaurantNumber()).ipAddress(null).build());
                    } else {
                        updatedStoreBreHierarchyNodes.add(BreHierarchyNode.builder().nodeName(i.getNodeName()).hierarchyLevel(HierarchyLevel.RESTAURANT.value()).regionNodeId(regionId).coopNodeId(coopId).marketNodeId(marketId).restaurantNo(i.getRestaurantNumber()).ipAddress(hierarchyStructureCreationBean.getExistingStoreIpAddress().get(i.getRestaurantNumber())).id(hierarchyStructureCreationBean.getExistingStoreMap().get(i.getRestaurantNumber())).created(hierarchyStructureCreationBean.getExistingStoreCreatedMap().get(i.getRestaurantNumber())).build());
                    }

                }
            }
        });
    }


    @Async
    @Transactional
    public void saveRestaurantInBulk(List<Restaurant> restaurantUpdatedList, StoreDataJob storeDataJob) {
        try {
            restaurantBulkRepository.saveAll(restaurantUpdatedList);
            storeDataJob.setStatus(StoreDataJobStatus.COMPLETED.name());
        } catch (Exception ex) {
            storeDataJob.setStatus(StoreDataJobStatus.FAILED.name());
        } finally {
            storeDataJobRepository.saveJobWithNativeQuery(storeDataJob);
        }
    }


    @Async
    @Transactional
    public void saveChangesForStoreDemographic(List<StoreDemographic> newStoreDemographics, List<StoreDemographic> existingStoreDemographics, StoreDataJob storeDataJob) {
        try {
            storeDemographicCRUDRepository.saveAll(newStoreDemographics);
            storeDemographicCRUDRepository.saveAll(existingStoreDemographics);
            storeDataJob.setStatus(StoreDataJobStatus.COMPLETED.name());
        } catch (Exception ex) {
            storeDataJob.setStatus(StoreDataJobStatus.FAILED.name());
        } finally {
            storeDataJobRepository.saveJobWithNativeQuery(storeDataJob);
        }
    }

    public boolean validateHierarchyDataForMarketRegionCoop(HierarchyStructureBean hierarchyStructureBean) {
        return !(StringUtils.isBlank(hierarchyStructureBean.getNodeName()) || StringUtils.isBlank(hierarchyStructureBean.getParentNodeName()) || StringUtils.isBlank(hierarchyStructureBean.getNodetype()));

    }

    public Boolean validateHierarchyDataForStore(HierarchyStructureBean hierarchyStructureBean) {
        return !(StringUtils.isBlank(hierarchyStructureBean.getNodeName()) || StringUtils.isBlank(hierarchyStructureBean.getRestaurantNumber()) || StringUtils.isBlank(hierarchyStructureBean.getParentNodeName()) || StringUtils.isBlank(hierarchyStructureBean.getNodetype()));

    }

    public void validateFile(MultipartFile nodeFile) {
        String fileName = nodeFile.getOriginalFilename();
        String contentType = nodeFile.getContentType();
        if (nodeFile.isEmpty() || !(fileName != null && fileName.contains(".csv")) || !(contentType != null && contentType.equals("text/csv"))) {
            throw new StoreDashboardException("Invalid File");
        }
    }


    @Transactional
    public StoreDataJob saveStoreDataOperation(StoreJobType storeJobType) {
        StoreDataJob storeDataJob = new StoreDataJob();
        storeDataJob.setOperationType(storeJobType.name());
        storeDataJob.setStatus(StoreDataJobStatus.IN_PROGRESS.name());
        return storeDataJobRepository.create(storeDataJob);
    }


    public void validateOnGoingStoreJobs() {
        QuerySpec querySpec = new QuerySpec(StoreDataJob.class);
        FilterSpec filterNameSpec = new FilterSpec(PathSpec.of("status"), FilterOperator.EQ, StoreDataJobStatus.IN_PROGRESS.name());
        querySpec.setFilters(Stream.of(filterNameSpec).collect(Collectors.toList()));
        List<StoreDataJob> storeDataJobs = storeDataJobRepository.findAll(querySpec);
        if (!storeDataJobs.isEmpty())
            throw new StoreDashboardException("Following operation is ongoing: " + storeDataJobs.get(0).getOperationType() + ". Please try after a minute");
    }

    public Map<Integer, String> getValueNameMapForStoreStatus() {
        return Arrays.stream(StoreStatus.values()).collect(Collectors.toMap(StoreStatus::value, StoreStatus::getName));
    }

    public List<Restaurant> getRestaurantByMarketNameAndRestaurantNumber(String marketName, String restaurantNo) {
        List<Market> marketList = marketRepository.findAll(marketRepository.prepareQuerySpecFilterForFilteringMarketByName(marketName));
        if (marketList.isEmpty()) {
            throw new StoreDashboardBadRequestException(StoreDashboardConstants.MARKET_NOT_FOUND);
        }

        List<Restaurant> restaurants = restaurantRepository.findAll(restaurantRepository.prepareQuerySpecFilterForRestaurantByRestaurantNumberAndMarketId(restaurantNo, marketList.get(0).getId()));
        if (restaurants.isEmpty()) {
            throw new StoreDashboardBadRequestException(StoreDashboardConstants.STORE_NOT_FOUND);
        }
        return restaurants;
    }

    public void validateStoreDetailRequestDTO(StoreDetailRequestDTO requestDTO) {
        if (StringUtils.isBlank(requestDTO.getMarketName()) || StringUtils.isBlank(requestDTO.getRestaurantNo()) || CollectionUtils.isEmpty(requestDTO.getHardwareDetails())) {
            throw new StoreDashboardBadRequestException(StoreDashboardConstants.INVALID_HARDWARE_DETAIL_OPERATION);
        }
        requestDTO.getHardwareDetails().forEach(detail -> {
            if (StringUtils.isBlank(detail.getNodeId()) && StringUtils.isBlank(detail.getMacAddress()) && StringUtils.isBlank(detail.getModelNumber()) && StringUtils.isBlank(detail.getSerialNumber())) {
                throw new StoreDashboardBadRequestException(StoreDashboardConstants.INVALID_HARDWARE_DETAIL_OPERATION);
            }
        });
    }

    public List<HardwareDetail> getRestaurantIdAndNodeId(Integer restaurantId, List<HardwareDetailRequestDTO> hardwareDetail) {
        List<String> nodeIds = hardwareDetail.stream().map(HardwareDetailRequestDTO::getNodeId).collect(Collectors.toList());
        return hardwareDetailRepository.prepareQuerySpecFilterWithRestaurantIdAndNodeIdForHardwareDetail(restaurantId, nodeIds);
    }

    public StoreDetailResponseDTO insertHardwareDetails(List<HardwareDetail> details, Restaurant restaurant, List<HardwareDetail> hardwareDetails) {
        List<HardwareDetailResponseDTO> detailsResponse = new ArrayList<>();
        details.forEach(detail -> {
            Optional<HardwareDetail> foundNodeId = hardwareDetails.stream().filter(hardware ->
                    hardware.getNodeId().equalsIgnoreCase(detail.getNodeId())
            ).findFirst();
            HardwareDetail hardwareDetail = null;
            if (foundNodeId.isPresent()) {
                hardwareDetail = foundNodeId.get();
            }
            if (hardwareDetail != null) {
                if (detail.getNodeId().equalsIgnoreCase(hardwareDetail.getNodeId())
                        && detail.getMacAddress().equalsIgnoreCase(hardwareDetail.getMacAddress())
                        && detail.getModelNumber().equalsIgnoreCase(hardwareDetail.getModelNumber())
                        && detail.getSerialNumber().equalsIgnoreCase(hardwareDetail.getSerialNumber())
                        && detail.getManufacturer().equalsIgnoreCase(hardwareDetail.getManufacturer())
                        && detail.getModelName().equalsIgnoreCase(hardwareDetail.getModelName())
                        && detail.getVersion().equalsIgnoreCase(hardwareDetail.getVersion())
                        && detail.getFamily().equalsIgnoreCase(hardwareDetail.getFamily())
                        && detail.getOsType().equalsIgnoreCase(hardwareDetail.getOsType())
                        && detail.getOsVersion().equalsIgnoreCase(hardwareDetail.getOsVersion()))
                {
                    detailsResponse.add(mapperUtils.map(hardwareDetail,HardwareDetailResponseDTO.class));
                }
                else {
                    hardwareDetail.setMacAddress(detail.getMacAddress());
                    hardwareDetail.setModelNumber(detail.getModelNumber());
                    hardwareDetail.setSerialNumber(detail.getSerialNumber());
                    hardwareDetail.setManufacturer(detail.getManufacturer());
                    hardwareDetail.setModelName(detail.getModelName());
                    hardwareDetail.setVersion(detail.getVersion());
                    hardwareDetail.setFamily(detail.getFamily());
                    hardwareDetail.setOsType(detail.getOsType());
                    hardwareDetail.setOsVersion(detail.getOsVersion());
                    detailsResponse.add(mapperUtils.map(hardwareDetailRepository.save(hardwareDetail), HardwareDetailResponseDTO.class));
                }
            } else {
                detail.setRestaurant(restaurant);
                detailsResponse.add(mapperUtils.map(hardwareDetailRepository.save(detail), HardwareDetailResponseDTO.class));
            }
        });
        return StoreDetailResponseDTO.builder().hardwareDetails(detailsResponse).build();
    }

    @Async
    @Transactional
    @org.springframework.transaction.annotation.Transactional(propagation = Propagation.REQUIRES_NEW)
    public void saveChangesForStoreDemographicUSIT(List<StoreDemographic> newStoreDemographics) {
            log.info("********************** Start Updating StoreDemographic **********************");
            storeDemographicCRUDRepository.saveAll(newStoreDemographics);
            log.info("********************** Updated StoreDemographic **********************");
    }

}
