package com.mcd.restaurant.component.service;

import com.mcd.restaurant.common.Constants;
import com.mcd.restaurant.component.controller.view.request.DeviceDetailRequestDTO;
import com.mcd.restaurant.component.error.ComponentBadRequestException;
import com.mcd.restaurant.model.*;
import com.mcd.restaurant.repository.ComponentRepository;
import com.mcd.restaurant.repository.MarketRepository;
import com.mcd.restaurant.repository.RestaurantRepository;
import com.mcd.restaurant.repository.VendorRepository;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.sql.Timestamp;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

@Component
public class ComponentUtils {

    @Autowired
    private MarketRepository marketRepository;

    @Autowired
    private RestaurantRepository restaurantRepository;

    @Autowired
    private VendorRepository vendorRepository;

    @Autowired
    private ComponentRepository componentRepository;

    public List<Restaurant> validateMarketAndRestaurants(String marketName, List<String> restaurantNames) {
        List<Market> markets = marketRepository.findAll(marketRepository.prepareQuerySpecFilterForFilteringMarketByName(marketName));
        if (CollectionUtils.isEmpty(markets)) {
            throw new ComponentBadRequestException(Constants.NO_MARKET_PRESENT_WITH_THE_NAME + marketName);
        }
        Integer marketId = markets.get(0).getId();

        List<Restaurant> restaurants = restaurantRepository.findAll(restaurantRepository.prepareQuerySpecFilterForRestaurantByRestaurantNumbersAndMarketId(restaurantNames, marketId));
        if (CollectionUtils.isEmpty(restaurants)) {
            throw new ComponentBadRequestException(String.format(Constants.NO_RESTAURANT_PRESENT_WITH_THE_NAME_S_FOR_THE_MARKET_ID, restaurantNames, marketId));
        }
        return restaurants;
    }

    public void validateRequest(List<DeviceDetailRequestDTO> deviceDetailRequest) {

        deviceDetailRequest.forEach(device -> {

            if (CollectionUtils.isEmpty(device.getDeviceProps())) {
                throw new ComponentBadRequestException(Constants.DEVICE_PROP_DETAILS_ARE_REQUIRED_SUCH_AS_LANE_NUMBER_IP_ADDRESS_MAC_ADDRESS);
            }

            if (StringUtils.isBlank(device.getMarketName()) || StringUtils.isBlank(device.getRestaurantName()) ||
                    StringUtils.isBlank(device.getReportedVersion()) || StringUtils.isBlank(device.getModelNumber())
                    || StringUtils.isBlank(device.getSerialNumber()) || StringUtils.isBlank(device.getHardwareVersion())) {
                throw new ComponentBadRequestException(Constants.MARKET_NAME_RESTAURANT_NAME_REPORTED_VERSION_MODEL_NUMBER_SERIAL_NUMBER_HARDWARE_VERSION_ARE_REQUIRED_FOR_THIS_OPERATION);
            }

            device.getDeviceProps().forEach(prop -> {
                        if (StringUtils.isBlank(prop.getPropertyValue())) {
                            throw new ComponentBadRequestException(String.format("Value of %s is required for this operation ", prop.getPropertyName()));
                        }
                        if (StringUtils.isBlank(prop.getPropertyName()) || DeviceComponentPropType.getTypeFromValue(prop.getPropertyName()) == null) {

                            throw new ComponentBadRequestException(String.format("%s is not valid for this operation ", prop.getPropertyName()));
                        }
                    }
            );
        });
    }

    public List<com.mcd.restaurant.model.Component> getAlreadyExistedRestroDevices(List<DeviceDetailRequestDTO> deviceDetailRequest) {
        List<com.mcd.restaurant.model.Component> alreadyExistedRestroDevices = new ArrayList<>();
        Map<String, List<String>> restaurantNamesBySerialNumbers = deviceDetailRequest.stream().collect(Collectors.groupingBy(DeviceDetailRequestDTO::getRestaurantName, Collectors.mapping(DeviceDetailRequestDTO::getSerialNumber, Collectors.toList())));
        for (Map.Entry<String, List<String>> restaurants : restaurantNamesBySerialNumbers.entrySet()) {
            List<com.mcd.restaurant.model.Component> existedComponent = validateRestaurants(restaurants.getKey());
            if (!CollectionUtils.isEmpty(existedComponent)) {
                //check if a new device is installed
                // inactive the device which is not present in the deviceDetailRequest, it's their in alreadyExistedRestroDevices
                checkAndInactiveOldDevice(restaurants, existedComponent);
                alreadyExistedRestroDevices.addAll(existedComponent);
            }
        }
        return alreadyExistedRestroDevices;
    }

    public List<com.mcd.restaurant.model.Component> validateRestaurants(String restaurantName) {

        List<Restaurant> restaurants = restaurantRepository.findAll(restaurantRepository.prepareQuerySpecFilterForRestaurantByName(restaurantName));
        Restaurant restro = restaurants.get(0);
        return componentRepository.getAllComponentByRestaurantId(restro.getId());
    }

    public List<Restaurant> validateMarketsWithRestaurants(List<DeviceDetailRequestDTO> deviceDetailRequest) {
        Map<String, List<String>> restaurantDetailsMap = deviceDetailRequest.stream().collect(Collectors.groupingBy(DeviceDetailRequestDTO::getMarketName, Collectors.mapping(DeviceDetailRequestDTO::getRestaurantName, Collectors.toList())));
        List<Restaurant> restaurantDetails = new ArrayList<>();
        for (Map.Entry<String, List<String>> key : restaurantDetailsMap.entrySet()) {
            List<Restaurant> selectedRestaurants = validateMarketAndRestaurants(key.getKey(), key.getValue());
            if (CollectionUtils.isEmpty(selectedRestaurants)) {
                throw new ComponentBadRequestException(Constants.RESTAURANT_IS_NOT_FOUND_UNDER_THE_MARKET);
            } else {
                restaurantDetails.addAll(selectedRestaurants);
            }
        }
        return restaurantDetails;
    }

    public List<com.mcd.restaurant.model.Component> insertDeviceData(List<DeviceDetailRequestDTO> deviceDetailRequest, List<Restaurant> restaurantDetails, ComponentType type, Vendor vendor, List<com.mcd.restaurant.model.Component> alreadyExistedRestroDevices) {
        List<com.mcd.restaurant.model.Component> componentData = null;

        componentData =
                deviceDetailRequest.stream().map(device -> {

                    Restaurant restaurant = getRestaurant(device, restaurantDetails);

                    //check if device is already existed
                    com.mcd.restaurant.model.Component existedDevice = checkExistedDevice(device, alreadyExistedRestroDevices);

                    if (existedDevice != null) {
                        return updateDeviceDetails(existedDevice, device);
                    } else {
                        com.mcd.restaurant.model.Component comp = com.mcd.restaurant.model.Component.builder()
                                .installDate(new Timestamp(System.currentTimeMillis()))
                                .status(DeviceStatusType.ACTIVE.getValue())
                                .modelNumber(device.getModelNumber())
                                .name(StringUtils.isBlank(device.getDeviceName()) ? vendor.getName() + "-" + vendor.getId() + "-" + device.getSerialNumber() : device.getDeviceName())
                                .reportedVersion(device.getReportedVersion())
                                .serialNumber(device.getSerialNumber())
                                .warranty(yearAfterDate())
                                .restaurant(restaurant)
                                .vendor(vendor)
                                .type(type)
                                .hardwareVersion(device.getHardwareVersion())
                                .displayName(device.getMarketName() + restaurant.getId() + "CAM")
                                .build();

                        List<ComponentProp> componentProps = new ArrayList<>();
                        device.getDeviceProps().forEach(prop -> {
                            componentProps.add(buildComponentProp(prop.getPropertyName(), prop.getPropertyValue(), comp));
                            if (prop.getPropertyName().equalsIgnoreCase(DeviceComponentPropType.LANENUMBER.getValue())) {
                                if (StringUtils.isBlank(device.getDeviceName())) {
                                    comp.setName(comp.getName() + "-" + prop.getPropertyValue());
                                }
                                comp.setDisplayName(comp.getDisplayName() + "0" + prop.getPropertyValue());
                            }
                        });
                        com.mcd.restaurant.model.Component componentDatum = insertDevicesInfo(comp, parentGroup -> componentProps);
                        componentRepository.save(componentDatum);
                        return componentDatum;
                    }

                }).collect(Collectors.toList());

        return componentData;
    }

    public Restaurant getRestaurant(DeviceDetailRequestDTO device, List<Restaurant> restaurantDetails) {
        Optional<Restaurant> restaurantOptional = restaurantDetails.stream().filter(restro ->
                restro.getName().equalsIgnoreCase(device.getRestaurantName())
        ).findFirst();

        Restaurant restaurant = null;
        if (restaurantOptional.isPresent()) {
            restaurant = restaurantOptional.get();
        }
        return restaurant;
    }

    public com.mcd.restaurant.model.Component checkExistedDevice(DeviceDetailRequestDTO deviceDetail, List<com.mcd.restaurant.model.Component> alreadyExistedRestroDevices) {
        Optional<com.mcd.restaurant.model.Component> existedDeviceOptional = alreadyExistedRestroDevices.stream().filter(restro -> restro.getSerialNumber().equalsIgnoreCase(deviceDetail.getSerialNumber())).findFirst();
        com.mcd.restaurant.model.Component device = null;
        if (existedDeviceOptional.isPresent()) {
            device = existedDeviceOptional.get();
        }
        return device;
    }

    public com.mcd.restaurant.model.Component updateDeviceDetails(com.mcd.restaurant.model.Component existedDevice, DeviceDetailRequestDTO updateRequestDeviceDetail) {
        HashMap<String, String> devicePropMap = new HashMap<>();
        updateRequestDeviceDetail.getDeviceProps().forEach(prop ->
                devicePropMap.put(String.valueOf(prop.getPropertyName()), prop.getPropertyValue())
        );
        if (!StringUtils.isBlank(updateRequestDeviceDetail.getDeviceName())) {
            existedDevice.setName(updateRequestDeviceDetail.getDeviceName());
        }
        existedDevice.setModelNumber(updateRequestDeviceDetail.getModelNumber());
        existedDevice.setReportedVersion(updateRequestDeviceDetail.getReportedVersion());
        existedDevice.setHardwareVersion(updateRequestDeviceDetail.getHardwareVersion());
        existedDevice.getComponentProps().forEach(device ->
                device.setPropertyValue(devicePropMap.get(device.getPropertyName()))
        );

        componentRepository.save(existedDevice);
        return existedDevice;

    }

    public ComponentProp buildComponentProp(String propertyName, String propertyValue, com.mcd.restaurant.model.Component comp) {
        return ComponentProp.builder()
                .propertyName(String.valueOf(propertyName))
                .propertyValue(propertyValue)
                .component(comp)
                .build();
    }

    public com.mcd.restaurant.model.Component insertDevicesInfo(com.mcd.restaurant.model.Component componentData, Function<com.mcd.restaurant.model.Component, List<ComponentProp>> componentProp) {
        componentData.setComponentProps(componentProp.apply(componentData));
        return componentData;
    }

    public Vendor fetchVendorKey(String vendorName) {
        List<Vendor> vendors = vendorRepository.getVendorByName(vendorName);
        if (vendors.isEmpty()) {
            throw new ComponentBadRequestException(Constants.VENDOR_NAME_NOT_FOUND);
        }
        return vendors.get(0);
    }

    public Timestamp yearAfterDate() {
        Calendar cal = Calendar.getInstance();
        cal.add(Calendar.YEAR, 1);
        Date yearAfterDateFromToday = cal.getTime();
        return new Timestamp(yearAfterDateFromToday.getTime());
    }

    public void checkAndInactiveOldDevice(Map.Entry<String, List<String>> deviceDetailRequest, List<com.mcd.restaurant.model.Component> alreadyExistedRestroDevices) {

        alreadyExistedRestroDevices.forEach(device ->
        {
            if (!deviceDetailRequest.getValue().contains(device.getSerialNumber())) {
                device.setStatus(DeviceStatusType.INACTIVE.getValue());
                componentRepository.save((device));
            }
        });

    }
}
