package com.mcd.restaurant.common;

import com.mcd.restaurant.service.ApiLoggingService;
import io.crnk.core.engine.document.ErrorData;
import io.crnk.core.engine.document.ErrorDataBuilder;
import io.crnk.core.engine.error.ErrorResponse;
import io.crnk.core.engine.error.ExceptionMapper;
import io.crnk.core.engine.http.HttpStatus;
import io.crnk.core.utils.Prioritizable;
import org.hibernate.exception.ConstraintViolationException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.persistence.PersistenceException;

/**
 * ExceptionMapper for JPA {@link PersistenceException} in order to add error id and hide error details.
 */
@Component
public class AssetPersistenceExceptionMapper implements ExceptionMapper<PersistenceException>, Prioritizable {

    @Autowired
    private ApiLoggingService apiLoggingService;

    @Override
    public ErrorResponse toErrorResponse(PersistenceException exception) {
        Throwable cause = exception.getCause();
        String errorId = apiLoggingService.logError(cause != null ? cause : exception);

        int status = HttpStatus.INTERNAL_SERVER_ERROR_500;
        String code = ErrorCode.EXCEPTION_MAPPER_JPA_500;
        if (cause != null && cause.getClass().isAssignableFrom(ConstraintViolationException.class)) {
            status = HttpStatus.UNPROCESSABLE_ENTITY_422;
            code = ErrorCode.EXCEPTION_MAPPER_CONSTRAIN_VIOLATION_422;
        }

        ErrorDataBuilder builder = ErrorData.builder()
                .setId(errorId)
                .setCode(code)
                .setStatus(Integer.toString(status))
                .setDetail("There was an issue with processing your request. Please try after sometime.");
        return ErrorResponse.builder()
                .setStatus(status)
                .setSingleErrorData(builder.build())
                .build();
    }

    @Override
    public PersistenceException fromErrorResponse(ErrorResponse errorResponse) {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean accepts(ErrorResponse errorResponse) {
        return false;
    }

    @Override
    public int getPriority() {
        return -1;
    }
}
