package com.mcd.restaurant.productVersions;

import com.mcd.restaurant.model.ProductVersions;
import com.mcd.restaurant.model.Products;
import com.mcd.restaurant.productversions.controller.view.request.CreateProductRequestDTO;
import com.mcd.restaurant.productversions.controller.view.request.ProductApplicationRequestDTO;
import com.mcd.restaurant.productversions.controller.view.request.UpdateProductRequestDTO;
import com.mcd.restaurant.productversions.controller.view.request.UpdateProductVersionRequestDTO;
import com.mcd.restaurant.productversions.error.ProductDeploymentBadRequestException;
import com.mcd.restaurant.productversions.error.ProductDeploymentException;
import com.mcd.restaurant.productversions.service.ProductDeploymentService;
import com.mcd.restaurant.productversions.utils.ProductDeploymentValidator;
import com.mcd.restaurant.repository.ProductRepository;
import com.mcd.restaurant.repository.ProductVersionRepository;
import io.crnk.core.exception.ResourceNotFoundException;
import io.crnk.core.queryspec.QuerySpec;
import io.crnk.core.resource.list.DefaultResourceList;
import io.crnk.core.resource.list.ResourceList;
import org.hibernate.exception.ConstraintViolationException;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;

import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertThrows;

class ProductDeploymentServiceTest {

    @InjectMocks
    private ProductDeploymentService productDeploymentService;
    @Mock
    private ProductDeploymentValidator productDeploymentValidator;
    @Mock
    private ProductRepository productRepository;
    @Mock
    private ProductVersionRepository productVersionRepository;

    @BeforeEach
    void setUp() {
        MockitoAnnotations.initMocks(this);
    }


    @Test
    void saveAllProductApplicationMissingRequest() {
        CreateProductRequestDTO requestDTO1 = mockCreateProductRequestDTO();
        requestDTO1.setProductName(null);
        CreateProductRequestDTO requestDTO2 = mockCreateProductRequestDTO();
        requestDTO2.setApplications(null);
        CreateProductRequestDTO requestDTO3 = mockCreateProductRequestDTO();
        requestDTO3.getApplications().get(0).setApplicationName(null);
        CreateProductRequestDTO requestDTO4 = mockCreateProductRequestDTO();
        requestDTO4.getApplications().get(0).setApplicationVersion(null);
        Mockito.doCallRealMethod().when(productDeploymentValidator).createProductRequestDTOValidator(Mockito.any(CreateProductRequestDTO.class));
        Mockito.doCallRealMethod().when(productDeploymentValidator).productApplicationRequestDTOValidator(Mockito.any(ProductApplicationRequestDTO.class));
        for (CreateProductRequestDTO requestDTO : Stream.of(requestDTO1, requestDTO2, requestDTO3, requestDTO4).collect(Collectors.toList())) {
            assertThrows(ProductDeploymentBadRequestException.class,
                    () -> productDeploymentService.saveAllProductApplication(requestDTO));

        }
    }

    @Test
    void saveAllProductApplicationExistingProductName() {
        CreateProductRequestDTO requestDTO = mockCreateProductRequestDTO();
        Mockito.doCallRealMethod().when(productDeploymentValidator).createProductRequestDTOValidator(Mockito.any(CreateProductRequestDTO.class));
        Mockito.doCallRealMethod().when(productDeploymentValidator).productApplicationRequestDTOValidator(Mockito.any(ProductApplicationRequestDTO.class));
        Mockito.when(productRepository.getProductByName(Mockito.anyString())).thenReturn(mockProductModel());
        Exception ex = assertThrows(ProductDeploymentBadRequestException.class,
                () -> productDeploymentService.saveAllProductApplication(requestDTO));
        assertEquals(String.format("Product with product name %s already exist", requestDTO.getProductName()), ex.getMessage());

    }

    @Test
    void saveAllProductApplicationGenericException() {
        CreateProductRequestDTO requestDTO = mockCreateProductRequestDTO();
        Mockito.doCallRealMethod().when(productDeploymentValidator).createProductRequestDTOValidator(Mockito.any(CreateProductRequestDTO.class));
        Mockito.doCallRealMethod().when(productDeploymentValidator).productApplicationRequestDTOValidator(Mockito.any(ProductApplicationRequestDTO.class));
        Mockito.when(productRepository.getProductByName(Mockito.anyString())).thenThrow(ConstraintViolationException.class);
        assertThrows(ProductDeploymentException.class,
                () -> productDeploymentService.saveAllProductApplication(requestDTO));

    }

    @Test
    void deleteProductVersionsProductNotFound() {
        Mockito.when(productRepository.findOne(Mockito.anyInt(), Mockito.any(QuerySpec.class))).thenThrow(ResourceNotFoundException.class);
        assertThrows(ProductDeploymentBadRequestException.class,
                () -> productDeploymentService.deleteProductVersions(2, 3));
    }

    @Test
    void deleteProductVersionsProductVersionInvalidMatch() {
        Products product1 = new Products();
        product1.setId(2);
        Products product2 = new Products();
        product2.setId(1);
        ProductVersions versions = new ProductVersions();
        versions.setProducts(product1);
        Mockito.when(productRepository.findOne(Mockito.anyInt(), Mockito.any(QuerySpec.class))).thenReturn(product2);
        Mockito.when(productVersionRepository.findOne(Mockito.anyInt(), Mockito.any(QuerySpec.class))).thenReturn(versions);
        Exception ex = assertThrows(ProductDeploymentBadRequestException.class,
                () -> productDeploymentService.deleteProductVersions(2, 3));
        assertEquals(String.format("productVersionId %s does not belong to productId %s", 2, 3), ex.getMessage());
    }

    @Test
    void deleteProductVersionsProductVersionDeployed() {
        Products product1 = new Products();
        product1.setId(2);
        ProductVersions versions = new ProductVersions();
        versions.setIsDeployed("True");
        versions.setProducts(product1);
        Mockito.when(productRepository.findOne(Mockito.anyInt(), Mockito.any(QuerySpec.class))).thenReturn(product1);
        Mockito.when(productVersionRepository.findOne(Mockito.anyInt(), Mockito.any(QuerySpec.class))).thenReturn(versions);
        Exception ex = assertThrows(ProductDeploymentBadRequestException.class,
                () -> productDeploymentService.deleteProductVersions(2, 3));
        assertEquals("Cannot execute this delete request as this version is already deployed", ex.getMessage());
    }

    @Test
    void deleteProductVersionsProductVersionNullPointer() {
        Mockito.when(productRepository.findOne(Mockito.anyInt(), Mockito.any(QuerySpec.class))).thenThrow(NullPointerException.class);
        assertThrows(ProductDeploymentException.class,
                () -> productDeploymentService.deleteProductVersions(2, 3));
    }

    @Test
    void updateProductsMissingRequest() {
        UpdateProductRequestDTO requestDTO1 = mockUpdateProductRequestDTO();
        requestDTO1.setProductId(null);
        UpdateProductRequestDTO requestDTO2 = mockUpdateProductRequestDTO();
        requestDTO2.setApplications(null);
        UpdateProductRequestDTO requestDTO3 = mockUpdateProductRequestDTO();
        requestDTO3.getApplications().get(0).setApplicationName(null);
        UpdateProductRequestDTO requestDTO4 = mockUpdateProductRequestDTO();
        requestDTO4.getApplications().get(0).setApplicationVersion(null);
        UpdateProductRequestDTO requestDTO5 = mockUpdateProductRequestDTO();
        requestDTO5.setVersionName(null);
        Mockito.doCallRealMethod().when(productDeploymentValidator).updateProductRequestDTOValidator(Mockito.any(UpdateProductRequestDTO.class));
        Mockito.doCallRealMethod().when(productDeploymentValidator).productApplicationRequestDTOValidator(Mockito.any(ProductApplicationRequestDTO.class));
        for (UpdateProductRequestDTO requestDTO : Stream.of(requestDTO1, requestDTO2, requestDTO3, requestDTO4, requestDTO5).collect(Collectors.toList())) {
            assertThrows(ProductDeploymentBadRequestException.class,
                    () -> productDeploymentService.updateProducts(requestDTO));

        }

    }

    @Test
    void updateProductsIdNotFound() {
        UpdateProductRequestDTO requestDTO = mockUpdateProductRequestDTO();
        Mockito.when(productRepository.findOne(Mockito.anyInt(), Mockito.any(QuerySpec.class))).thenThrow(ResourceNotFoundException.class);
        Mockito.doCallRealMethod().when(productDeploymentValidator).updateProductRequestDTOValidator(Mockito.any(UpdateProductRequestDTO.class));
        Mockito.doCallRealMethod().when(productDeploymentValidator).productApplicationRequestDTOValidator(Mockito.any(ProductApplicationRequestDTO.class));
        assertThrows(ProductDeploymentBadRequestException.class,
                () -> productDeploymentService.updateProducts(requestDTO));


    }

    @Test
    void updateProductsNullPointer() {
        UpdateProductRequestDTO requestDTO = mockUpdateProductRequestDTO();
        Mockito.when(productRepository.findOne(Mockito.anyInt(), Mockito.any(QuerySpec.class))).thenThrow(NullPointerException.class);
        Mockito.doCallRealMethod().when(productDeploymentValidator).updateProductRequestDTOValidator(Mockito.any(UpdateProductRequestDTO.class));
        Mockito.doCallRealMethod().when(productDeploymentValidator).productApplicationRequestDTOValidator(Mockito.any(ProductApplicationRequestDTO.class));
        assertThrows(ProductDeploymentException.class,
                () -> productDeploymentService.updateProducts(requestDTO));


    }

    @Test
    void updateProductsVersionExist() {
        UpdateProductRequestDTO requestDTO = mockUpdateProductRequestDTO();
        Products product = new Products();
        product.setId(2);
        product.setName("Abc");
        ProductVersions versions = new ProductVersions();
        Mockito.when(productRepository.findOne(Mockito.anyInt(), Mockito.any(QuerySpec.class))).thenReturn(product);
        Mockito.doCallRealMethod().when(productDeploymentValidator).updateProductRequestDTOValidator(Mockito.any(UpdateProductRequestDTO.class));
        Mockito.doCallRealMethod().when(productDeploymentValidator).productApplicationRequestDTOValidator(Mockito.any(ProductApplicationRequestDTO.class));
        Mockito.when(productVersionRepository.getProductVersionByProductIdAndVersionName(Mockito.anyInt(), Mockito.anyString())).thenReturn(Stream.of(versions).collect(Collectors.toList()));
        Exception ex = assertThrows(ProductDeploymentBadRequestException.class,
                () -> productDeploymentService.updateProducts(requestDTO));
        assertEquals(String.format("Product-version %s already exist for the product %s", requestDTO.getVersionName(), product.getName()), ex.getMessage());
    }

    @Test
    void updateProductsVersionBadRequest() {
        UpdateProductVersionRequestDTO requestDTO1 = mockUpdateProductVersionRequestDTO();
        requestDTO1.setProductId(null);
        UpdateProductVersionRequestDTO requestDTO2 = mockUpdateProductVersionRequestDTO();
        requestDTO2.setProductVersionId(null);
        UpdateProductVersionRequestDTO requestDTO3 = mockUpdateProductVersionRequestDTO();
        requestDTO3.setApplications(null);
        UpdateProductVersionRequestDTO requestDTO4 = mockUpdateProductVersionRequestDTO();
        requestDTO4.getApplications().get(0).setApplicationVersion(null);
        UpdateProductVersionRequestDTO requestDTO5 = mockUpdateProductVersionRequestDTO();
        Mockito.when(productRepository.findOne(Mockito.anyInt(), Mockito.any())).thenThrow(ResourceNotFoundException.class);
        Mockito.doCallRealMethod().when(productDeploymentValidator).updateProductVersion(Mockito.any(UpdateProductVersionRequestDTO.class));
        for (UpdateProductVersionRequestDTO requestDTO : Stream.of(requestDTO1, requestDTO2, requestDTO3, requestDTO4, requestDTO5).collect(Collectors.toList())) {
            assertThrows(ProductDeploymentBadRequestException.class,
                    () -> productDeploymentService.updateProductVersions(requestDTO));
        }
    }

    @Test
    void updateProductsVersionBadRequestProductVersionMissMatch() {
        UpdateProductVersionRequestDTO requestDTO = mockUpdateProductVersionRequestDTO();
        Products product1=new Products();
        product1.setId(1);
        Products product2=new Products();
        product2.setId(2);
        ProductVersions version=new ProductVersions();
        version.setProducts(product2);
        Mockito.when(productRepository.findOne(Mockito.anyInt(), Mockito.any())).thenReturn(product1);
        Mockito.when(productVersionRepository.findOne(Mockito.anyInt(), Mockito.any())).thenReturn(version);
        Mockito.doCallRealMethod().when(productDeploymentValidator).updateProductVersion(Mockito.any(UpdateProductVersionRequestDTO.class));
        Exception ex=assertThrows(ProductDeploymentBadRequestException.class,
                () -> productDeploymentService.updateProductVersions(requestDTO));
        assertEquals(String.format("productVersionId %s does not belong to productId %s", 1, 1), ex.getMessage());


    }

    @Test
    void updateProductsVersionBadRequestProductVersionDeployed() {
        UpdateProductVersionRequestDTO requestDTO = mockUpdateProductVersionRequestDTO();
        Products product1=new Products();
        product1.setId(1);
        ProductVersions version=new ProductVersions();
        version.setProducts(product1);
        version.setIsDeployed("true");
        Mockito.when(productRepository.findOne(Mockito.anyInt(), Mockito.any())).thenReturn(product1);
        Mockito.when(productVersionRepository.findOne(Mockito.anyInt(), Mockito.any())).thenReturn(version);
        Mockito.doCallRealMethod().when(productDeploymentValidator).updateProductVersion(Mockito.any(UpdateProductVersionRequestDTO.class));
        Exception ex=assertThrows(ProductDeploymentBadRequestException.class,
                () -> productDeploymentService.updateProductVersions(requestDTO));
        assertEquals("Cannot execute this update request as this version is already deployed", ex.getMessage());

    }

    @Test
    void updateProductsVersionBadRequestProductVersionInvalidUpdate() {
        UpdateProductVersionRequestDTO requestDTO = mockUpdateProductVersionRequestDTO();
        Products product1=new Products();
        product1.setName("abc");
        product1.setId(1);
        ProductVersions version=new ProductVersions();
        version.setVersionName("1.1");
        version.setProducts(product1);
        ProductVersions versionDummy=new ProductVersions();
        versionDummy.setId(2);
        Mockito.when(productRepository.findOne(Mockito.anyInt(), Mockito.any())).thenReturn(product1);
        Mockito.when(productVersionRepository.findOne(Mockito.anyInt(), Mockito.any())).thenReturn(version);
        Mockito.doCallRealMethod().when(productDeploymentValidator).updateProductVersion(Mockito.any(UpdateProductVersionRequestDTO.class));
        Mockito.when(productVersionRepository.getProductVersionByProductIdAndVersionName(Mockito.anyInt(),Mockito.anyString())).thenReturn(Stream.of(versionDummy).collect(Collectors.toList()));
        Exception ex=assertThrows(ProductDeploymentBadRequestException.class,
                () -> productDeploymentService.updateProductVersions(requestDTO));
        assertEquals(String.format("Product-version %s already exist for the product %s", requestDTO.getVersionName(), product1.getName()), ex.getMessage());

    }

    public UpdateProductRequestDTO mockUpdateProductRequestDTO() {
        UpdateProductRequestDTO requestDTO = new UpdateProductRequestDTO();
        requestDTO.setProductId(1);
        requestDTO.setVersionName("Abc");
        requestDTO.setApplications(mockProductApplicationRequestDTO());
        return requestDTO;
    }

    public UpdateProductVersionRequestDTO mockUpdateProductVersionRequestDTO() {
        UpdateProductVersionRequestDTO requestDTO = new UpdateProductVersionRequestDTO();
        requestDTO.setProductId(1);
        requestDTO.setProductVersionId(1);
        requestDTO.setVersionName("Abc");
        requestDTO.setApplications(mockProductApplicationRequestDTO());
        return requestDTO;
    }

    public CreateProductRequestDTO mockCreateProductRequestDTO() {
        CreateProductRequestDTO requestDTO = new CreateProductRequestDTO();
        requestDTO.setProductName("IOT");
        requestDTO.setApplications(mockProductApplicationRequestDTO());
        return requestDTO;
    }

    public List<ProductApplicationRequestDTO> mockProductApplicationRequestDTO() {
        ProductApplicationRequestDTO requestDTO = new ProductApplicationRequestDTO();
        requestDTO.setApplicationName("bred-helloworld");
        requestDTO.setApplicationVersion("90");
        requestDTO.setApplicationURL("Test");
        return Stream.of(requestDTO).collect(Collectors.toList());
    }

    public ResourceList<Products> mockProductModel() {
        Products products = new Products();
        DefaultResourceList<Products> productsDefaultResourceList = new DefaultResourceList<>();
        productsDefaultResourceList.add(products);
        return productsDefaultResourceList;
    }
}
