package com.mcd.restaurant;

import io.restassured.RestAssured;
import io.restassured.builder.RequestSpecBuilder;
import io.restassured.specification.RequestSpecification;
import net.minidev.json.JSONObject;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.CsvSource;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.web.server.LocalServerPort;
import org.springframework.restdocs.RestDocumentationContextProvider;
import org.springframework.restdocs.RestDocumentationExtension;
import org.springframework.restdocs.operation.preprocess.Preprocessors;
import org.springframework.restdocs.restassured3.RestAssuredRestDocumentation;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit.jupiter.SpringExtension;

import static org.hamcrest.CoreMatchers.is;

@ExtendWith({SpringExtension.class, RestDocumentationExtension.class})
@SpringBootTest(webEnvironment = SpringBootTest.WebEnvironment.RANDOM_PORT)
@ActiveProfiles("test")
class OwnerOperatorTests {

    private RequestSpecification spec;

    @LocalServerPort
    int port;

    @BeforeEach
    public void setUp(RestDocumentationContextProvider restDocumentation) {
        RestAssured.port = port;

        this.spec = new RequestSpecBuilder()
                .addFilter(RestAssuredRestDocumentation.documentationConfiguration(restDocumentation)
                        .operationPreprocessors().withResponseDefaults(Preprocessors.prettyPrint()))
                .build();
    }

    @Test
    void getOwnerWhenIDIsInvalid() {
        RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/owner_operators/100")
                .then().assertThat().statusCode(is(404));
    }

    @Test
    void getOwnerWhenIDIsNull() {
        RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/owner_operators/null")
                .then().assertThat().statusCode(is(500));
    }

    @Test
    void getOwnerWhenIDIsMissing() {
        RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/owner_operators/").then()
                .statusCode(is(200));
    }

    @Test
    void postOwnerWhenInputJSONIsEmpty() {
        String jsonBody = "";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(jsonBody).when()
                .post("/restaurant_assets/owner_operators").then().assertThat().statusCode(is(400));
    }

    @Test
    void postOwnerWhenInputJSONIsInvalid() {
        String jsonBody = "{}";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(jsonBody).when()
                .post("/restaurant_assets/owner_operators").then().assertThat().statusCode(is(400));
    }

    @Test
    void postOwnerWhenInputJSONIsInvalidDataKeyMissing() {
        JSONObject body = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "owner2");
        // fill data JSON
        body.put("type", "owner_operators");
        body.put("attributes", attributeJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/owner_operators").then().assertThat().statusCode(is(400));
    }

    @Test
    void postOwnerWhenTypeNameBothMissing() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("onsiteSupportEngineer", "Onsite_Support_Engineer_New");
        attributeJSON.put("preferences", "Preferences_New");
        attributeJSON.put("contactInfo", "123-456-789");
        attributeJSON.put("breSubscription", "Bre_Subscription_New");
        attributeJSON.put("contactInfoType", "phone");
        dataJSON.put("attributes", attributeJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/owner_operators").then().assertThat().statusCode(is(422));
    }

    @Test
    void postOwnerWhenOnsiteSupportEngineerIsMissing() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Owner2");
        attributeJSON.put("preferences", "Preferences_New");
        attributeJSON.put("contactInfo", "123-456-789");
        attributeJSON.put("breSubscription", "Bre_Subscription_New");
        attributeJSON.put("contactInfoType", "phone");
        // fill data JSON
        dataJSON.put("type", "owner_operators");
        dataJSON.put("attributes", attributeJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/owner_operators").then().assertThat().statusCode(is(201));
    }

    @Test
    void postOwnerWhenOnsiteSupportEngineerIsEmpty() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Owner2");
        attributeJSON.put("onsiteSupportEngineer", "");
        attributeJSON.put("preferences", "Preferences_New");
        attributeJSON.put("contactInfo", "123-456-789");
        attributeJSON.put("breSubscription", "Bre_Subscription_New");
        attributeJSON.put("contactInfoType", "phone");
        // fill data JSON
        dataJSON.put("type", "owner_operators");
        dataJSON.put("attributes", attributeJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/owner_operators").then().assertThat().statusCode(is(201));
    }

    @Test
    void postOwnerWhenPreferncesIsMissing() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Owner2");
        attributeJSON.put("onsiteSupportEngineer", "Onsite_Support_Engineer_New");
        attributeJSON.put("contactInfo", "123-456-789");
        attributeJSON.put("breSubscription", "Bre_Subscription_New");
        attributeJSON.put("contactInfoType", "phone");
        // fill data JSON
        dataJSON.put("type", "owner_operators");
        dataJSON.put("attributes", attributeJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/owner_operators").then().assertThat().statusCode(is(201));
    }

    @Test
    void postOwnerWhenPreferncesIsEmpty() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Owner2");
        attributeJSON.put("onsiteSupportEngineer", "Onsite_Support_Engineer_New");
        attributeJSON.put("preferences", "");
        attributeJSON.put("contactInfo", "123-456-789");
        attributeJSON.put("breSubscription", "Bre_Subscription_New");
        attributeJSON.put("contactInfoType", "phone");
        // fill data JSON
        dataJSON.put("type", "owner_operators");
        dataJSON.put("attributes", attributeJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/owner_operators").then().assertThat().statusCode(is(201));
    }

    @Test
    void postOwnerWhenContactInfoIsMissing() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Owner2");
        attributeJSON.put("onsiteSupportEngineer", "Onsite_Support_Engineer_New");
        attributeJSON.put("preferences", "Preferences_New");
        attributeJSON.put("breSubscription", "Bre_Subscription_New");
        attributeJSON.put("contactInfoType", "phone");
        // fill data JSON
        dataJSON.put("type", "owner_operators");
        dataJSON.put("attributes", attributeJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/owner_operators").then().assertThat().statusCode(is(201));
    }

    @Test
    void postOwnerWhenContactInfoIsEmpty() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Owner2");
        attributeJSON.put("onsiteSupportEngineer", "Onsite_Support_Engineer_New");
        attributeJSON.put("preferences", "preferces1");
        attributeJSON.put("contactInfo", "");
        attributeJSON.put("breSubscription", "Bre_Subscription_New");
        attributeJSON.put("contactInfoType", "phone");
        // fill data JSON
        dataJSON.put("type", "owner_operators");
        dataJSON.put("attributes", attributeJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/owner_operators").then().assertThat().statusCode(is(201));
    }

    @Test
    void postOwnerWhenBreSubscriptionIsMissing() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Owner2");
        attributeJSON.put("onsiteSupportEngineer", "Onsite_Support_Engineer_New");
        attributeJSON.put("preferences", "Preferences_New");
        attributeJSON.put("contactInfoType", "phone");
        // fill data JSON
        dataJSON.put("type", "owner_operators");
        dataJSON.put("attributes", attributeJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/owner_operators").then().assertThat().statusCode(is(201));
    }

    @Test
    void postOwnerWhenBreSubscriptionIsEmpty() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Owner2");
        attributeJSON.put("onsiteSupportEngineer", "Onsite_Support_Engineer_New");
        attributeJSON.put("preferences", "preferces1");
        attributeJSON.put("contactInfo", "12-34-556");
        attributeJSON.put("breSubscription", "");
        attributeJSON.put("contactInfoType", "phone");
        // fill data JSON
        dataJSON.put("type", "owner_operators");
        dataJSON.put("attributes", attributeJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/owner_operators").then().assertThat().statusCode(is(201));
    }

    @Test
    void postOwnerWhenContactInfoTypeIsMissing() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Owner2");
        attributeJSON.put("onsiteSupportEngineer", "Onsite_Support_Engineer_New");
        attributeJSON.put("preferences", "Preferences_New");
        attributeJSON.put("contactInfo", "12-34-556");
        attributeJSON.put("breSubscription", "Bre_Subscription_New");
        // fill data JSON
        dataJSON.put("type", "owner_operators");
        dataJSON.put("attributes", attributeJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/owner_operators").then().assertThat().statusCode(is(201));
    }

    @Test
    void postOwnerWhenContactInfoTypeIsEmpty() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Owner2");
        attributeJSON.put("onsiteSupportEngineer", "Onsite_Support_Engineer_New");
        attributeJSON.put("preferences", "preferces1");
        attributeJSON.put("contactInfo", "12-34-556");
        attributeJSON.put("breSubscription", "Bre_Subscription_New");
        attributeJSON.put("contactInfoType", "");
        // fill data JSON
        dataJSON.put("type", "owner_operators");
        dataJSON.put("attributes", attributeJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/owner_operators").then().assertThat().statusCode(is(201));
    }

    @ParameterizedTest
    @CsvSource({"12,404", "abc,500", "null,500", ",500"})
    void patchOwnerWhenOwnerIdIsInvalid(String id, int code) {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Owner2");
        attributeJSON.put("onsiteSupportEngineer", "Onsite_Support_Engineer_New");
        attributeJSON.put("preferences", "Preferences_New");
        attributeJSON.put("contactInfo", "123-456-789");
        attributeJSON.put("breSubscription", "Bre_Subscription_New");
        attributeJSON.put("contactInfoType", "phone");
        // fill data JSON
        dataJSON.put("type", "owner_operators");
        dataJSON.put("attributes", attributeJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .patch("/restaurant_assets/owner_operators/abc").then().assertThat().statusCode(is(500));
    }

    @Test
    void patchOwnerWhenOwnerIdIsMising() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "Owner2");
        attributeJSON.put("onsiteSupportEngineer", "Onsite_Support_Engineer_New");
        attributeJSON.put("preferences", "Preferences_New");
        attributeJSON.put("contactInfo", "123-456-789");
        attributeJSON.put("breSubscription", "Bre_Subscription_New");
        attributeJSON.put("contactInfoType", "phone");
        // fill data JSON
        dataJSON.put("type", "owner_operators");
        dataJSON.put("attributes", attributeJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .patch("/restaurant_assets/owner_operators/").then().assertThat().statusCode(is(400));
    }


    @Test
    void patchOwnerWhenInputJSONIsEmptyString() {
        String body = "";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .patch("/restaurant_assets/owner_operators/1").then().assertThat().statusCode(is(400));
    }

    @Test
    void patchOwnerWhenInputJSONIsEmptyAndOwnerIdIsInvalid() {
        String body = "";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .patch("/restaurant_assets/owner_operators/abc").then().assertThat().statusCode(is(500));
    }
}
