package com.mcd.restaurant;

import io.restassured.RestAssured;
import io.restassured.builder.RequestSpecBuilder;
import io.restassured.specification.RequestSpecification;
import net.minidev.json.JSONObject;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.web.server.LocalServerPort;
import org.springframework.restdocs.RestDocumentationContextProvider;
import org.springframework.restdocs.RestDocumentationExtension;
import org.springframework.restdocs.operation.preprocess.Preprocessors;
import org.springframework.restdocs.restassured3.RestAssuredRestDocumentation;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit.jupiter.SpringExtension;

import static org.hamcrest.CoreMatchers.is;

@ExtendWith({SpringExtension.class, RestDocumentationExtension.class})
@SpringBootTest(webEnvironment = SpringBootTest.WebEnvironment.RANDOM_PORT)
@ActiveProfiles("test")
class DeploymentGroupTests {

    private RequestSpecification spec;

    @LocalServerPort
    int port;

    @BeforeEach
    public void setUp(RestDocumentationContextProvider restDocumentation) {
        RestAssured.port = port;

        this.spec = new RequestSpecBuilder()
                .addFilter(RestAssuredRestDocumentation.documentationConfiguration(restDocumentation)
                        .operationPreprocessors().withResponseDefaults(Preprocessors.prettyPrint()))
                .build();
    }

    @Test
    void getdeploymentGroupWhenIDIsInvalid() {
        RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/deployment_groups/456")
                .then().assertThat().statusCode(is(404));
    }

    @Test
    void getdeploymentGroupWhenIDIsNull() {
        RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/deployment_groups/null")
                .then().assertThat().statusCode(is(500));
    }

    @Test
    void getdeploymentGroupWhenIDIsMissing() {
        RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/deployment_groups/")
                .then().statusCode(is(200));
    }

    @Test
    void postdeploymentGroupWhenInputJSONIsEmpty() {
        String jsonBody = "";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(jsonBody).when()
                .post("/restaurant_assets/deployment_groups").then().assertThat().statusCode(is(400));
    }

    @Test
    void postdeploymentGroupWhenInputJSONIsInvalid() {
        String jsonBody = "{}";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(jsonBody).when()
                .post("/restaurant_assets/deployment_groups").then().assertThat().statusCode(is(400));
    }

    @Test
    void postdeploymentGroupWhenInputJSONIsInvalidDataKeyMissing() {
        JSONObject body = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("deployedVersion", "0.0.3");
        attributeJSON.put("type", "docker");
        // fill data JSON
        body.put("type", "deployment_groups");
        body.put("attributes", attributeJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/deployment_groups").then().assertThat().statusCode(is(400));
    }

    @Test
    void patchComponentPropertiesWhenComponentPropertiesTyeIdIsInvalid() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("deployedVersion", "0.0.5");
        attributeJSON.put("type", "docker");
        // fill data JSON
        dataJSON.put("type", "deployment_groups");
        dataJSON.put("attributes", attributeJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .patch("/restaurant_assets/deployment_groups/abc").then().assertThat().statusCode(is(500));
    }

    @Test
    void patchComponentPropertiesWhenComponentPropertiesIdIsMising() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("deployedVersion", "0.0.3");
        attributeJSON.put("type", "docker");
        // fill data JSON
        dataJSON.put("type", "deployment_groups");
        dataJSON.put("attributes", attributeJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .patch("/restaurant_assets/deployment_groups/").then().assertThat().statusCode(is(400));
    }

    @Test
    void patchComponentPropertiesWhenComponentPropertiesIdIsNull() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        attributeJSON.put("name", "deployment_groups2");
        attributeJSON.put("description", "ffff");
        // fill data JSON
        dataJSON.put("type", "deployment_groups");
        dataJSON.put("attributes", attributeJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .patch("/restaurant_assets/deployment_groups/null").then().assertThat().statusCode(is(500));
    }

    @Test
    void patchComponentPropertiesWhenInputJSONIsEmptyString() {
        String body = "";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .patch("/restaurant_assets/deployment_groups/1").then().assertThat().statusCode(is(400));
    }

    @Test
    void patchComponentPropertiesWhenInputJSONIsEmptyAndComponentPropertiesIdIsInvalid() {
        String body = "";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .patch("/restaurant_assets/deployment_groups/abc").then().assertThat().statusCode(is(500));
    }
}
