package com.mcd.restaurant;

import io.restassured.RestAssured;
import io.restassured.builder.RequestSpecBuilder;
import io.restassured.specification.RequestSpecification;
import net.minidev.json.JSONObject;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.web.server.LocalServerPort;
import org.springframework.restdocs.RestDocumentationContextProvider;
import org.springframework.restdocs.RestDocumentationExtension;
import org.springframework.restdocs.operation.preprocess.Preprocessors;
import org.springframework.restdocs.restassured3.RestAssuredRestDocumentation;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit.jupiter.SpringExtension;

import static org.hamcrest.CoreMatchers.is;

@ExtendWith({SpringExtension.class, RestDocumentationExtension.class})
@SpringBootTest(webEnvironment = SpringBootTest.WebEnvironment.RANDOM_PORT)
@ActiveProfiles("test")
class ComponentAuditHistoryTests {

    private RequestSpecification spec;

    @LocalServerPort
    int port;

    @BeforeEach
    public void setUp(RestDocumentationContextProvider restDocumentation) {
        RestAssured.port = port;

        this.spec = new RequestSpecBuilder()
                .addFilter(RestAssuredRestDocumentation.documentationConfiguration(restDocumentation).operationPreprocessors()
                        .withResponseDefaults(Preprocessors.prettyPrint()))
                .build();
    }

    @Test
    void getComponentAuditHistoryWhenIDIsInvalid() {
        RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/component_audit_history/123")
                .then().assertThat().statusCode(is(404));
    }

    @Test
    void getComponentAuditHistoryWhenIDIsNull() {
        RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/component_audit_history/null")
                .then().assertThat().statusCode(is(500));
    }

    @Test
    void getComponentAuditHistoryWhenIDIsMissing() {
        RestAssured.given(this.spec).accept("application/json").when().get("/restaurant_assets/component_audit_history/").then()
                .statusCode(is(200));
    }

    @Test
    void postComponentAuditHistoryWhenInputJSONIsEmpty() {
        String jsonBody = "";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(jsonBody).when()
                .post("/restaurant_assets/component_audit_history").then().assertThat().statusCode(is(400));
    }

    @Test
    void postComponentAuditHistoryWhenInputJSONIsInvalid() {
        String jsonBody = "{}";

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(jsonBody).when()
                .post("/restaurant_assets/component_audit_history").then().assertThat().statusCode(is(400));
    }

    @Test
    void postComponentAuditHistoryWhenInputJSONIsInvalidDataKeyMissing() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        JSONObject subDataJSON = new JSONObject();
        JSONObject outerDataJSON = new JSONObject();
        JSONObject componentJSON = new JSONObject();
        attributeJSON.put("action", "triggered from onboarding lambda");
        attributeJSON.put("srcOfAction", "iot onboarding lambda");
        attributeJSON.put("descript", "triggered from onboarding lambda for logs");
        subDataJSON.put("type", "component");
        subDataJSON.put("id", 1);
        outerDataJSON.put("data", subDataJSON);
        componentJSON.put("components", outerDataJSON);
        // fill data JSON
        dataJSON.put("type", "component_audit_history");
        dataJSON.put("attributes", attributeJSON);
        dataJSON.put("relationships", componentJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/component_props").then().assertThat().statusCode(is(400));
    }

    @Test
    void postComponentAuditHistoryWhenComponentIdIsMissing() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        JSONObject subDataJSON = new JSONObject();
        JSONObject outerDataJSON = new JSONObject();
        JSONObject componentJSON = new JSONObject();
        attributeJSON.put("action", "triggered from onboarding lambda");
        attributeJSON.put("srcOfAction", "iot onboarding lambda");
        attributeJSON.put("description", "triggered from onboarding lambda for logs");
        subDataJSON.put("type", "component");
        subDataJSON.put("id", 125);
        outerDataJSON.put("data", subDataJSON);
        componentJSON.put("components", outerDataJSON);
        // fill data JSON
        dataJSON.put("type", "component_audit_history");
        dataJSON.put("attributes", attributeJSON);
        dataJSON.put("relationships", componentJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/component_audit_history").then().assertThat().statusCode(is(500));
    }

    @Test
    void postComponentAuditHistoryWhenComponentIdIsEmpty() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        JSONObject subDataJSON = new JSONObject();
        JSONObject outerDataJSON = new JSONObject();
        JSONObject componentJSON = new JSONObject();
        attributeJSON.put("action", "triggered from onboarding lambda");
        attributeJSON.put("srcOfAction", "iot onboarding lambda");
        attributeJSON.put("description", "triggered from onboarding lambda for logs");
        subDataJSON.put("type", "component");
//		subDataJSON.put("id", 1);
        outerDataJSON.put("data", subDataJSON);
        componentJSON.put("components", outerDataJSON);
        // fill data JSON
        dataJSON.put("type", "component_audit_history");
        dataJSON.put("attributes", attributeJSON);
        dataJSON.put("relationships", componentJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/component_audit_history").then().assertThat().statusCode(is(500));
    }

    @Test
    void postComponentAuditHistoryWhenComponentIdIsNull() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        JSONObject subDataJSON = new JSONObject();
        JSONObject outerDataJSON = new JSONObject();
        JSONObject componentJSON = new JSONObject();
        attributeJSON.put("action", "triggered from onboarding lambda");
        attributeJSON.put("srcOfAction", "iot onboarding lambda");
        attributeJSON.put("description", "triggered from onboarding lambda for logs");
        subDataJSON.put("type", "component");
        subDataJSON.put("id", null);
        outerDataJSON.put("data", subDataJSON);
        componentJSON.put("components", outerDataJSON);
        // fill data JSON
        dataJSON.put("type", "component_audit_history");
        dataJSON.put("attributes", attributeJSON);
        dataJSON.put("relationships", componentJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/component_audit_history").then().assertThat().statusCode(is(500));
    }

    @Test
    void postComponentAuditHistoryWhenComponentIdIsInvalid() {
        JSONObject body = new JSONObject();
        JSONObject dataJSON = new JSONObject();
        JSONObject attributeJSON = new JSONObject();
        JSONObject subDataJSON = new JSONObject();
        JSONObject outerDataJSON = new JSONObject();
        JSONObject componentJSON = new JSONObject();
        attributeJSON.put("action", "triggered from onboarding lambda");
        attributeJSON.put("srcOfAction", "iot onboarding lambda");
        attributeJSON.put("description", "triggered from onboarding lambda for logs");
        subDataJSON.put("type", "component");
        subDataJSON.put("id", 1);
        outerDataJSON.put("data", subDataJSON);
        componentJSON.put("components", outerDataJSON);
        // fill data JSON
        dataJSON.put("type", "component_audit_history");
        dataJSON.put("attributes", attributeJSON);
        dataJSON.put("relationships", componentJSON);
        body.put("data", dataJSON);

        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body.toString()).when()
                .post("/restaurant_assets/component_audit_history").then().assertThat().statusCode(is(500));
    }

    @Test
    void patchComponentAuditHistoryWhenInputJSONIsEmptyString() {
        String body = "";
        RestAssured.given(this.spec).contentType("application/vnd.api+json").body(body).when()
                .patch("/restaurant_assets/component_audit_history/null").then().assertThat().statusCode(is(500));
    }
}