package com.mcd.restaurant.deployment.service;

import com.mcd.restaurant.deployment.controller.view.beans.DeploymentHelper;
import com.mcd.restaurant.deployment.controller.view.request.DeployComponentRequest;
import com.mcd.restaurant.deployment.controller.view.request.DeployRequest;
import com.mcd.restaurant.model.DeploymentHistory;
import com.mcd.restaurant.model.DeploymentGroup;
import com.mcd.restaurant.model.Component;
import com.mcd.restaurant.model.ProductVersions;
import com.mcd.restaurant.model.DeploymentTag;
import com.mcd.restaurant.model.BundleRelease;
import org.apache.commons.lang3.StringUtils;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;

/**
 * Utility to Build Deployment Objects.
 */
public final class DeploymentBuilder {

    public static final String PRODUCT = "Product";
    public static final String BUNDLE = "Bundle";
    public static final String APPLICATION = "Application";
    public static final String MULTIPLE = "Multiple";

    private DeploymentBuilder() {
        // private constructor
    }

    public static DeploymentHistory buildDeploymentHistory(DeploymentGroup deploymentGroup, Component component, DeployComponentRequest deployComponentRequest, ProductVersions versions, DeploymentHelper helper) {

        DeploymentHistory deploymentHistory = new DeploymentHistory();
        deploymentHistory.setComponent(component);
        deploymentHistory.setRestaurant(component.getRestaurant());
        if ((!StringUtils.isBlank(component.getReportedVersion()) && component.getReportedVersion().equalsIgnoreCase(deployComponentRequest.getUpdateToVersion())) || component.getType().getName().equalsIgnoreCase("CRD")) {
            deploymentHistory.setStatus(DeploymentStatus.ALREADYPRESENT.value());
        } else if (deploymentGroup.getStatus().equalsIgnoreCase(DeploymentStatus.Completed.value())) {
            deploymentHistory.setStatus(DeploymentStatus.InProgress.value());
        } else {
            deploymentHistory.setStatus(deploymentGroup.getStatus());
        }
        if (helper.getDeploymentType().equals(DeploymentType.firmware)) {
            deploymentHistory.setApplicationName(deployComponentRequest.getVendor());
        } else {
            deploymentHistory.setApplicationName(deployComponentRequest.getComponentName());
        }

        deploymentHistory.setApplicationVersion(deployComponentRequest.getUpdateToVersion());
        deploymentHistory.setType(helper.getDeploymentType().name());
        deploymentHistory.setDeployment_group(deploymentGroup);
        deploymentHistory.setProductVersions(versions);
        return deploymentHistory;
    }


    public static DeploymentTag buildDeploymentTag(Optional<String> name) {
        DeploymentTag deploymentTag = new DeploymentTag();
        name.ifPresent(deploymentTag::setName);
        return deploymentTag;
    }

    public static DeploymentGroup buildDeploymentGroup(DeployRequest deployRequest, DeploymentTag deploymentTag, Map<Integer, ProductVersions> versionsMap, BundleRelease bundleRelease) {
        DeploymentGroup deploymentGroup = new DeploymentGroup();
        deploymentGroup.setDeploymentTag(deploymentTag);
        deploymentGroup.setScheduled(deployRequest.getScheduledAt());
        deploymentGroup.setStatus(deployRequest.getStatus().value());
        deploymentGroup.setDeployedBy(deployRequest.getDeployedBy());
        if (Boolean.TRUE.equals(deployRequest.getIsBundleDeployment())) {
            deploymentGroup.setBundleRelease(bundleRelease);
        }
        if (Boolean.TRUE.equals(deployRequest.getIsBundleDeployment())) {
            deploymentGroup.setDeployedVersion(bundleRelease.getReleaseName());
        } else if (Boolean.FALSE.equals(deployRequest.getIsProductDeployment()) && deployRequest.getDeployComponents().size() > 1) {
            deploymentGroup.setDeployedVersion(MULTIPLE);
        } else if (Boolean.TRUE.equals(deployRequest.getIsProductDeployment()) && deployRequest.getDeployProducts().size() > 1) {
            deploymentGroup.setDeployedVersion(MULTIPLE);
        } else {
            if (Boolean.FALSE.equals(deployRequest.getIsProductDeployment())) {
                DeployComponentRequest request = deployRequest.getDeployComponents().get(0);
                deploymentGroup.setDeployedVersion(deployRequest.getDeploymentType().equals(DeploymentType.docker) ? request.getComponentName() + ":" + request.getUpdateToVersion() : request.getVendor() + ":" + request.getUpdateToVersion());
            } else {
                ProductVersions versions = versionsMap.get(deployRequest.getDeployComponents().get(0).getProductVersionId());
                deploymentGroup.setDeployedVersion(versions.getProducts().getName() + ":" + versions.getVersionName());
            }
        }
        handleDeploymentType(deployRequest, deploymentGroup);
        return deploymentGroup;
    }

    private static void handleDeploymentType(DeployRequest deployRequest, DeploymentGroup deploymentGroup) {
        if (deployRequest.getIsBundleDeployment().equals(Boolean.TRUE)) {
            deploymentGroup.setType(BUNDLE);
        } else if (deployRequest.getIsProductDeployment().equals(Boolean.TRUE)) {
            deploymentGroup.setType(PRODUCT);
        } else {
            deploymentGroup.setType(APPLICATION);
        }
    }

    public static DeploymentGroup addHistoryToDeploymentGroup(DeploymentGroup deploymentGroup, Function<DeploymentGroup, List<DeploymentHistory>> historyProvider) {
        deploymentGroup.setDeploymentHistory(historyProvider.apply(deploymentGroup));
        return deploymentGroup;
    }


}
