package com.mcd.restaurant.bundledrelease.validator;

import com.mcd.restaurant.bundledrelease.controller.view.request.BundledReleaseRequest;
import com.mcd.restaurant.bundledrelease.controller.view.request.UpdateBundleDeploymentDetailsRequest;
import com.mcd.restaurant.bundledrelease.controller.view.request.UpdateBundleReleaseRequest;
import com.mcd.restaurant.bundledrelease.error.BundleReleaseInvalidRequestException;
import com.mcd.restaurant.repository.BundleReleaseRepository;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

/**
 * Common class to validate Bundle Release request object
 */
@Component
public class BundleReleaseValidator {
    @Autowired
    private BundleReleaseRepository bundleReleaseRepository;
    private static final String INVALID_BUNDLE_RELEASE_REQUEST = "Invalid bundle release request";
    private static final String BUNDLE_RELEASE_PRODUCT_APP_MISMATCH_REQUEST = "Product Deployment details should be only passed in productVersionIdList";

    public void validateBundleReleaseRequest(BundledReleaseRequest bundledReleaseRequest) {

        if ((CollectionUtils.isEmpty(bundledReleaseRequest.getBundledAppVersionRequestList()) && CollectionUtils.isEmpty(bundledReleaseRequest.getProductVersionIdList())) || StringUtils.isBlank(bundledReleaseRequest.getReleaseName()) || StringUtils.isBlank(bundledReleaseRequest.getCreatedBy())) {
            throw new BundleReleaseInvalidRequestException(INVALID_BUNDLE_RELEASE_REQUEST);
        }
        if (!bundleReleaseRepository.findByReleaseName(bundledReleaseRequest.getReleaseName()).isEmpty()) {
            throw new BundleReleaseInvalidRequestException("ReleaseName exist: " + bundledReleaseRequest.getReleaseName());
        }
        if (!CollectionUtils.isEmpty(bundledReleaseRequest.getBundledAppVersionRequestList())) {
            bundledReleaseRequest.getBundledAppVersionRequestList().forEach(app -> {
                if (app.getProductVersionId() != null) {
                    throw new BundleReleaseInvalidRequestException(BUNDLE_RELEASE_PRODUCT_APP_MISMATCH_REQUEST);
                }
            });
        }

    }

    public void validateUpdateBundleReleaseRequest(UpdateBundleReleaseRequest bundleReleaseRequest) {
        if ((CollectionUtils.isEmpty(bundleReleaseRequest.getBundledAppVersionRequestList()) && CollectionUtils.isEmpty(bundleReleaseRequest.getProductVersionIdList())) || bundleReleaseRequest.getReleaseId() == null || StringUtils.isBlank(bundleReleaseRequest.getUpdatedBy())) {
            throw new BundleReleaseInvalidRequestException(INVALID_BUNDLE_RELEASE_REQUEST);
        }
        if (!CollectionUtils.isEmpty(bundleReleaseRequest.getBundledAppVersionRequestList())) {
            bundleReleaseRequest.getBundledAppVersionRequestList().forEach(app -> {
                if (app.getProductVersionId() != null) {
                    throw new BundleReleaseInvalidRequestException(BUNDLE_RELEASE_PRODUCT_APP_MISMATCH_REQUEST);
                }
            });
        }

    }

    public void validateUpdateBundleDeploymentDetailsRequest(UpdateBundleDeploymentDetailsRequest deploymentDetailsRequest) {
        if (deploymentDetailsRequest.getReleaseId() == null) {
            throw new BundleReleaseInvalidRequestException("BundleId is required");
        }
        if (!StringUtils.isBlank(deploymentDetailsRequest.getGitTag()) && StringUtils.isBlank(deploymentDetailsRequest.getCommitId())) {
            throw new BundleReleaseInvalidRequestException("CommitId is required, with Git Tag");
        }
        if (!StringUtils.isBlank(deploymentDetailsRequest.getCommitId()) && StringUtils.isBlank(deploymentDetailsRequest.getGitTag())) {
            throw new BundleReleaseInvalidRequestException("GitTag is required, with CommitId");
        }
        if(!StringUtils.isBlank(deploymentDetailsRequest.getCommitId()) && !StringUtils.isBlank(deploymentDetailsRequest.getGitTag()) && deploymentDetailsRequest.getStatus()==null){
            throw new BundleReleaseInvalidRequestException("Status is required, with CommitId and Git Tag");
        }
        if (StringUtils.isBlank(deploymentDetailsRequest.getCommitId()) && StringUtils.isBlank(deploymentDetailsRequest.getGitTag()) && StringUtils.isBlank(deploymentDetailsRequest.getRemarks())) {
            throw new BundleReleaseInvalidRequestException("Invalid request");
        }
    }
}