
package com.mcd.restaurant.bundledrelease.utils;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.mcd.restaurant.bundledrelease.controller.view.response.BundleReleaseGenericResponse;
import com.mcd.restaurant.bundledrelease.controller.view.response.SQSApplicationResponse;
import com.mcd.restaurant.bundledrelease.enums.BundledReleaseStatus;
import com.mcd.restaurant.bundledrelease.error.BundleReleaseInvalidRequestException;
import com.mcd.restaurant.model.BundleRelease;
import com.mcd.restaurant.model.BundleReleaseAppVersion;
import com.mcd.restaurant.model.Restaurant;
import com.mcd.restaurant.repository.BundleReleaseAppVersionRepository;
import com.mcd.restaurant.repository.BundleReleaseRepository;
import com.mcd.restaurant.repository.ProductApplicationRepository;
import io.crnk.core.exception.BadRequestException;
import io.crnk.core.queryspec.FilterOperator;
import io.crnk.core.queryspec.FilterSpec;
import io.crnk.core.queryspec.PathSpec;
import io.crnk.core.queryspec.QuerySpec;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.services.sqs.SqsClient;
import software.amazon.awssdk.services.sqs.model.SendMessageRequest;
import software.amazon.awssdk.services.sqs.model.SendMessageResponse;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.ArrayList;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Stream;

@Component("bundleReleaseUtility")
@Slf4j
public class BundleReleaseUtility {
    @Autowired
    private BundleReleaseRepository bundleReleaseRepository;
    @Autowired
    private BundleReleaseAppVersionRepository bundleReleaseAppVersionRepository;
    @Autowired
    private ProductApplicationRepository productApplicationRepository;
    @Value("${aws.sqs.queue.url}")
    String awsSqsQueueUrl;
    SqsClient sqsClient;


    public BundleReleaseUtility(@Value("${aws.sqs.queue.region}")
                                        String regionValue) {
        Region region = Region.of(regionValue);
        sqsClient = SqsClient.builder()
                .region(region).build();
    }

    public QuerySpec prepareQuerySpecFilter(Integer releaseId) {
        QuerySpec querySpec = new QuerySpec(BundleReleaseAppVersion.class);
        FilterSpec filterSpec = new FilterSpec(PathSpec.of("bundleRelease"), FilterOperator.EQ, releaseId);
        querySpec.setFilters(Stream.of(filterSpec).collect(Collectors.toList()));
        return querySpec;
    }

    public BundleReleaseGenericResponse generateResponse(Integer size, BundleReleaseGenericResponse bundledReleaseResponse) {
        bundledReleaseResponse.setIsSuccess(Boolean.TRUE);
        bundledReleaseResponse.setTotalApps(size);
        return bundledReleaseResponse;
    }

    public BundleRelease validateBundleReleaseStatus(Integer releaseId) {
        BundleRelease bundleRelease = bundleReleaseRepository.findOne(releaseId, new QuerySpec(BundleRelease.class));
        if (bundleRelease.getStatus().equals(BundledReleaseStatus.INITIATED.value()) || bundleRelease.getStatus().equals(BundledReleaseStatus.STAGED.value()) || bundleRelease.getStatus().equals(BundledReleaseStatus.FAILED.value())) {
            throw new BundleReleaseInvalidRequestException(String.format("Bundle is '%s', hence cannot be edited", bundleRelease.getStatus()));
        }
        return bundleRelease;
    }

    public BundleRelease validateDeleteBundleReleaseStatus(Integer releaseId) {
        BundleRelease bundleRelease = bundleReleaseRepository.findOne(releaseId, new QuerySpec(BundleRelease.class));
        if (bundleRelease.getStatus().equals(BundledReleaseStatus.INITIATED.value()) || bundleRelease.getStatus().equals(BundledReleaseStatus.STAGED.value())) {
            throw new BundleReleaseInvalidRequestException(String.format("Bundle is '%s', hence cannot be edited", bundleRelease.getStatus()));
        }
        return bundleRelease;
    }


    public QuerySpec prepareQuerySpecFilterForVersionInBundleAppVersion(String appName, String appVersion) {
        FilterSpec filterSpecVersion = new FilterSpec(PathSpec.of("reportedVersion"), FilterOperator.EQ, appVersion);
        FilterSpec filterSpecName = new FilterSpec(PathSpec.of("name"), FilterOperator.EQ, appName);
        FilterSpec filterType = new FilterSpec(PathSpec.of("type.name"), FilterOperator.EQ, "docker");
        FilterSpec filterSpecFinal = FilterSpec.and(Stream.of(filterSpecName, filterSpecVersion, filterType).collect(Collectors.toList()));
        QuerySpec querySpec = new QuerySpec(com.mcd.restaurant.model.Component.class);
        querySpec.setFilters(Stream.of(filterSpecFinal).collect(Collectors.toList()));
        return querySpec;
    }


    public Map<String, String> prepareMapForSourceAppNameAndVersion(List<BundleReleaseAppVersion> bundleReleaseAppVersion) {
        return bundleReleaseAppVersion.stream()
                .collect(Collectors.toMap(BundleReleaseAppVersion::getAppName, BundleReleaseAppVersion::getVersion
                ));
    }

    public List<BundleReleaseAppVersion> findAppVersionsToBeUpdated(Integer bundleReleaseId) {
        return bundleReleaseAppVersionRepository.findAll(prepareQuerySpecFilter(bundleReleaseId));
    }


    public FilterSpec getBundleReleaseFilter(QuerySpec querySpec, String keyName) {
        if (!CollectionUtils.isEmpty(querySpec.getFilters())) {
            Optional<FilterSpec> filterSpec = querySpec.getFilters().stream().filter(query -> query.getPath().equals(PathSpec.of(keyName))).findAny();
            if (filterSpec.isPresent())
                return filterSpec.get();
        }
        throw new BadRequestException("Applied filter is not valid");
    }

    public List<Restaurant> identifyCommonRestaurant(List<List<Restaurant>> restaurantList) {

        Map<Restaurant, Integer> testMap;
        List<Restaurant> eligibleRestaurant = new ArrayList<>();
        testMap = restaurantList.stream().flatMap(List<Restaurant>::stream).collect(Collectors.toMap(Function.identity(), v -> 1, Integer::sum));

        testMap.forEach((k, v) -> {
            if (v.equals(restaurantList.size())) {
                eligibleRestaurant.add(k);
            }
        });
        return eligibleRestaurant;


    }

    public String createSQSRequestBody(List<SQSApplicationResponse> bundledApps, Integer bundleId, String releaseName, String method) throws JsonProcessingException {
        try {
            Map<String, Object> messageBodyMap = new HashMap<>();
            messageBodyMap.put("bundleId", bundleId);
            messageBodyMap.put("bundleName", releaseName);
            messageBodyMap.put("httpMethod", method);
            if (!method.equalsIgnoreCase("delete")) {
                messageBodyMap.put("applications", bundledApps);
            }
            return new ObjectMapper().writeValueAsString(messageBodyMap);
        } catch (JsonProcessingException e) {
            log.error(e.getMessage());
            throw e;
        }
    }

    @Async
    public void sendSqsMessage(List<SQSApplicationResponse> bundledApps, Integer bundleId, String releaseName, String method) throws JsonProcessingException {
        try {
            String text = createSQSRequestBody(bundledApps, bundleId, releaseName, method);
            String queueUrl = awsSqsQueueUrl;
            log.info("SQS:URL- " + queueUrl);
            SendMessageRequest messageRequest = SendMessageRequest.builder()
                    .queueUrl(queueUrl)
                    .messageBody(text).messageGroupId("BRE-DEV-IMAGE-CACHING")
                    .build();
            SendMessageResponse response = sqsClient.sendMessage(messageRequest);
            final String sequenceNumber = response.sequenceNumber();
            final String messageId = response.messageId();
            log.info("SendMessage succeed with messageId "
                    + messageId + ", sequence number " + sequenceNumber);
        } catch (Exception ex) {
            log.error("Caught an AmazonServiceException, which means " +
                    "your request made it to Amazon SQS, but was " +
                    "rejected with an error response for some reason" + ex.getMessage());
            throw ex;
        }
    }


}