import boto3
import os
import json
import logging
import requests
from aws_requests_auth.boto_utils import BotoAWSRequestsAuth
from datetime import datetime

# Define a custom logger
logger = logging.getLogger('SQS_Onboarding')
logger.setLevel(logging.DEBUG)
formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')

# Define a stream handler to output logs to stdout
stdout_handler = logging.StreamHandler()
stdout_handler.setLevel(logging.INFO)
stdout_handler.setFormatter(formatter)
logger.addHandler(stdout_handler)

# Define a stream handler to output error logs to stderr
stderr_handler = logging.StreamHandler()
stderr_handler.setLevel(logging.ERROR)
stderr_handler.setFormatter(formatter)
logger.addHandler(stderr_handler)

headers = {
    'Content-type': 'application/json',
    'Accept': 'application/json'
}

def get_aws_auth_signature():
    assetServiceUrl = os.environ.get("assetServiceUrl", "asset.api.dev.bre.mcd.com")
    aws_region = "us-east-1"
    aws_auth = BotoAWSRequestsAuth(
        aws_host=assetServiceUrl.replace("https://", ""),
        aws_region=aws_region,
        aws_service="execute-api"
    )
    return assetServiceUrl, aws_auth

# Get region name and network info from config-mgmt service
def call_configManagementService(marketName, restaurantNumber):
    config_management_url = os.environ.get("configManagementServiceUrl", "config-management.api.dev.bre.mcd.com")
    # print(config_management_url)
    aws_region = "us-east-1"
    aws_auth = BotoAWSRequestsAuth(
        aws_host=config_management_url.replace("https://", ""),
        aws_region=aws_region,
        aws_service="execute-api"
    )
    response = requests.get(url=f"{config_management_url}/v1/scope?scope_code={restaurantNumber}&market={marketName}", auth=aws_auth)
    
    if response.status_code == 200:
        responseFromRequest = json.loads(response.text)['data']
        # regionName = responseFromRequest['region_scope']
        # networkInfo = responseFromRequest['networkinfo']
        # regionNameFormatted = regionName.replace(" Region", "")
        # regionNameFormatted = regionNameFormatted.replace(" ", "_").lower()
        scopeId = responseFromRequest["id"]
        return scopeId
    else:
        logger.error("Error in config management api response.")
        os._exit(0)

# method to get restaurant id based on restaurant name and market name
def getRestaurantId(restaurantName,marketName, headers):
    logger.info("Get Restaurant Id")
    assetServiceUrl, aws_auth = get_aws_auth_signature()
    final_response = "-1"
    url = f"{assetServiceUrl}/restaurant_assets/restaurants?filter[market.name]={marketName}&filter[name]={restaurantName}"
    response = requests.get(url, headers=headers, auth=aws_auth)
    logger.info("response")
    data = json.loads(response.text)
    totalResourceCount = data["meta"]["totalResourceCount"]
    if totalResourceCount == 1:
        restaurantId = data["data"][0]["id"]
        final_response = restaurantId
    return final_response

# getMarketId
def getMarketId(marketName, headers):
    logger.info("Get MarketId")
    assetServiceUrl, aws_auth = get_aws_auth_signature()
    url = f"{assetServiceUrl}/restaurant_assets/markets?filter[name]={marketName}"
    
    response = requests.get(url, headers=headers, auth=aws_auth)
    logger.info("getMarketId response : " + response.text)
    data = json.loads(response.text)
    totalResourceCount = data["meta"]["totalResourceCount"]
    if totalResourceCount == 1:
        marketId = data["data"][0]["id"]
        logger.info(marketId)
        finalResponse = marketId
    else:
        logger.error("Error in asset api response.")
        os._exit(0)
    return finalResponse

# method to check if restaurant id is present on hierarchy_nodes
def getHierarchyNodesId(marketAssetId, restaurant_name, headers):
    logger.info("Get Hierarchy NodesId")
    assetServiceUrl, aws_auth = get_aws_auth_signature()
    final_response = "-1"
    url = f"{assetServiceUrl}/restaurant_assets/hierarchy_nodes?filter[hierarchyLevel.marketId]={marketAssetId}&filter[mcdRestaurantId]={restaurant_name}"
    response = requests.get(url, headers=headers, auth=aws_auth)
    logger.info(f"getHierarchyNodesId: "+ response.text)
    response_data = json.loads(response.text)
    if response_data["meta"]["totalResourceCount"] == 1:
        final_response = response_data["data"][0]["id"]
    return final_response

# method to add restaurant based on restaurant name, market id
def getNewRestaurantId(restaurant_name, marketName, headers):
    logger.info("update restaurant table")
    assetServiceUrl, aws_auth = get_aws_auth_signature()
    payload = {}
    body = {}
    status = "Not Onboarded"
    marketAssetId = getMarketId(marketName, headers)
    restaurantId = getRestaurantId(restaurant_name, marketName, headers)
    logger.info(f"restaurantId: "+ restaurantId)
    headers_list = {
            'Content-type': 'application/vnd.api+json',
            'Accept': 'application/json'
        }
    if restaurantId == "-1":
        get_hierarchy_nodes_id = getHierarchyNodesId(marketAssetId, restaurant_name, headers)
        if get_hierarchy_nodes_id != "-1":
            payload = {
                "data": {
                    "type": "restaurants",
                    "attributes": {"name": restaurant_name, "storeStatus":status},
                    "relationships": {
                        "market": {"data": {"id": marketAssetId, "type": "markets"}},
                        "hierarchyNode": {"data": {"id": get_hierarchy_nodes_id, "type": "hierarchy_nodes"}},
                    },
                },
            }
        else:
            payload = {
                "data": {
                    "type": "restaurants",
                    "attributes": {"name": restaurant_name, "storeStatus":status},
                    "relationships": {"market": {"data": {"id": marketAssetId, "type": "markets"}}},
                }
            }
        url = f"{assetServiceUrl}/restaurant_assets/restaurants"
        response = requests.post(url, json=payload, headers=headers_list, auth=aws_auth)
        logger.info(f"post response: " + response.text)
        response_data = json.loads(response.text)
        newRestaurantId = response_data["data"]["id"]
        final_response = newRestaurantId
        logger.info(f"newRestaurantId: " + final_response)

    else:
        final_response = restaurantId

    return final_response

# Function to check if an SQS queue already exists
def queue_exists(queue_name):
    logger.info('Checking if SQS queue exists')
    try:
        sqs = boto3.client('sqs')
        response = sqs.get_queue_url(QueueName=queue_name)
        return True
    except:
        return False

# Function to create an SQS queue
def create_queue(queue_name, kms_key_id, environment):
    logger.info('Creating new SQS queue')
    sqs = boto3.client('sqs')
    response = sqs.create_queue(
        QueueName=queue_name,
        Attributes={
            'FifoQueue': 'true',
            'ContentBasedDeduplication': 'true',
            'KmsMasterKeyId': kms_key_id
        },
        tags= {

        'Name'          : f"US-EAST-{environment}-BRE-ARCH-BRE",
        'Environment'   : f"US-EAST-{environment}-US-BRE-PROD1-ARCH",
        'GBL'           : "195500432366",
        'ApplicationID' : "APP1238134",
        'Market'        : "US",
        'Owner'         : "BRE_AOT_AWS_Account_Notifications@us.mcd.com",
        'Workload'      : f"{environment}"
    }
)
    return response['QueueUrl']

# Function to delete an SQS queue
def delete_queue(queue_url):
    logger.info('Deleting SQS queue')
    sqs = boto3.client('sqs')
    sqs.delete_queue(QueueUrl=queue_url)

# Create or Delete sqs when action is Onboard/offboard
# Create or Delete sqs when action is Onboard/offboard

def sqs_action(restaurant_ids, action, environment, kms_key_id, marketName, headers):
    logger.info('Started SQS based on Action')
    QUEUE_NAMES = set()
    kms_key = kms_key_id
    env = environment

    # Create the names of the SQS queues for each restaurant
    for restaurant_id in restaurant_ids:
        Scope_id = call_configManagementService(marketName, restaurant_id)
        RestaurantId = getNewRestaurantId(restaurant_id, marketName, headers)
        if RestaurantId == "-1":
            logger.error("RestaurantId is not found failed to create sqs")
        QUEUE_NAMES.add(f'US-EAST-{environment}-BRE-ARCH-BRE-{RestaurantId}-SQS-DEPLOYMENT.fifo')
        QUEUE_NAMES.add(f'US-EAST-{environment}-BRE-ARCH-BRE-{Scope_id}-CM-SQS-DEPLOYMENT.fifo')

    # Define the names of the SQS queues
    if action == 'OnboardCluster':
        # Check if the queues already exist
        queue_urls = []
        for queue_name in QUEUE_NAMES:
            if not queue_exists(queue_name):
                queue_url = create_queue(queue_name, kms_key, env)
                queue_urls.append(queue_url)
            else:
                sqs = boto3.client('sqs')
                queue_url = sqs.get_queue_url(QueueName=queue_name)['QueueUrl']
                queue_urls.append(queue_url)
        return f"Created queues with URLs: {queue_urls}"
    elif action == 'OffboardCluster':
        # Delete the queues
        for queue_name in QUEUE_NAMES:
            if queue_exists(queue_name):
                sqs = boto3.client('sqs')
                queue_url = sqs.get_queue_url(QueueName=queue_name)['QueueUrl']
                delete_queue(queue_url)
        return f"Deleted queues: {QUEUE_NAMES}"
    else:
        return 'Invalid action specified'


def send_sns_messages(restaurant_ids, topic_arn, product_list, marketName, action, AWS_REGION):
    logger.info('Sending SNS to Trigger Onboarding Lambda')

    # Create an SNS client
    sns = boto3.client('sns', region_name=AWS_REGION)

    # Divide the list of IDs into smaller lists of maximum size 5 each
    # id_lists = [restaurant_ids[i:i+5] for i in range(0, len(restaurant_ids), 5)]
    
    # Create and publish an SNS message for each smaller list of IDs
    # for id_list in id_lists:
    # Create the message attributes dictionary

    message_attributes = {
        'id_count': {
            'DataType': 'Number',
            'StringValue': str(len(restaurant_ids))
        },
        'RestaurantNumber': {
            'DataType': 'String',
            'StringValue': json.dumps(restaurant_ids)
        },
        'action': {
            'DataType': 'String',
            'StringValue': action
        },
        'marketName': {
            'DataType': 'String',
            'StringValue': marketName
        },
        'productList': {
            'DataType': 'String',
            'StringValue': json.dumps(product_list)
        }
    }

    # Create the message body
    message_body = "Onboard {} to GitOps Repository".format(', '.join(str(restaurant_ids)))

    # Publish the SNS message with the message body and message attributes
    response = sns.publish(
        TopicArn=topic_arn,
        Subject=message_body,
        Message=message_body,
        MessageAttributes=message_attributes
    )
        
    return f"successfully Onboard-lambda trigger by SNS with MessageId : " + response['MessageId']

# Define a Lambda function handler
def lambda_handler(event, context):
    try:
        aws_region = os.environ.get("REGION", "us-east-1")
        data = event["body"]
        if type(data) is str:
            data = json.loads(data)
        logger.info('Event body "{}"'.format(data))
        action = data["action"]
        restaurant_ids = data["restaurant_ids"]
        environment = os.environ.get("ENVIRONMENT")
        kms_key_id = os.environ.get('kms_key_id')
        marketName = os.environ.get("marketName")
        # PRODUCT_LIST = '["brecm", "infrastructure", "imagecache", "bred", "evd", "aidt", "iot", "menu"]'
        PRODUCT_LIST = data["product_list"]
        AWS_REGION = "us-east-1"
        TOPIC_ARN = os.environ.get('SNS_TOPIC_ARN')
        # TOPIC_ARN = 'arn:aws:sns:us-east-1:283388140277:gitops-onboarding-sns'
        # Define the names of the SQS queues
        if action == 'OnboardCluster':
            message = sqs_action(restaurant_ids, action, environment, kms_key_id, marketName, headers)
            trigger_lambda_SNS = send_sns_messages(restaurant_ids, TOPIC_ARN, PRODUCT_LIST, marketName, action, AWS_REGION)
            
        elif action == 'OffboardCluster':
            # Delete the queues
            message = sqs_action(restaurant_ids, action, environment, kms_key_id, marketName, headers)
            trigger_lambda_SNS = send_sns_messages(restaurant_ids, TOPIC_ARN, PRODUCT_LIST, marketName, action, AWS_REGION)
        else:
            message = f"Started {action}"
            trigger_lambda_SNS = send_sns_messages(restaurant_ids, TOPIC_ARN, PRODUCT_LIST, marketName, action, AWS_REGION)
            
        return {
            'statusCode': 200,
            'body': json.dumps({'message': trigger_lambda_SNS}),
            "headers" : {
            'Access-Control-Allow-Origin' : '*'
            }
        }
            
    except Exception as e:
        return {
            'statusCode': 500,
            'body': json.dumps({'message': f"Error occurred: {e}"}),
            "headers" : {
            'Access-Control-Allow-Origin' : '*'
            }
        }