import re
from subprocess import call
import git
import os
import yaml
from logging2 import Logger, LogLevel, StdOutHandler, StdErrHandler
import json
import boto3
import shutil
import sys
import requests
from aws_requests_auth.boto_utils import BotoAWSRequestsAuth
from datetime import datetime

stdout = StdOutHandler(level=LogLevel.info)
stderr = StdErrHandler(level=LogLevel.error)
logger = Logger("gitopsOnboarding", handlers=[stdout, stderr])
applications_list = set()

headers = {
    'Content-type': 'application/json',
    'Accept': 'application/json'
}

productList = []

# Checkout GitOps repo
def gitCheckout(breEnv, marketName, regionName, action):
    # Form gitops repo name based on environment, market and region    
    gitopsRepo = "brep-gitops-" + breEnv + "-" + marketName.lower() + "-" + regionName
    gitUrl = "github.com/bre-org/" + gitopsRepo + ".git"
    gitUrlProtocol = "https"
    gitBranch = os.environ.get("gitBranch", "main")
    # Secret Manager key Name
    smName = os.environ.get("smName", "github/bre-cloud-git-reconcile")
    awsRegion = os.environ.get("AWS_REGION", "us-east-1")

    logger.debug('gitUrl: "{}"'.format(gitUrl))
    logger.debug('gitUrlProtocol: "{}"'.format(gitUrlProtocol))
    logger.debug('gitBranch: "{}"'.format(gitBranch))
    logger.debug('smName: "{}"'.format(smName))
    logger.debug('awsRegion: "{}"'.format(awsRegion))

    # Create a Secrets Manager client
    session = boto3.session.Session()
    client = session.client(service_name="secretsmanager", region_name=awsRegion)
    get_secret_value_response = client.get_secret_value(SecretId=smName)
    secret = get_secret_value_response["SecretString"]
    jsecret = json.loads(secret)
    gitusername = jsecret["username"]
    gitpassword = jsecret["password"]

    checkoutDirectory = "/tmp/" + gitopsRepo
    # Pull latest changes from remote if repo exists locally
    if os.path.exists(checkoutDirectory):
        logger.info('"{}" Exist, Pulling the latest changes from remote'.format(checkoutDirectory))
        repo = git.Repo(checkoutDirectory)
        pullRemoteChanges = repo.remotes.origin.pull()
        logger.info('Pulled changes to local: "{}"'.format(pullRemoteChanges))
    # Clone repo if it doesn't exist locally
    else:
        try:
            logger.info('Cloning git repository "{}"'.format(gitUrl))
            git.Repo.clone_from(
                gitUrlProtocol + "://" + gitusername + ":" + gitpassword + "@" + gitUrl,
                checkoutDirectory,
                branch=gitBranch,
            )
        except Exception as e:
            errorMessage = str(e)
            # Create the repo if it is not found on remote and action is OnboardCluster
            if str(e).find("Repository not found") != -1 and action == "OnboardCluster":
                logger.info('Repository "{}" does not exist. Proceeding to create repo'.format(gitUrl))
                git_api_url = "https://api.github.com"
                # Give access to the team
                gitOpsTeamId = os.environ.get("gitOpsTeamId", "6407833")
                payload = '{"name": "' + gitopsRepo + '", "private": true, "team_id": "' + gitOpsTeamId + '" }'
                # logger.info('Payload "{}"'.format(payload))
                headers = {
                    "Authorization": "token " + gitpassword,
                    "Accept": "application/vnd.github.v3+json"
                }
                try:
                    r = requests.post(git_api_url + "/orgs/bre-org/repos", data=payload, headers=headers)
                except requests.exceptions.RequestException as err:
                    raise SystemExit(err)
                # Initialize the repo, add files and make first push to remote
                os.makedirs(checkoutDirectory)
                repo = git.Repo.init(checkoutDirectory)
                # logger.info('Repo init done "{}"'.format(repo))
                with repo.config_writer() as git_config:
                    git_config.set_value('user', 'email', 'bre.lambda@us.mcd.com')
                    git_config.set_value('user', 'name', 'BRE_LAMBDA')
                origin = repo.create_remote("origin",
                                            gitUrlProtocol + "://" + gitusername + ":" + gitpassword + "@" + gitUrl)
                readmeFile = os.path.join(checkoutDirectory, "README.md")
                with open(readmeFile, "w") as file:
                    file.write(gitopsRepo)

                if repo.is_dirty(untracked_files=True):
                    logger.info('Untracked files are detected')
                    repo.git.add('--all')
                    repo.index.commit('GitOpsOnboardingLambda: Adding files for initial commit.')
                    repo.git.checkout("-b", gitBranch)
                    repo.git.push('--set-upstream', 'origin', gitBranch)
                    logger.info('Successfully commited to Git repository')
                else:
                    logger.info('Nothing to commit, working tree clean')
            else:
                logger.error(errorMessage)
                os._exit(0)
    return checkoutDirectory


# creating PRODUCTLIST as json
def createPTlist(PRODUCTLIST):
    productJson = {
        "status": "Success",
        "data": [],
        "error": None
    }

    for item in PRODUCTLIST:
        product, tag = item.split(":")
        productJson["data"].append({
            "product": product,
            "tag": tag
        })

    return productJson


# Add products to cluster
def addProduct(dstPath, clusterKustomizationFile, productList, restaurantNumber, awsAccountID, breEnv, marketName, productTagsList,
               networkInfo):
    with open(clusterKustomizationFile, "a") as c_kustomization:
        # Get initial tags of all products in json format
        with open("templates/product-tag.json") as jsonFile:
            if productTagsList is None:
                productTags = json.load(jsonFile)['data']
            else:
                productTags = productTagsList['data']

            logger.info(f"productTags : {productTags}")
        # Create directories and files for all products in product list
        for product in productList:
            productDir = os.path.join(dstPath, "kustomizations/" + product)
            # Error handling in case of onboarding products to existing restaurant
            if not os.path.exists(productDir):
                os.makedirs(productDir)
                c_kustomization.write("  - kustomizations/%s\n" % product)
                logger.info("Adding product {} ".format(product))
                productKustomizationTemplate = "templates/product-kustomization.yaml.template"
                kustomizationFile = os.path.join(productDir, "kustomization.yaml")
                shutil.copyfile(productKustomizationTemplate, kustomizationFile)

                if (product == "brecm"):
                    fluxKustomizationTemplate = "templates/brecm-flux-kustomization.yaml.template"
                elif (product == "iot"):
                    fluxKustomizationTemplate = "templates/iot-flux-kustomization.yaml.template"
                elif (product == "aidt" or product == "menu"):
                    fluxKustomizationTemplate = "templates/flux-kustomization-canary.yaml.template"
                else:
                    fluxKustomizationTemplate = "templates/flux-kustomization.yaml.template"

                fluxKustomizationFile = os.path.join(productDir, "flux-kustomization.yaml")
                shutil.copy(fluxKustomizationTemplate, fluxKustomizationFile)
                with open(fluxKustomizationFile, "r") as f:
                    fluxKustomFile = f.read()
                fluxKustomFile = fluxKustomFile.replace("<productName>", product)
                fluxKustomFile = fluxKustomFile.replace("<restaurantNumber>", restaurantNumber)
                fluxKustomFile = fluxKustomFile.replace("<accountNumber>", awsAccountID)
                fluxKustomFile = fluxKustomFile.replace("<env>", breEnv)
                fluxKustomFile = fluxKustomFile.replace("<market>", marketName)
                fluxKustomFile = fluxKustomFile.replace("<networkInfo>", networkInfo)
                with open(fluxKustomizationFile, "w") as f:
                    f.write(fluxKustomFile)
                gitRepositoryTemplate = "templates/gitrepository.yaml.template"
                gitRepositoryFile = os.path.join(productDir, "gitrepository.yaml")
                shutil.copy(gitRepositoryTemplate, gitRepositoryFile)

                for keyval in productTags:
                    if product == keyval["product"]:
                        tag = keyval["tag"]
                with open(gitRepositoryFile, "r") as g:
                    gitFile = g.read()
                gitFile = gitFile.replace("<productName>", product)
                gitFile = gitFile.replace("<tag>", tag)
                with open(gitRepositoryFile, "w") as g:
                    g.write(gitFile)
            else:
                logger.info("Product {} already onboarded.".format(product))
        logger.info("Initiating Component insert")
        update_component_for_store(productList, productTags, restaurantNumber, marketName)


# Onboard cluster to GitOps repository
def onboardNewCluster(restaurantNumber, networkInfo, awsAccountID, breEnv, marketName, gitPath, productList, action, headers, productTagsList):
    dstPath = os.path.join(gitPath, "clusters/" + restaurantNumber)
    # Check if cluster is not onboarded already
    if not os.path.exists(dstPath):
        os.makedirs(dstPath)
        logger.info("Created Directory {} ".format(dstPath))
        clusterKustomizationTemplate = "templates/cluster-kustomization.yaml.template"
        clusterKustomizationFile = os.path.join(dstPath, "kustomization.yaml")
        shutil.copyfile(clusterKustomizationTemplate, clusterKustomizationFile)

        releasespecdstPath = os.path.join(dstPath, "releasespec")
        if not os.path.exists(releasespecdstPath):
            os.makedirs(releasespecdstPath)
            logger.info("Created Directory {} ".format(releasespecdstPath))
            releaseSpecTemplate = "templates/releasespec.yaml.template"
            releaseSpecFile = os.path.join(releasespecdstPath, "releasespec.yaml")
            shutil.copyfile(releaseSpecTemplate, releaseSpecFile)

            releaseSpecfluxKustomizationFile = os.path.join(dstPath, "flux-kustomization.yaml")
            releaseSpecfluxKustomizationTemplate = "templates/releasespec-flux-kustomization.yaml.template"
            shutil.copy(releaseSpecfluxKustomizationTemplate, releaseSpecfluxKustomizationFile)
            with open(releaseSpecfluxKustomizationFile, "r") as f:
                fluxKustomFile = f.read()
                fluxKustomFile = fluxKustomFile.replace("<restaurantNumber>", restaurantNumber)
            with open(releaseSpecfluxKustomizationFile, "w") as f:
                f.write(fluxKustomFile)

            releaseSpecKustomizationTemplate = "templates/releasespec-kustomization.yaml.template"
            releaseSpecKustomizationFile = os.path.join(releasespecdstPath, "kustomization.yaml")
            shutil.copyfile(releaseSpecKustomizationTemplate, releaseSpecKustomizationFile)
        
        addProduct(dstPath, clusterKustomizationFile, productList, restaurantNumber, awsAccountID, breEnv, marketName, productTagsList,
                   networkInfo)
        git_push(restaurantNumber, gitPath, action)
        update_restaurant_table(restaurantNumber, marketName, action, headers)
    else:
        logger.error('Restaurant already onboarded to GitOps')
        os._exit(0)

    # Offboard cluster from GitOps repository


def offboardCluster(restaurantNumber, gitPath, action, marketName, headers):
    deleteRest = "clusters/" + restaurantNumber
    deleteRestFullPath = os.path.join(gitPath, deleteRest)
    logger.info("Removing existing {} git directory!".format(deleteRestFullPath))
    shutil.rmtree(deleteRestFullPath)
    git_push(restaurantNumber, gitPath, action)
    update_restaurant_table(restaurantNumber, marketName, action, headers)

# Onboard product to existing cluster
def onboardNewProduct(restaurantNumber, networkInfo, awsAccountID, breEnv, marketName, gitPath, productList, action, headers, productTagsList):
    dstPath = os.path.join(gitPath, "clusters/" + restaurantNumber)
    if os.path.exists(dstPath):
        clusterKustomizationFile = os.path.join(dstPath, "kustomization.yaml")
        # Add products to the cluster
        addProduct(dstPath, clusterKustomizationFile, productList, restaurantNumber, awsAccountID, breEnv, marketName, productTagsList,
                   networkInfo)
        git_push(restaurantNumber, gitPath, action)
    else:
        logger.error('Restaurant not onboarded to GitOps yet. Cannot perform action OnboardProduct!')
        os._exit(0)


# Offboard products from existing cluster
def offboardProduct(restaurantNumber, gitPath, productList, action):
    clusterPath = os.path.join(gitPath, "clusters/" + restaurantNumber)
    if os.path.exists(clusterPath):
        mandatoryProducts = ["brecm", "infrastructure", "imagecache"]
        clusterKustomizationFile = os.path.join(clusterPath, "kustomization.yaml")
        for product in productList:
            if product not in mandatoryProducts:
                productDir = os.path.join(clusterPath, "kustomizations/" + product)
                if os.path.exists(productDir):
                    shutil.rmtree(productDir)
                    logger.info("Removed product {}!".format(product))
                else:
                    logger.info("Product {} was not present in the restaurant!".format(product))
            else:
                logger.info("Offboarding product {} is not allowed!".format(product))
        # Delete references of offboarded products from kustomization.yaml
        newKlusterKustomizationFile = os.path.join(clusterPath, "newKustomization.yaml")
        with open(clusterKustomizationFile) as oldfile, open(newKlusterKustomizationFile, 'w') as newfile:
            for line in oldfile:
                if not any(product in line and product not in mandatoryProducts for product in productList):
                    newfile.write(line)
        os.replace(newKlusterKustomizationFile, clusterKustomizationFile)
        git_push(restaurantNumber, gitPath, action)
    else:
        logger.error('Restaurant not onboarded to GitOps yet. Cannot perform action OffboardProduct!')
        os._exit(0)


# Push changes to GitOps repository
def git_push(restaurantNumber, gitPath, action):
    # To set git config
    repo = git.Repo(gitPath)
    with repo.config_writer() as git_config:
        git_config.set_value('user', 'email', 'bre.lambda@us.mcd.com')
        git_config.set_value('user', 'name', 'BRE_LAMBDA')
    # To check configuration values, use `config_reader()`
    with repo.config_reader() as git_config:
        print(git_config.get_value('user', 'email'))
        print(git_config.get_value('user', 'name'))
    if repo.is_dirty(untracked_files=True):
        logger.info('Untracked files are detected')
        repo.git.add('--all')
        repo.index.commit('GitOpsOnboardingLambda: Performing ' + action + ' on restaurant number ' + restaurantNumber)
        origin = repo.remote(name='origin')
        origin.push()
        logger.info('Successfully commited to Git repository')
    else:
        logger.info('Nothing to commit, working tree clean')


def update_component_for_store(productList, productTags, restaurantNo, marketName):
    product_version = {}
    for product in productList:
        for keyval in productTags:
            if product == keyval["product"]:
                product_version[product] = keyval["tag"]

    productKeyValue = get_request_for_application_detail(productList, productTags)
    assetServiceUrl, aws_auth = get_aws_auth_signature()
    productApplications = []
    logger.info("Fetching Info for product")
    response = requests.post(url=f"{assetServiceUrl}/products/version/details", auth=aws_auth, json=productKeyValue,
                             headers=headers)
    if response.status_code == 200:
        responseFromRequest = json.loads(response.text)['data']
        responseProductDetails = responseFromRequest['productDetails']
        for key, value in product_version.items():
            if (key in responseProductDetails):
                if (value in responseProductDetails[key]):
                    logger.info("Application list present in db for product: " + key)
                    productApplications.extend(responseProductDetails[key][value])
            else:
                logger.info("fetching Application list from repo for product: " + key)
                gitApplicationList = fetchApplicationfromGit(key, value)
                productApplications.extend(gitApplicationList)
        add_component(restaurantNo, marketName, productApplications)
        logger.info("Inserted Components for newly onboarded store")
    else:
        logger.error("Error in asset service api response.")
        os._exit(0)


def get_request_for_application_detail(productList, productTags):
    requested_application_list = list()
    for product in productList:
        for keyval in productTags:
            if product == keyval["product"]:
                application_dict = {
                    'productName': product,
                    'versionName': keyval["tag"]
                }
                requested_application_list.append(application_dict)
    productKeyValue = {"productVersion": requested_application_list}
    return productKeyValue


# add component entries for newly onboarded store
def add_component(store_name: str, market: str, applications_list: list):
    assetServiceUrl, aws_auth = get_aws_auth_signature()
    requested_application_list = list()
    body = prepare_request_for_bulk_component_add(applications_list, market, requested_application_list, store_name)

    response = requests.post(url=f"{assetServiceUrl}/deployment/component/data", json=body, headers=headers,
                             auth=aws_auth)
    logger.info("Inserting Application list from repo for product:  " + response.text)
    if (response.status_code != 201):
        logger.error("Error in asset service api response while adding components to newly onboarded store")
        os._exit(0)


def get_aws_auth_signature():
    assetServiceUrl = os.environ.get("assetServiceUrl", "asset.api.dev.bre.mcd.com")
    aws_region = "us-east-1"
    aws_auth = BotoAWSRequestsAuth(
        aws_host=assetServiceUrl.replace("https://", ""),
        aws_region=aws_region,
        aws_service="execute-api"
    )
    return assetServiceUrl, aws_auth


# prepare request body for asset api call for adding components to newly onboarded store
def prepare_request_for_bulk_component_add(applications_list, market, requested_application_list, store_name):
    print(applications_list)
    for application in applications_list:
        application = application.replace('\\n"', '').strip()
        application_name_version = application.split("/")[-1]
        if ":" in application_name_version:
            application_name = application_name_version.split(":")[0]
            application_version = application_name_version.split(":")[1]
            application_dict = {
                'applicationName': application_name,
                'applicationVersion': application_version
            }
            requested_application_list.append(application_dict)
    body = {
        'storeName': store_name,
        'marketName': market,
        'components': requested_application_list
    }
    return body


def fetchApplicationfromGit(gitApplicationList, gitApplicationVersion):
    requested_application_list = []
    git_path = git_checkout(gitApplicationList, gitApplicationVersion)
    logger.info('gitPath path: "{}"'.format(git_path))
    application_list = get_application_list(git_path)
    shutil.rmtree(git_path)
    for application in application_list:
        application = application.replace('\\n"', '').strip()
        application_name_version = application.split("/")[-1]
        requested_application_list.append(application_name_version)
    applications_list.clear()
    return requested_application_list


def git_checkout(product_name, tag):
    aws_region = os.environ.get("AWS_REGION", "us-east-1")
    # Secret Manager key Name
    sm_name = os.environ.get("smName", "github/bre-cloud-git-reconcile")

    # Create a Secrets Manager client
    session = boto3.session.Session()
    client = session.client(service_name="secretsmanager", region_name=aws_region)
    get_secret_value_response = client.get_secret_value(SecretId=sm_name)
    secret = get_secret_value_response["SecretString"]
    jsecret = json.loads(secret)
    gitusername = jsecret["username"]
    gitpassword = jsecret["password"]

    git_repo = f"brep-{product_name}-baseline"
    git_url = f"github.com/bre-org/{git_repo}.git"

    checkout_directory = f"/tmp/{git_repo}"
    logger.info('Checking out git repository "{}"'.format(git_url))
    git_clone_url = f"https://{gitusername}:{gitpassword}@{git_url}"
    repo = git.Repo.clone_from(
        git_clone_url,
        checkout_directory,
        branch="main"
    )
    repo.git.checkout(tag)
    repo.head.set_reference(repo.tags[tag].commit)
    return checkout_directory


def get_application_list(git_path):
    logger.info("Get application list")
    obj = os.scandir(git_path)
    for entry in obj:
        if entry.is_dir():
            get_application_list(f"{git_path}/{entry.name}")
        elif entry.is_file() and entry.name.endswith(".yaml"):
            with open(f"{git_path}/{entry.name}", "r", encoding="utf-8") as yaml_file:
                yaml_object = yaml.dump_all(yaml_file)
                application_strings = re.findall(r'mcd.jfrog.io.*', str(yaml_object))
                applications_list.update(application_strings)
            yaml_file.close()
    obj.close()
    return applications_list


# Get region name and network info from config-mgmt service
def call_configManagementService(marketName, restaurantNumber):
    config_management_url = os.environ.get("configManagementServiceUrl", "config-management.api.dev.bre.mcd.com")
    # print(config_management_url)
    aws_region = "us-east-1"
    aws_auth = BotoAWSRequestsAuth(
        aws_host=config_management_url.replace("https://", ""),
        aws_region=aws_region,
        aws_service="execute-api"
    )
    response = requests.get(url=f"{config_management_url}/v1/scope?scope_code={restaurantNumber}&market={marketName}",
                            auth=aws_auth)
    # print("call_configManagementService")
    # print(response.text)
    # print(response.status_code)
    if response.status_code == 200:
        responseFromRequest = json.loads(response.text)['data']
        regionName = responseFromRequest['region_scope']
        networkInfo = responseFromRequest['networkinfo']
        regionNameFormatted = regionName.replace(" Region", "")
        regionNameFormatted = regionNameFormatted.replace(" ", "_").lower()
        return regionNameFormatted, networkInfo
    else:
        logger.error("Error in config management api response.")
        os._exit(0)

# getMarketId
def getMarketId(marketName, headers):
    logger.info("Get MarketId")
    assetServiceUrl, aws_auth = get_aws_auth_signature()
    url = f"{assetServiceUrl}/restaurant_assets/markets?filter[name]={marketName}"
    
    response = requests.get(url, headers=headers, auth=aws_auth)
    logger.info("getMarketId response : " + response.text)
    data = json.loads(response.text)
    totalResourceCount = data["meta"]["totalResourceCount"]
    if totalResourceCount == 1:
        marketId = data["data"][0]["id"]
        logger.info(marketId)
        finalResponse = marketId
    else:
        logger.error("Error in asset api response.")
        os._exit(0)
    return finalResponse

# method to check if restaurant id is present on hierarchy_nodes
def getHierarchyNodesId(marketAssetId, restaurant_name, headers):
    logger.info("Get Hierarchy NodesId")
    assetServiceUrl, aws_auth = get_aws_auth_signature()
    final_response = "-1"
    url = f"{assetServiceUrl}/restaurant_assets/hierarchy_nodes?filter[hierarchyLevel.marketId]={marketAssetId}&filter[mcdRestaurantId]={restaurant_name}"
    logger.info(url)
    response = requests.get(url, headers=headers, auth=aws_auth)
    logger.info(f"getHierarchyNodesId: "+ response.text)
    response_data = json.loads(response.text)
    if response_data["meta"]["totalResourceCount"] == 1:
        final_response = response_data["data"][0]["id"]
        logger.info(final_response)
    return final_response

# method to get restaurant id based on restaurant name and market name
def getRestaurantId(restaurantName,marketName, headers):
    logger.info("Get Restaurant Id")
    assetServiceUrl, aws_auth = get_aws_auth_signature()
    final_response = "-1"
    url = f"{assetServiceUrl}/restaurant_assets/restaurants?filter[market.name]={marketName}&filter[name]={restaurantName}"
    response = requests.get(url, headers=headers, auth=aws_auth)
    logger.info("response")
    data = json.loads(response.text)
    totalResourceCount = data["meta"]["totalResourceCount"]
    if totalResourceCount == 1:
        restaurantId = data["data"][0]["id"]
        final_response = restaurantId
    logger.info(final_response)
    return final_response

# method to add restaurant based on restaurant name, market id
def update_restaurant_table(restaurant_name, marketName, action, headers):
    logger.info("update restaurant table")
    assetServiceUrl, aws_auth = get_aws_auth_signature()
    payload = {}
    body = {}

    if action == "OnboardCluster":
        status = "Onboarded"
    else:
        status = "Offboarded"

    marketAssetId = getMarketId(marketName, headers)

    restaurantId = getRestaurantId(restaurant_name, marketName, headers)

    logger.info(f"restaurantId: "+ restaurantId)

    headers_list = {
            'Content-type': 'application/vnd.api+json',
            'Accept': 'application/json'
        }
    
    if restaurantId == "-1":
        get_hierarchy_nodes_id = getHierarchyNodesId(marketAssetId, restaurant_name, headers)
        logger.info(get_hierarchy_nodes_id)
        if get_hierarchy_nodes_id != "-1":
            payload = {
                "data": {
                    "type": "restaurants",
                    "attributes": {"name": restaurant_name, "storeStatus":status},
                    "relationships": {
                        "market": {"data": {"id": marketAssetId, "type": "markets"}},
                        "hierarchyNode": {"data": {"id": get_hierarchy_nodes_id, "type": "hierarchy_nodes"}},
                    },
                },
            }
        else:
            payload = {
                "data": {
                    "type": "restaurants",
                    "attributes": {"name": restaurant_name, "storeStatus":status},
                    "relationships": {"market": {"data": {"id": marketAssetId, "type": "markets"}}},
                }
            }
        url = f"{assetServiceUrl}/restaurant_assets/restaurants"
        response = requests.post(url, json=payload, headers=headers_list, auth=aws_auth)
        logger.info(f"post response: " + response.text)
        response_data = json.loads(response.text)
        if response.status_code == 200:
            final_response = response_data
        else:
            logger.error("Error in asset api response.")
            os._exit(0)
    else:
        url = f"{assetServiceUrl}/restaurant_assets/restaurants/{restaurantId}"
        body = {
            "data":{
                "attributes":{
                    "storeStatus":status
                }
            }
        }
        response = requests.patch(url, json=body, headers=headers_list, auth=aws_auth)
        logger.info(f"patch response:"+ response.text)
        response_data = json.loads(response.text)
        if response.status_code == 200:
            final_response = response_data
        else:
            logger.error("Error in asset api response.")
            os._exit(0)
    return final_response


def lambda_handler(event, context):
    # Read Event values
    print(event)
    snsEvent = event["Records"][0]["Sns"]
    sns_messageAttributes = snsEvent["MessageAttributes"]
    restaurant = sns_messageAttributes["RestaurantNumber"]["Value"]
    restaurantNum = json.loads(restaurant)
    restaurantNumber = restaurantNum[0]
    logger.info("RestaurantNumber {}".format(restaurantNumber))
    marketName = sns_messageAttributes["marketName"]["Value"]
    breEnv = os.environ.get("breEnv", "dev")
    # Get region name and network info of the restaurant
    regionName, networkInfo = call_configManagementService(marketName, restaurantNumber)
    # Valid actions: OnboardCluster, OffboardCluster, OnboardProduct and OffboardProduct
    action = sns_messageAttributes["action"]["Value"]
    # Get the git path of the restaurant
    gitPath = gitCheckout(breEnv, marketName, regionName, action)
    logger.info('gitPath path: "{}"'.format(gitPath))
    awsAccountID = context.invoked_function_arn.split(":")[4]
    if (action != "OffboardCluster"):
        productString = sns_messageAttributes["productList"]["Value"]
        if ":" in json.loads(productString)[0]:
            productTagsList = createPTlist(json.loads(productString))
            productList_keys = [item["product"] for item in productTagsList["data"]]
            productList = productList_keys
        else:
            productTagsList = None
            productList = json.loads(productString)
        logger.info(f"productList: {productList}")
        logger.info(f"productTagsList: {productTagsList}")
        logger.info('List of products: "{}"'.format(productList))
    if (action == "OnboardCluster"):
        onboardNewCluster(restaurantNumber, networkInfo, awsAccountID, breEnv, marketName, gitPath, productList, action, headers, productTagsList)
    elif (action == "OffboardCluster"):
        offboardCluster(restaurantNumber, gitPath, action, marketName, headers)
    elif (action == "OnboardProduct"):
        onboardNewProduct(restaurantNumber, networkInfo, awsAccountID, breEnv, marketName, gitPath, productList, action, headers, productTagsList)
    elif (action == "OffboardProduct"):
        offboardProduct(restaurantNumber, gitPath, productList, action)
    else:
        logger.info('Not valid action passed')
