import { CoreModule, ApiService, TableSortDirection } from '@bref/core';
import { Injectable, Inject } from '@angular/core';
import { Restaurant, RestaurantList } from '../interfaces/restaurant.interface';
import { Vendor } from '../interfaces/vendor.interface'
import { ArtifactoryStorage, RestaurantsApiResponse, RestaurantVendorsApiResponse, RestaurantData, RestaurantIncludedData, VendorData, HierarchyApiDatatList, HierarchyApiResponse, RestarantHeirarchyResponse, Device, DeviceData, RestaurantDeviceApiResponse } from '../interfaces/restaurant-api-response.interface';
import { ComponentFilters } from '../interfaces/component-filters.interface';
import { SolutionsEnvService } from '../solutions-env.service';
import { DevicePropList, DevicePropResponse } from '../interfaces/device-onboarding-api.interface';
@Injectable({
    providedIn: CoreModule
})
export class RestaurantsApiService {

    private vendorUrl: string;
    private restaurantUrl: string;
    private artifactoryBaseUrl: string;
    private hierarchyUrl: string;
    private deviceUrl: string;

    constructor(private api: ApiService) {
        this.vendorUrl = SolutionsEnvService.get('vendorUrl');
        this.artifactoryBaseUrl = SolutionsEnvService.get('artifactoryBaseUrl');
        this.restaurantUrl = SolutionsEnvService.get('restaurantUrl');
        this.hierarchyUrl = SolutionsEnvService.get('hierarchyBaseUrl');
        this.deviceUrl = SolutionsEnvService.get('deviceUrl');
    }

    private getBackendBaseHeaders() {
        return {
            Authorization: "Bearer " + SolutionsEnvService.get('msalIDToken')
        };
    }

    public async getVendors(): Promise<Vendor[]> {
        const queryParams = new URLSearchParams({
            'page[limit]': '-1'
        });
        const response: RestaurantVendorsApiResponse = await this.api.get<RestaurantVendorsApiResponse>(`${this.vendorUrl}?${queryParams.toString()}`, null, null, this.getBackendBaseHeaders());
        return response.data.map(vendorData => this.reshapeVendorResponse(vendorData));
    }

    public async getModelNumberByVendor(vendorName: string, deviceType: string): Promise<string[]> {
        if (vendorName.toLowerCase() === 'hme') {
            const response = {} as ArtifactoryStorage;
            response.children = [{
                "uri": "/BS7000",
                "folder": true
            }];
            return this.reshapeArtifactoryStorageResponse(response);
        } else {
            return this.api.get<ArtifactoryStorage>(this.artifactoryBaseUrl + '/storage/IOT/' + vendorName.toLowerCase() + '/' + deviceType.toLowerCase(), null, 'There\'s something wrong, please select another device instead.', this.getBackendBaseHeaders()).then((response) => {
                return this.reshapeArtifactoryStorageResponse(response);
            },
                (reason) => {
                    return [];
                });
        }
    }

    public async getFirmwareVersions(vendorName: string, deviceType: string, modelNumber: string): Promise<string[]> {
        if (vendorName.toLowerCase() === 'hme') {
            const response = {} as ArtifactoryStorage;
            response.children = [{
                "uri": "/na",
                "folder": true
            }];
            return this.reshapeArtifactoryStorageResponse(response);
        } else {
            const response: ArtifactoryStorage = await this.api.get<ArtifactoryStorage>(this.artifactoryBaseUrl + '/storage/IOT/' + vendorName.toLowerCase() + '/' + deviceType.toLowerCase() + '/' + modelNumber.toLowerCase() + '/firmware', null, 'There\'s something wrong, please select another model number instead.', this.getBackendBaseHeaders());
    
            return this.reshapeArtifactoryStorageResponse(response);
        }
    }


    public async getHardwareVersions(vendorName: string, modelNumber: string): Promise<string[]> {
        const response: ArtifactoryStorage = await this.api.get<ArtifactoryStorage>(this.artifactoryBaseUrl + '/storage/IOT/' + vendorName.toLowerCase() + '/' + modelNumber + '/hardware', null, 'There\'s something wrong, please select another model number instead.', this.getBackendBaseHeaders());

        return this.reshapeArtifactoryStorageResponse(response);
    }

    public async getStores(page: number, pageSize: number, sortBy: string, sortDirection: TableSortDirection, filters: ComponentFilters): Promise<RestaurantList> {
        const response: RestaurantsApiResponse = await this.api.get<RestaurantsApiResponse>(this.buildStoreUrl(page, pageSize, sortBy, sortDirection, filters), null, null, this.getBackendBaseHeaders());
        return {
            data: response.data.map(restaurantData => this.reshapeRestaurantResponse(restaurantData, response.included)),
            totalItemCount: response.meta.totalResourceCount
        };
    }

    public async getAllStores(): Promise<Restaurant[]> {
        const queryParams = new URLSearchParams({
            'include': 'market',
            'page[limit]': '-1'
        });
        const response: RestaurantsApiResponse = await this.api.get<RestaurantsApiResponse>(`${this.restaurantUrl}?${queryParams.toString()}`, null, null, this.getBackendBaseHeaders());
        return response.data.map(restaurantData => this.reshapeRestaurantResponse(restaurantData, response.included));
    }

    public async getAllDevices(vendorId): Promise<Device[]> {
        const response: RestaurantDeviceApiResponse = await this.api.get<RestaurantDeviceApiResponse>(`${this.vendorUrl}/${vendorId}/device`, null, null, this.getBackendBaseHeaders());
        return response.data.map(deviceData => this.reshapeDeviceResponse(deviceData));
    }

    public async getStoreHeirarchy(storeId: string): Promise<RestarantHeirarchyResponse> {
        const response: RestarantHeirarchyResponse = await this.api.get<RestarantHeirarchyResponse>(`${this.restaurantUrl}/${storeId}/hierarchyNode`, null, null, this.getBackendBaseHeaders());
        return response;
    }

    public async getHierarchyLevelNodeValues(filters): Promise<HierarchyApiDatatList> {
        const urlSearchParams = new URLSearchParams();
        urlSearchParams.append('filter[id]', filters?.hierarchyLevelID.toString());
        const response = await this.api.get<HierarchyApiResponse>(`${this.hierarchyUrl}?${urlSearchParams.toString()}`, null, null, this.getBackendBaseHeaders());
        let included = null;
        if(response.included !== undefined){
            included = response.included.map(heirarchyData => heirarchyData)
        }
        return {
            totalItemCount: response.meta.totalResourceCount,
            data: response.data.map(heirarchyData => heirarchyData),
            included : included
        }
    }

    public async getDeviceProps(deviceId): Promise<DevicePropList[]> {
        const response = await this.api.get<DevicePropResponse>(`${this.deviceUrl}/${deviceId}/deviceProp`, null, null, this.getBackendBaseHeaders());
        return response.data;
    }

    public async getHierarchyDeploymentNodeValues(id) {
        let selectedFilters = {
            hierarchyLevelID: id,
        };
        const coOp = await this.getHierarchyLevelNodeValues(selectedFilters).then((response) => response.data);
        selectedFilters = {
            hierarchyLevelID: Array.from(new Set (coOp.map(node => node?.attributes?.parentNodeId))),
        };
        const region = await this.getHierarchyLevelNodeValues(selectedFilters).then((res) => res.data);
        selectedFilters = {
            hierarchyLevelID: Array.from(new Set(region.map(node => node?.attributes?.parentNodeId))),
        };
        const market = await this.getHierarchyLevelNodeValues(selectedFilters).then((res) => res.data);
        return {
            deploymentCoOp : coOp,
            deploymentRegion: region,
            deploymentMarket: market,
            deploymentStore:  {}
        };
    }

    public async searchRestaurants(searchTerm, likeTerm): Promise<Restaurant[]> {
        const filterLikeTerm = `filter[${likeTerm}][like]`
        const searchParams = new URLSearchParams({
            'sort': 'name,id',
            [filterLikeTerm]: `%${searchTerm.replace(/%/g, '')}%`
        });
        const response: RestaurantsApiResponse = await this.api.get<RestaurantsApiResponse>(`${this.restaurantUrl}?${searchParams.toString()}`, null, null, this.getBackendBaseHeaders());
        return response.data.map(restaurantData => this.reshapeRestaurantResponse(restaurantData, response.included));
    }

    public async query(relativeUrl): Promise<RestaurantList> {
        const response = await this.api.get<RestaurantsApiResponse>(`${this.restaurantUrl}${relativeUrl}`, null, null, this.getBackendBaseHeaders());
        return {
            data: response.data.map(restaurantData => this.reshapeRestaurantResponse(restaurantData, response.included)),
            totalItemCount: response.meta.totalResourceCount
        };
    }

    private buildStoreUrl(page: number, pageSize: number, sortBy: string, sortDirection: TableSortDirection, filters: ComponentFilters): string {
        const searchParams = new URLSearchParams({
            'sort': 'id',
            'page[offset]': ((page - 1) * pageSize).toString(),
            'page[limit]': pageSize.toString(),
        });
        if (filters.storeName?.length > 0) {
            searchParams.append('filter[name][like]', filters.storeName.map((x) => `%${x}%`).join(','))
        }
        if (filters.storeId?.length > 0) {
            searchParams.append('filter[id]', filters.storeId.join(','))
        }
        return `${this.restaurantUrl}?${searchParams.toString()}`;
    }

    private reshapeArtifactoryStorageResponse(response: ArtifactoryStorage): string[] {
        const data = [];
        response.children.forEach(child => {
            const formattedUri = child.uri.replace(/\//g, ''); //remove the slash from the URI and add to array
            data.push(formattedUri);
        });
        return data;
    }

    private reshapeRestaurantResponse(restaurantData: RestaurantData, included: RestaurantIncludedData[]): Restaurant {
        return {
            id: restaurantData.id,
            name: restaurantData.attributes.name,
            marketName: included?.find(x => x.type === 'markets' && x.id === restaurantData.relationships.market?.data?.id)?.attributes?.name
        };
    }

    private reshapeDeviceResponse(deviceData: DeviceData): Device {
        return {
            id: deviceData.id,
            name: deviceData.attributes.name,
            deviceFlag: deviceData.attributes.deviceFlag,
            abbreviation: deviceData.attributes.abbreviation
        }
    }

    private reshapeVendorResponse(vendorData: VendorData): Vendor {
        return {
            id: vendorData.id,
            name: vendorData.attributes.name,
            contactInfo: vendorData.attributes.contactInfo,
            contactInfoType: vendorData.attributes.contactInfoType,
            deviceFlag: vendorData.attributes.deviceFlag,
            notes: vendorData.attributes.notes
        }
    }
}