import { Injectable, Inject } from '@angular/core';
import { OnboardingInfo, Component } from '../interfaces/component.interface';
import { OnboardingDeviceApiRequestBody, OnboardingNonIoTDeviceApiRequestBody, OnboardingDeviceApiResponse } from '../interfaces/onboarding-api.interface';
import { CoreModule, ApiService } from '@bref/core';
import { SolutionsEnvService } from '../solutions-env.service';

@Injectable({
    providedIn: CoreModule
})
export class OnboardingApiService {
    private iotBaseUrl: string;
    private nonIotBaseUrl: string;

    constructor(private api: ApiService) {
        this.iotBaseUrl = SolutionsEnvService.get('deviceOnboardingBaseUrl');
        this.nonIotBaseUrl = SolutionsEnvService.get('nonIotDeviceOnboardingBaseUrl');
    }

    private getBackendBaseHeaders() {
        return {
            Authorization: "Bearer " + SolutionsEnvService.get('msalIDToken')
        };
    }

    async getAvailableDevices(deviceType: string): Promise<Component[]> {
        return [
            {
                id: null,
                status: null,
                manufacturer: 'Hella Aglaia',
                model: 'APS-90E',
                name: null,
                type: null,
                store: null,
                storeId: null,
                function: null,
                ipAddress: null,
                serialNumber: undefined,
                warranty: null,
                technicalName: null,
                market: null,
                lineOfBusiness: null,
                cellLocation: null,
                supplier: null,
                description: null,
                mainUsage: null,
                certificate: null,
                version: null,
                laneNumber: null,
                macAddress: null,
                onboardingStatus: null,
                hardwareVersion: null,
                installDate: null,
                hierarchyNode: null,
                isIOTDevice: null,
            }
        ]
    }

    public async addNewDevice(deviceInfo): Promise<number> {
        const request: OnboardingDeviceApiRequestBody = {
            restaurantId: parseInt(deviceInfo.store),
            workflow: 'iot',
            deviceDetailsList: deviceInfo.deviceDetailsList.map((details) => {
                return {
                    deviceName: details.deviceName,
                    vendorId: parseInt(details.vendor),
                    modelNumber: details.modelNumber,
                    serialNumber: details.serialNumber,
                    warranty: details.warranty ? details.warranty.toISOString() : '',
                    installDate: details.installDate.toISOString(),
                    reportedVersion: details.firmwareVersion,
                    hardwareVersion: details.hardwareVersion,
                    devicePropItems: {}
                };
            })
        };

        const response: OnboardingDeviceApiResponse = await this.api.post<OnboardingDeviceApiResponse>(this.iotBaseUrl, request, null, null, this.getBackendBaseHeaders());
        return response.startDate;
    }

    public async addNewNonIoTDevice(deviceInfo): Promise<number> {
        const request: OnboardingNonIoTDeviceApiRequestBody = {
            restaurantId: parseInt(deviceInfo.store),
            workflow: 'iot',
            deviceDetailsList: deviceInfo.deviceDetailsList.map((details) => {
                return {
                    deviceName: details.deviceName,
                    vendorId: parseInt(details.vendor),
                    modelNumber: details.modelNumber,
                    serialNumber: details.serialNumber,
                    warranty: details.warranty ? details.warranty.toISOString() : '',
                    installDate: details.installDate.toISOString(),
                    action: details.action,
                    reportedVersion: details.firmwareVersion,
                    hardwareVersion: details.hardwareVersion,
                    devicePropItems: details.deviceProps
                };
            })
        };
         const response: OnboardingDeviceApiResponse = await this.api.post<OnboardingDeviceApiResponse>(this.nonIotBaseUrl, request, null, null, this.getBackendBaseHeaders());
         return response.startDate;
    }
}