import { Injectable, Inject } from '@angular/core';
import { ApiService, CoreModule, TableSortDirection, and, or, attribute, operator } from '@bref/core';
import { AssetType, ComponentFilters, FilterPropertyKey, AttributeKey, Operator } from '../interfaces/component-filters.interface';
import { ComponentList, activityLogDetails } from '../interfaces/component.interface';
import { AssetTypeApiResponse, ComponentsApiResponse, ComponentApiResponse, ComponentData, IncludedData ,VendorApiData } from '../interfaces/components-api-response.interface';
import { Component } from '../interfaces/component.interface';
import { SolutionsEnvService } from '../solutions-env.service';
@Injectable({
  providedIn: CoreModule
})
export class ComponentsApiService {

  private baseUrl: string;
  private componentTypeUrl: string;
  private restaurantHierarchyNodeUrl:string;
  private deviceDetails:string;
  private vendorDetails: string;

  private baseIncludeSearchParams: string = 'componentAuditHistory,restaurant.hierarchyNode,componentProps,vendor,type,vendor.device,vendor.device.deviceProp';
  private readonly DEVICE_TYPE = ['GREENGRASS_DEVICE','DEVICE'];
  private vendorSearchParams: string = 'vendor_contacts,vendor_contacts.contacts';
  private vendorDetailParams: string = 'vendor_contacts.level,vendor_contacts.contacts,device.deviceProp';

  constructor(private api: ApiService) { 
    this.baseUrl = SolutionsEnvService.get('ComponentsUrl');
    this.componentTypeUrl =  SolutionsEnvService.get('componentTypeUrl');
    this.restaurantHierarchyNodeUrl = SolutionsEnvService.get('hierarchyBaseUrl');
    this.deviceDetails = SolutionsEnvService.get('restaurantBaseUrl')+'/restaurant_assets/deviceDetails';
    this.vendorDetails = SolutionsEnvService.get('vendorUrl');
  }

  private getBackendBaseHeaders() {
    return { 
      Authorization: "Bearer " +  SolutionsEnvService.get('msalIDToken')
    };
  }

  async getAssetTypes(): Promise<AssetType[]> {
    const configResponse = await this.api.get<AssetTypeApiResponse>(`${this.componentTypeUrl}`, null, null, this.getBackendBaseHeaders());
    return configResponse.data.map((x) => {
      const type: AssetType = {
        id: x.id,
        name: x.attributes?.name
      };
      return type;
    });
  }

  public async getDevices(page: number, pageSize: number, sortBy: string, sortDirection: TableSortDirection, filters: ComponentFilters): Promise<ComponentList> {
    const response: ComponentsApiResponse = await this.api.get<ComponentsApiResponse>(this.buildDeviceUrl(page, pageSize, sortBy, sortDirection, filters), null, null, this.getBackendBaseHeaders());
    return {
      totalComponentCount: response.meta.totalResourceCount,
      data: response.data.map(deviceData => this.reshapeComponentResponse(deviceData, response.included))
    };   
  }

  public async getDevicesList(page: number, pageSize: number, sortBy: string, sortDirection: TableSortDirection, filters: ComponentFilters) {
    const searchParams = new URLSearchParams({
      'page[limit]': '-1',
    });
    const response: ComponentsApiResponse = await this.api.get<ComponentsApiResponse>(`${this.deviceDetails}?${searchParams.toString()}`, null, null, this.getBackendBaseHeaders());
   
    const devicesVal = response.data.reduce((results, item) => {
      results.push({
        id : item.id,
        serialNumber : item.attributes.serialNumber,
        vendorName: item.attributes.vendorName,
        modelNumber: item.attributes.modelNumber,
        hardwareVersion: item.attributes.hardwareVersion ? item.attributes.hardwareVersion : '-',
        firmwareVersion: item.attributes.firmwareVersion ? item.attributes.firmwareVersion : '-',
        storeName: item.attributes.storeName,
        onboardingStatus: this.getDeviceDetailsStatus(item.attributes.onboardingStatus),
        deviceName: item.attributes.deviceName,
        displayName: item.attributes.displayName ? item.attributes.displayName : '-',
        identificationName: item.attributes.identificationName ? item.attributes.identificationName : '-',
      });
      return results;
    },[]);
    return {
      totalComponentCount: response.meta.totalResourceCount,
      data: devicesVal
    };   
  }

  public async getVendorsList(page: number, pageSize: number, sortBy: string, sortDirection: TableSortDirection) {
    const searchParams = new URLSearchParams({
      'include': `${this.vendorSearchParams}`,
      'page[limit]': '-1',
    });
    const response: any = await this.api.get<any>(`${this.vendorDetails}?${searchParams.toString()}`, null, null, this.getBackendBaseHeaders());
    return {
      totalComponentCount: response.meta.totalResourceCount,
      data: response.data.map(vendorData => this.reshapeVendorResponse(vendorData,response.included))     
    };   
  }

  
  getDeviceDetailsStatus(onboardingStatus) {
    let status;

    switch (onboardingStatus) {
      case 'Failed':
        status = "Onboarding Failed";
        break;
      case "Onboarding message sent":
        status = "Onboarding In-progress";
        break;
      case "Certificate message sent":
        status = "Onboarding In-progress";
        break;
      case "Complete":
        status = "Onboarding Completed";
        break;
      case "Success":
        status = "Onboarding Completed";
        break;
      case "Device Successfully Onboarded":
        status = "Onboarding Completed";
        break;
      }
    return status;
  }

  public async searchDevices(searchTerm, likeTerm): Promise<Component[]> {
    const filterLikeTerm = `filter[${likeTerm}][like]`
    const filterLikeTerm1 = `filter[${likeTerm}][in]`
    const searchParams = new URLSearchParams({
      'include': 'type',
      'sort': 'name,id',
      'filter[type.name]': this.DEVICE_TYPE.toString(),
      [filterLikeTerm]: `%${searchTerm.replace(/%/g, '')}%`
    });
    const response: ComponentsApiResponse = await this.api.get<ComponentsApiResponse>(`${this.baseUrl}?${searchParams.toString()}`, null, null, this.getBackendBaseHeaders());
    return response.data.map(deviceData => this.reshapeComponentResponse(deviceData, response.included));
  }

  public async getAssets(page: number, pageSize: number, sortBy: string, sortDirection: TableSortDirection, filters: ComponentFilters): Promise<ComponentList> {
    const response: ComponentsApiResponse = await this.api.get<ComponentsApiResponse>(this.buildComponentUrl(page, pageSize, sortBy, sortDirection, filters) , null, null, this.getBackendBaseHeaders());
    return {
      totalComponentCount: response.meta.totalResourceCount,
      data: response.data.map(deviceData => this.reshapeComponentResponse(deviceData, response.included))
    };
  }

  public async searchAssets(searchTerm, filterTerm): Promise<Component[]> {
    const filterLikeTerm = `filter[${filterTerm}][like]`
    const searchParams = new URLSearchParams({
      'include': 'type',
      'sort': 'id,name',
      'filter[type.name][neq]': this.DEVICE_TYPE.toString(),
      [filterLikeTerm]: `%${searchTerm.replace(/%/g, '')}%`
    });
    const response: ComponentsApiResponse = await this.api.get<ComponentsApiResponse>(`${this.baseUrl}?${searchParams.toString()}`, null, null, this.getBackendBaseHeaders());
    return response.data.map(deviceData => this.reshapeComponentResponse(deviceData, response.included));
  }

  public processDeviceProps(deviceprops,allProps){

  }

  public async getComponent(componentId: number): Promise<Component> {
    const searchParams = new URLSearchParams({
      'include': `${this.baseIncludeSearchParams}`
    });
    const response: ComponentApiResponse = await this.api.get<ComponentApiResponse>(`${this.baseUrl}/${componentId}?${searchParams.toString()}`, null, null, this.getBackendBaseHeaders());
    return this.reshapeComponentResponse(response.data, response.included);
  }

  
  
  public async getVendorComponent(componentId: number): Promise<object> {
    const searchParams = new URLSearchParams({
      'include': `${this.vendorDetailParams}`      
    });    

    const response: any = await this.api.get<any>(`${this.vendorDetails}/${componentId}?${searchParams.toString()}`, null, null, this.getBackendBaseHeaders());
    let vendorDevices = response.data.relationships.device.data;   
    
    let devicesData: any = [];
    let vendorComp:any={};
    let contactsArray:any = [];
    let mcdContactsArray: any =[];
    let reshapedProps:any=[];

    // All device props
    let allDeviceprops: any = response.included?.filter(item => {
      return item.type === "device_props"
    })  
   

    let listOfVendorContacts: any = response.included?.filter(item => {
      return item.type === 'vendor_contacts'
    })

    let listOfContacts: any = response.included?.filter(item => {
      return item.type === 'contacts'
    })

    // All devices data
    devicesData = response.included?.filter(item => {
      return item.type === 'device'
    })

    let reshapedDevicesData:any = [];
    let vendorDevicesInfo: any = [];

    vendorDevices.map(dev => {
      let deviceInfo = devicesData.find(item => {
        return item.id == dev.id;
      })
      vendorDevicesInfo.push(deviceInfo);
    })

    let reshapedVendorInfo: any =[];

    
    vendorDevicesInfo.map(info => {
      let devicePropsArr: any = [];
      let devicePropsInfo = info.relationships.deviceProp.data;

      devicePropsInfo.map(inf => {
        let info = allDeviceprops.find(prop => {
          return prop.id === inf.id
        })
        devicePropsArr.push({
          name: this.convertCamelCaseToWords(info.attributes.propertyName),
          type: "New"
        })
      })

      reshapedVendorInfo.push({
        deviceType: info.attributes.name,
        deviceTypeCategory: info.attributes.deviceFlag?"AWS":'non AWS',
        deviceTypeAttributes: devicePropsArr,
        deviceAbbreviation: info.attributes?.abbreviation
      });
    })

    let deviceTypeAttributes;   

     vendorComp.devices = reshapedVendorInfo;
    listOfContacts?.map(contact => {
      let filteredLevel = listOfVendorContacts.find(i => {return i.relationships.contacts.data.id == contact.id});
      
      let reshapedContact :any = {
        name: contact.attributes.contact_name,
        level: filteredLevel.attributes.level,
        email: contact.attributes.email_id,
        phone: contact.attributes.phone_number,
        notes: filteredLevel.attributes.notes
      }; 
      if(!contact.is_MCD){
        contactsArray.push(reshapedContact);
      }else{
        mcdContactsArray.push(reshapedContact);
      }    
          
    })

    vendorComp.contacts = contactsArray;
    vendorComp.mcdcontacts = mcdContactsArray;
    vendorComp.vendorData = response;
    return vendorComp;
  }

  public async getVendorContact(componentId: number,vendorcontacts): Promise<any> {
    const searchParams = new URLSearchParams({
      'include': `${this.vendorSearchParams}`,
      'page[limit]': '-1',
    });
    const response: any = await this.api.get<any>(`${this.vendorDetails}?${searchParams.toString()}`, null, null, this.getBackendBaseHeaders());
    let includedData = response.included;
    let vendorContact:any=[];
    let contactsatVendor = includedData?.find(i => i.type === 'vendor_contacts');
    
    vendorcontacts.map(contact => {
      let filteredContact = includedData?.find(i => i.type === 'contacts' && i.id === contact.id);
      filteredContact&&vendorContact.push(filteredContact);      
    })
    
    return {
      vendorcontact: vendorContact,
      includedata: includedData
    };
    
  }


  

  public async getRestauratDetails(hierarchyNodeId){
    const urlSearchParams = new URLSearchParams({
      'filter[id]': hierarchyNodeId,
      'page[limit]': '-1',
    });
    const response: ComponentApiResponse = await this.api.get<ComponentApiResponse>(`${this.restaurantHierarchyNodeUrl}/?${urlSearchParams.toString()}`, null, null, this.getBackendBaseHeaders());
    return this.reshapeRestaurantsResponse(response.data);
  }

  private buildDeviceUrl(page: number, pageSize: number, sortBy: string, sortDirection: TableSortDirection, filters: ComponentFilters): string {
    const sortMap = {
      'id': 'id,name',
      'name': 'name,id',
    }
    const searchParams = new URLSearchParams({
      'include': this.baseIncludeSearchParams,
      'sort': !!sortMap[sortBy] ? `${sortDirection === TableSortDirection.asc ? '' : '-'}${sortMap[sortBy]}` : 'id',
      'page[offset]': ((page - 1) * pageSize).toString(),
      'page[limit]':'-1',
    });

    // reshape device filters to assign keys "id", "name" etc
    const selectedFilters: ComponentFilters = Object.entries(filters).reduce((acc, [key, value]) => {
      if (key === 'componentId' && value?.length > 0) {
        acc[key] = { [FilterPropertyKey.ID]: value };
      } else if (key === 'componentName' && value?.length > 0) {
        acc[key] = { [FilterPropertyKey.NAME]: value };
      } else if (key === 'storeId' && value?.length > 0) {
        acc[key] = { [FilterPropertyKey.ID]: value };
      } else if (key === 'storeName' && value?.length > 0) {
        acc[key] = { [FilterPropertyKey.NAME]: value }
      }
      return acc;
    }, {})


    const filterJson = and(
      { type: { name: this.DEVICE_TYPE } },
      or(selectedFilters?.componentId, operator(Operator.LIKE, false, selectedFilters?.componentName)),
      attribute(AttributeKey.RESTAURANT, or(selectedFilters?.storeId, operator(Operator.LIKE, false, selectedFilters?.storeName)))
    );

    searchParams.append('filter', JSON.stringify(filterJson))
    return `${this.baseUrl}?${searchParams.toString()}`;
  }

  private buildComponentUrl(page: number, pageSize: number, sortBy: string, sortDirection: TableSortDirection, filters: ComponentFilters): string {
    const searchParams = new URLSearchParams({
      'include': this.baseIncludeSearchParams,
      'sort': !!sortBy ? `${sortDirection === TableSortDirection.asc ? '' : '-'}${sortBy}` : 'id',
      'page[offset]': ((page - 1) * pageSize).toString(),
      'page[limit]': pageSize.toString(),
      'filter[type.name][neq]': this.DEVICE_TYPE.toString(),
    });

    if (filters.componentId?.length > 0 && filters.componentName?.length > 0) {
      const filterJSON = {
        'OR': {
          'id': filters.componentId,
          'LIKE': {
            'name': filters.componentName.map((x) => `${x}`)
          }
        }
      }
      searchParams.append('filter', JSON.stringify(filterJSON));
    }
    
    if (filters.componentType?.length > 0) {
      searchParams.append('filter[type.id]', filters.componentType.join(','));
    }
    if (filters.componentId?.length > 0 && !(filters.componentName?.length > 0)) {
      searchParams.append('filter[id]', filters.componentId.join(','));
    }
    if (!(filters.componentId?.length > 0) && filters.componentName?.length > 0) {
      searchParams.append('filter[name][like]', filters.componentName.join(','));
    }
    return `${this.baseUrl}?${searchParams.toString()}`;
  }

  private reshapeRestaurantsResponse(response) {
    return {
      mcdInternalNodeName: response[0].attributes.mcdInternalNodeName,
      parentNodeId: response[0].attributes.parentNodeId
    };
  }
 
 /**
 * 
 * @param included API response included data
 * @param contactId vendor contact details
 * @returns primary contact from list of vendor contacts
 */ 
private getVendorContacts(included, contactId){
  if(contactId.data.length > 0){ 
    const contactIds = contactId.data.map(item => item.id);
    let notesContactPrimary = included.find(i => contactIds.includes(i.id) && (i.attributes.level === 'Primary' || i.attributes.level === 'primary'));
    return (notesContactPrimary ? included.find(i => i.id === notesContactPrimary.relationships.contacts.data.id) : "");
  } else{
    return "";
  }
}  



/**
 * 
 * @param vendorData API response component data
 * @param included API response included data
 * @returns reshaped Vendor data
 */
 private reshapeVendorResponse(vendorData: VendorApiData ,included:IncludedData[]){

 let contactDetails = this.getVendorContacts(included,vendorData.relationships.vendor_contacts);
 
 let contactName = contactDetails?contactDetails.attributes.contact_name:"";
 let contactInformation = contactDetails?contactDetails.attributes.email_id + " | " + contactDetails.attributes.phone_number:"";

  return {
    id: vendorData.id,
    name: vendorData.attributes.name,
    created: vendorData.attributes.created,
    contactName: contactName,
    contactInfo: contactInformation,
  }
 }

/**
 * 
 * @param componentData API response component data
 * @param included API response included data
 * @returns Component data of type Component
 */
  private reshapeComponentResponse(componentData: ComponentData, included: IncludedData[]): Component {
    return {
      id: componentData.id,
      status: '-',
      name: componentData.attributes.name,
      displayName: componentData.attributes.displayName,
      model: componentData.attributes.modelNumber.toString(),
      type: componentData.relationships?.type?.data?.id,
      store: this.getStoreName(included, componentData.relationships.restaurant?.data?.id),
      storeId: this.getRestaurantIdentifier(included, componentData.relationships.restaurant?.data?.id),
      function: '-',
      ipAddress: this.getIPAddress(included, componentData.id),
      serialNumber: componentData.attributes.serialNumber,
      cellLocation: null,
      certificate: null,
      description: null,
      lineOfBusiness: null,
      mainUsage: null,
      manufacturer: null,
      market: this.getMarketName(included, componentData.relationships.restaurant?.data?.id),
      technicalName: null,
      warranty: componentData.attributes.warranty,
      supplier: this.getVendorName(included, componentData.relationships.vendor?.data?.id),
      version: componentData.attributes.reportedVersion,
      hardwareVersion: componentData.attributes.hardwareVersion ? componentData.attributes.hardwareVersion : '-',
      laneNumber: this.getLaneInfo(included,componentData.id),
      macAddress: this.getMACAddress(included,componentData.id),
      onboardingStatus: this.getOnboardingStatus(included,componentData.id),
      installDate: componentData.attributes.installDate,
      hierarchyNode: this.getHierarchyNodeId(included),
      isIOTDevice: this.getDeviceFlag(included),
      activityLog: this.getActivityLogDetails(included),
      dynamicFields: this.getDynamicFields(included),
      installerName: this.getInstallerDetails(included, 'InstallerName'),
      installerEmail: this.getInstallerDetails(included, 'InstallerEmail')
    };
  }
/**
 * This function gets the dynamic fields added from device props and then loops through included with
 * the same property name to fetch the property value since component props includes device props as well as other props
 * @param included of type IncludedData 
 * @returns dyanmic fields of device
 */
  getDynamicFields(included: IncludedData[]) {
  const componentProps = included?.filter(i => i.type === 'component_props');
   return included?.reduce((results, item) => {
      if (item.type === 'device_props' && componentProps?.find(i => i.attributes.propertyName === item.attributes?.propertyName)?.attributes?.propertyValue) {
        results.push({
          'label': this.convertCamelCaseToWords(item.attributes?.propertyName),
          'value': componentProps?.find(i => i.attributes.propertyName === item.attributes?.propertyName)?.attributes?.propertyValue
        });
      }
      return results
    }, [])
  }

   /**
   * Converts a give string in camel case format to seperate words
   * @param string string
   */
    convertCamelCaseToWords(string) {
      const convertString  = string
      // insert a space before all caps
      .replace(/([A-Z])/g, ' $1')
      // uppercase the first character
      .replace(/^./, function(str){ return str.toUpperCase(); });
      if(convertString.split(" ")[0] === 'Ip' || convertString.split(" ")[0] === 'Mac') {
        return convertString.split(" ")[0].toUpperCase()+" "+convertString.split(" ")[1];
      } else {
        return convertString;
      }
      
    }

  getActivityLogDetails(included) {
    let activityLogDetails = included?.filter(i => i.type === 'component_audit_history');
   return activityLogDetails?.map(element => {
    return {
      'heading' : element.attributes.action,
      'sourceType': element.attributes.srcOfAction,
      'description': element.attributes.description,
      'createdDate': element.attributes.created
    }
   });
  }

  private getStoreName(included: IncludedData[], restaurantId: string) {
   return included?.find(i => i.type === 'restaurants' && i.id === restaurantId)
    ?.attributes?.name;
  }

  private getRestaurantIdentifier(included: IncludedData[], restaurantId: string): string {
    const marketName = this.getMarketName(included, restaurantId) || '';
    return marketName ? `${marketName.toUpperCase()}-${restaurantId}` : `${restaurantId}`;
  }

  private getMarketName(included: IncludedData[], restaurantId: string): string {
    const marketId = included
      ?.find(i => i.type === 'restaurants' && i.id === restaurantId)
      ?.relationships?.market?.data?.id;
    return included
      ?.find(i => i.type === 'markets' && i.id === marketId)
      ?.attributes?.name;
  }

  private getIPAddress(included: IncludedData[], deviceId: string): string {
    return included
      ?.find(i => i.type === 'component_props' && i.attributes.propertyName === 'ipAddress')
      ?.attributes?.propertyValue
      || '-';
  }

  private getHierarchyNodeId(included: IncludedData[]) {
    return included
      ?.find(i => i.type === 'hierarchy_nodes')
      ?.attributes?.parentNodeId;
  }

  /**
   * 
   * @param included API response Included data
   * verifies if the device is IOT or non IOT
   * @returns booleans
   */
  private getDeviceFlag(included: IncludedData[]) {
    return included
      ?.find(i => i.type === 'device')
      ?.attributes?.deviceFlag;
  }
  /**
   * 
   * @param included API response Included data
   * @param vendorId 
   * @returns vendor name
   */
  private getVendorName(included: IncludedData[], vendorId: string): string {
    return included
      ?.find(i => i.type === 'vendors' && i.id === vendorId)
      ?.attributes?.name;
  }
  /**
   * 
   * @param included API response Included data
   * @param deviceId 
   * @returns MACAddress
   */
  private getMACAddress(included: IncludedData[], deviceId: string): string {
    return included
      ?.find(i => i.type === 'component_props' && i.attributes.propertyName === 'macAddress')
      ?.attributes?.propertyValue
      || '-';
  }
  private getLaneInfo(included: IncludedData[], deviceId: string): string {
    return included
      ?.find(i => i.type === 'component_props' && i.attributes.propertyName === 'laneNumber')
      ?.attributes?.propertyValue
      || '-';
  }
  private getOnboardingStatus(included: IncludedData[], deviceId: string): string {
    return this.getDeviceDetailsStatus(included
      ?.find(i => i.type === 'component_props' && i.attributes.propertyName === 'OnboardingStatus')
      ?.attributes?.propertyValue)
      || '-';
  }
  private getInstallerDetails(included: IncludedData[], propertyName: string): string {
    return included
      ?.find(i => i.type === 'component_props' && i.attributes.propertyName === propertyName)
      ?.attributes?.propertyValue
      || '-';
  }
}